﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
* @examplesource{VfxBinarySegmentation/main.cpp,PageSampleVfxBinarySegmentation}
*
* @brief
*  バイナリ分割を用いたエフェクト表示のサンプルプログラム
*/

/**
* @page PageSampleVfxBinarySegmentation バイナリ分割を用いたエフェクト表示
* @tableofcontents
*
* @image html Applications\VfxBinarySegmentation\VfxBinarySegmentation.png
*
* @brief
*  サンプルプログラム VfxBinarySegmentation の解説です。
*
* @section PageSampleVfxBinarySegmentation_SectionBrief 概要
*  nn::vfx を使用してエフェクトの分割バイナリを再生する最もシンプルなサンプルです。
*
*  エフェクトランタイムの機能の使い方については、
*  @ref nn::vfx "Vfx の関数リファレンス" も併せて参照して下さい。
*
* @section PageSampleVfxBinarySegmentation_SectionFileStructure ファイル構成
*  本サンプルプログラムは @link ../../../Samples/Sources/Applications/VfxBinarySegmentation
*  Samples/Sources/Applications/VfxBinarySegmentation @endlink 以下にあります。
*
* @section PageSampleVfxBinarySegmentation_SectionNecessaryEnvironment 必要な環境
*  画面表示が利用可能である必要があります。
*
* @section PageSampleVfxBinarySegmentation_SectionHowToOperate 操作方法
*  このサンプルを実行するとエフェクトが自動的に再生され、一定回数描画した後終了します。
*
* @section PageSampleVfxBinarySegmentationt_SectionPrecaution 注意事項
*  特にありません。
*
* @section PageSampleVfxBinarySegmentation_SectionHowToExecute 実行手順
*  サンプルプログラムをビルドし、実行してください。
*
* @section PageSampleVfxBinarySegmentation_SectionDetail 解説
*
* @subsection PageSampleVfxBinarySegmentation_SectionSampleProgram サンプルプログラム
*  以下に本サンプルプログラムのソースコードを引用します。
*
*  VfxBinarySegmentation/main.cpp
*  @includelineno VfxBinarySegmentation/main.cpp
*
* @subsection PageSampleVfxBinarySegmentation_SectionSampleDetail サンプルプログラムの解説
*  サンプルプログラムの処理の流れは以下の通りです。
*  - Gfxの初期化
*  - カメラ・プロジェクション行列の初期化
*  - VFXランタイムの初期化
*  - VFXの分割バイナリをロード
*  - エフェクト生成
*  - ループ開始
*  - VFXランタイムの計算処理
*  - VFXランタイムの描画処理
*  - ループ開始に戻る
*  - VFXランタイムの終了処理
*  - Gfxの終了処理
*
*/

#include <nn/TargetConfigs/build_Base.h>

#include <nn/nn_Result.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_Common.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs.h>
#include <nn/gfx/util/gfx_DescriptorPoolAllocator.h>
#include <nn/vfx.h>
#include <nn/vi.h>
#include <nn/util/util_StringUtil.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nns/gfx/gfx_GraphicsFramework.h>


#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#include <nn/hws/hws_Message.h>
#endif

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
#include <nv/nv_MemoryManagement.h>
#endif

void*                           g_HeapPtr;              // エフェクトヒープバッファポインタ
nn::vfx::StandardHeap           g_VfxHeap;              // エフェクトヒープ

                                                        //------------------------------------------------------------------------------
                                                        //  テクスチャデスクリプタスロットをアロケート
                                                        //------------------------------------------------------------------------------
bool AllocateDescriptorSlotForTexture( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
{
    if ( pUserData )
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>( pUserData );
        int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        if ( slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex )
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView );

        pPool->BeginUpdate();
        {
            pPool->SetTextureView( slotIndex, &textureView );
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot( dstSlot, slotIndex );
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  テクスチャデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForTexture( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::TextureView& textureView, void* pUserData )
{
    NN_UNUSED( textureView );

    if ( pUserData )
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>( pUserData );
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView );
        int slotIndex = pPool->GetDescriptorSlotIndex( *dstSlot );
        graphicsFramework->FreeDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, slotIndex );
    }
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットをアロケート
//------------------------------------------------------------------------------
bool AllocateDescriptorSlotForSampler( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData )
{
    if ( pUserData )
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>( pUserData );
        int slotIndex = graphicsFramework->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        if ( slotIndex == nn::gfx::util::DescriptorPoolAllocator::InvalidIndex )
        {
            return false;
        }

        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler );

        pPool->BeginUpdate();
        {
            pPool->SetSampler( slotIndex, &sampler );
        }
        pPool->EndUpdate();
        pPool->GetDescriptorSlot( dstSlot, slotIndex );
        return true;
    }

    return false;
}

//------------------------------------------------------------------------------
//  サンプラデスクリプタスロットを開放する
//------------------------------------------------------------------------------
void FreeDescriptorSlotForSampler( nn::gfx::DescriptorSlot* dstSlot, const nn::gfx::Sampler& sampler, void* pUserData )
{
    NN_UNUSED( sampler );

    if ( pUserData )
    {
        nns::gfx::GraphicsFramework* graphicsFramework = reinterpret_cast<nns::gfx::GraphicsFramework*>( pUserData );
        nn::gfx::DescriptorPool* pPool = graphicsFramework->GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler );
        int slotIndex = pPool->GetDescriptorSlotIndex( *dstSlot );
        graphicsFramework->FreeDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, slotIndex );
    }
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ取得用関数
//------------------------------------------------------------------------------
void* Allocate( size_t size )
{
    return malloc( size );
}

//------------------------------------------------------------------------------
//  nn::fs用メモリ解放用関数
//------------------------------------------------------------------------------
void Deallocate( void* p, size_t size )
{
    NN_UNUSED( size );
    free( p );
}

//------------------------------------------------------------------------------
//  nn::fs を利用したファイルロード関数
//------------------------------------------------------------------------------
bool FileLoad(
    void**           pOutBuffer,
    size_t*          pOutBufferSize,
    const char*      filePath,
    nn::vfx::Heap*   pAllocator,
    int              alignment )
{
    char filePathName[ nn::fs::EntryNameLengthMax ];
    memcpy( reinterpret_cast< void* >( &filePathName ), filePath, strlen( filePath ) + 1 );

    nn::fs::FileHandle fsHandle;
    nn::Result ret = nn::fs::OpenFile( &fsHandle, static_cast< char* >( filePathName ), static_cast< nn::fs::OpenMode >( nn::fs::OpenMode_Read ) );
    if ( ret.IsFailure() )
    {
        return false;
    }

    int64_t fsSize;
    nn::fs::GetFileSize( &fsSize, fsHandle );
    *pOutBufferSize = static_cast< size_t >( fsSize );

    *pOutBuffer = pAllocator->Alloc( *pOutBufferSize + 4, alignment );
    if ( !( *pOutBuffer ) )
    {
        nn::fs::CloseFile( fsHandle );
        return false;
    }

    memset( *pOutBuffer, 0, *pOutBufferSize + 4 );
    nn::fs::ReadFile( fsHandle, 0, *pOutBuffer, *pOutBufferSize );
    nn::fs::CloseFile( fsHandle );

    return true;
}

nns::gfx::GraphicsFramework::FrameworkInfo  g_GraphicsFrameworkInfo;
nns::gfx::GraphicsFramework                 g_GraphicsFramework;
nn::util::Vector3fType                      g_CamPosVec = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 15, 25 );  // カメラ位置
nn::util::Vector3fType                      g_CamLookAtVec = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 0, 0 );    // カメラ注視点
nn::util::Vector3fType                      g_CamUpVec = NN_UTIL_VECTOR_3F_INITIALIZER( 0, 1, 0 );    // カメラ上方向
nn::util::Matrix4x3fType                    g_ViewMtx;                              // ビュー行列
nn::util::Matrix4x4fType                    g_ProjectionMatrix;                     // 射影変換行列
float                                       g_NearZ = 0.1f;             // Z Near
float                                       g_FarZ = 10000.0f;         // Z Far
nn::vfx::System*                            g_pVfxSystem = nullptr;          // nn::vfx システム



                                                                             //  コマンドバッファイベントコールバック
void OutOfMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer, const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_UNUSED( pCommandBuffer );
    NN_UNUSED( arg );
}


// テクスチャバイナリのロード
bool LoadTextureBinary( const char* filePath, void** textureBinary, nn::gfx::ResTextureFile** pResTextureFile )
{
    size_t textureBinarySize = 0;

    bool result = FileLoad( textureBinary, &textureBinarySize, filePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment );
    if ( result == false )
    {
        return false;
    }

    *pResTextureFile = nn::gfx::ResTextureFile::ResCast( *textureBinary );

    return true;
}

// ResTextureFileを初期化する
void InitialieResTextureFile( nn::gfx::ResTextureFile* pResTextureFile )
{
    pResTextureFile->Initialize( g_GraphicsFramework.GetDevice() );

    // テクスチャ初期化
    for ( int idxTexture = 0, textureCount = pResTextureFile->GetTextureDic()->GetCount();
        idxTexture < textureCount; ++idxTexture )
    {
        pResTextureFile->GetResTexture( idxTexture )->Initialize( g_GraphicsFramework.GetDevice() );
    }

    // ディスクリプタスロット
    for ( int i = 0, textureCount = pResTextureFile->GetTextureDic()->GetCount();
        i < textureCount; i++ )
    {
        nn::gfx::ResTexture* resTexture = pResTextureFile->GetResTexture( i );
        nn::gfx::TextureView& textureView = *( static_cast<nn::gfx::TextureView*>( resTexture->GetTextureView() ) );

        nn::gfx::DescriptorSlot slot;
        AllocateDescriptorSlotForTexture( &slot, textureView, &g_GraphicsFramework );

        resTexture->SetUserDescriptorSlot( slot );
    }
}

// シェーダバイナリをロード
bool LoadShaderBinary( const char* filePath, void** shaderBinary, nn::gfx::ResShaderFile** pResShaderFile )
{
    size_t shaderBinarySize = 0;

    bool result = FileLoad( shaderBinary, &shaderBinarySize, filePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment );
    if ( result == false )
    {
        return false;
    }

    *pResShaderFile = nn::gfx::ResShaderFile::ResCast( *shaderBinary );
    ( *pResShaderFile )->GetShaderContainer()->Initialize( g_GraphicsFramework.GetDevice() );

    return true;
}

// G3dプリミティブのロード
bool LoadG3dPrimitiveBinary( const char* filePath, void** g3dPrimitiveBinary, nn::g3d::ResFile** pResG3dPrimitiveFile )
{
    size_t textureBinarySize = 0;

    bool result = FileLoad( g3dPrimitiveBinary, &textureBinarySize, filePath, &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment );
    if ( result == false )
    {
        return false;
    }

    *pResG3dPrimitiveFile = nn::g3d::ResFile::ResCast( *g3dPrimitiveBinary );

    return true;
}


//------------------------------------------------------------------------------
//  メイン関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // VFXヒープ初期化
    int memSize = 1024 * 1024 * 64 ;
    g_HeapPtr = malloc( memSize );
    NN_ASSERT_NOT_NULL( g_HeapPtr );
    g_VfxHeap.Initialize( g_HeapPtr, memSize );

#if !defined( NN_BUILD_TARGET_PLATFORM_OS_CAFE )
    // ファイルシステムのアロケータ設定
    nn::fs::SetAllocator( Allocate, Deallocate );
#endif

    char* mountRomCacheBuffer = nullptr;
    {
        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize( &cacheSize );
        NN_ASSERT( result.IsSuccess() );

        mountRomCacheBuffer = new( std::nothrow ) char[ cacheSize ];
        NN_ASSERT_NOT_NULL( mountRomCacheBuffer );

        result = nn::fs::MountRom( "Contents", mountRomCacheBuffer, cacheSize );
        NN_ASSERT( result.IsSuccess() );
        NN_UNUSED( result );
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_NN ) && defined( NN_BUILD_APISET_NX )
    static const size_t   GraphicsSystemMemorySize = 8 * 1024 * 1024;
    g_GraphicsFramework.InitializeGraphicsSystem( GraphicsSystemMemorySize );
#endif

    //------------------------------------------
    // GraphicsFrameWorkの初期化
    //------------------------------------------
    g_GraphicsFrameworkInfo.SetDefault();
    g_GraphicsFrameworkInfo.SetDisplayWidth( 1280 );
    g_GraphicsFrameworkInfo.SetDisplayHeight( 720 );
    g_GraphicsFrameworkInfo.SetColorBufferFormat( nn::gfx::ImageFormat_R16_G16_B16_A16_Float );
    g_GraphicsFrameworkInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 32 * 1024 * 1024 );
    g_GraphicsFramework.Initialize( g_GraphicsFrameworkInfo );

    // カメラ・プロジェクション行列初期化
    nn::util::MatrixLookAtRightHanded( &g_ViewMtx, g_CamPosVec, g_CamLookAtVec, 0.0f );
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >( g_GraphicsFramework.GetDisplayWidth() ) / static_cast< float >( g_GraphicsFramework.GetDisplayHeight() );
    nn::util::MatrixPerspectiveFieldOfViewRightHanded( &g_ProjectionMatrix, fovy, aspect, g_NearZ, g_FarZ );

    // ビュー行列
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::MatrixLookAtRightHanded( &viewMatrix, g_CamPosVec, g_CamLookAtVec, 0.0f );


    //------------------------------------------
    // VFXランタイムの初期化
    //------------------------------------------
    // Vfx System 生成
    nn::vfx::Config  config;
    config.SetEffectHeap( &g_VfxHeap );
    config.SetEffectDynamicHeap( &g_VfxHeap );
    config.SetGfxDevice( g_GraphicsFramework.GetDevice() );
    g_pVfxSystem = new nn::vfx::System( config );
    g_pVfxSystem->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->RegisterSamplerToDescriptorPool( AllocateDescriptorSlotForSampler, &g_GraphicsFramework );


    //------------------------------------------
    // VFXバイナリをロードする
    //------------------------------------------
    void*     useResidentTextureBinary      = nullptr;
    void*     useSeparateBinary             = nullptr;
    const int useResidentTextureBinaryId    = 0;
    const int useSeparateBinaryId           = 1;
    size_t binarySize = 0;

    // 常駐テクスチャバイナリを利用するエフェクトバイナリをロードし、エフェクトシステムに登録する
    bool result = false;
    result = FileLoad( &useResidentTextureBinary, &binarySize, "Contents:/UseResidentTexture.ptcl", &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment );
    NN_ASSERT( result, "UseResidentTexture.ptcl Load Failed" );
    g_pVfxSystem->EntryResource( &g_VfxHeap, useResidentTextureBinary, useResidentTextureBinaryId, true );
    g_pVfxSystem->GetResource( useResidentTextureBinaryId )->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework );

    // 分割出力された テクスチャ、G3dプリミティブ シェーダバイナリ を利用するエフェクトバイナリをロードし、エフェクトシステムに登録する
    result = FileLoad( &useSeparateBinary, &binarySize, "Contents:/SeparateBinary.ptcl", &g_VfxHeap, nn::vfx::SystemParameters_ParticleBinaryDataAlignment );
    NN_ASSERT( result, "SeparateBinary.ptcl Load Failed" );
    g_pVfxSystem->EntryResource( &g_VfxHeap, useSeparateBinary, useSeparateBinaryId, true );
    g_pVfxSystem->GetResource( useSeparateBinaryId )->RegisterTextureViewToDescriptorPool( AllocateDescriptorSlotForTexture, &g_GraphicsFramework );


    //-------------------------------------------------------------------------
    // 常駐テクスチャ バイナリロードし、エフェクトバイナリとバインドする
    //-------------------------------------------------------------------------
    void* residentTextureBinary = nullptr;
    nn::gfx::ResTextureFile* pResidentResTextureFile = nullptr;

    // 常駐テクスチャとして出力したバイナリをロード
    result = LoadTextureBinary( "Contents:/ResidentTextures.bntx", &residentTextureBinary, &pResidentResTextureFile );
    NN_ASSERT( result, "Vfx ResidentTextureBinarySize Load Failed" );

    // ResTextureFileを初期化して、ディスクリプタ登録を行う。
    InitialieResTextureFile( pResidentResTextureFile );

    // エフェクトバイナリにロードした常駐テクスチャをバインドする
    g_pVfxSystem->GetResource( useResidentTextureBinaryId )->BindExternalResTextureFile( pResidentResTextureFile );


    //-------------------------------------------------------------------------
    // 分割出力されたテクスチャ、G3dプリミティブ シェーダ バイナリロードし、エフェクトバイナリとバインドする
    //-------------------------------------------------------------------------

    // 分割出力したテクスチャバイナリをロード、バインドする
    void* textureBinary = nullptr;
    nn::gfx::ResTextureFile* pSeparateResTextureFile = nullptr;

    result = LoadTextureBinary( "Contents:/Texture.bntx", &textureBinary, &pSeparateResTextureFile );
    NN_ASSERT( result, "Vfx TextureBinaryFile Load Failed" );

    // ResTextureFileを初期化して、ディスクリプタ登録を行う。
    InitialieResTextureFile( pSeparateResTextureFile );

    // エフェクトバイナリに nn::gfx::ResTextureFile をバインドする
    g_pVfxSystem->GetResource( useSeparateBinaryId )->BindExternalResTextureFile( pSeparateResTextureFile );


    // 分割出力したG3dプリミティブバイナリをロード、バインドする
    void* g3dPrimitiveBinary = nullptr;
    nn::g3d::ResFile* pG3dPrimitiveResFile = nullptr;

    result = LoadG3dPrimitiveBinary( "Contents:/G3dPrimitive.bfres", &g3dPrimitiveBinary, &pG3dPrimitiveResFile );
    NN_ASSERT( result, "Vfx G3dPrimitiveFile Load Failed" );

    // G3dプリミティブバイナリ をセットアップ
    pG3dPrimitiveResFile->Setup( g_GraphicsFramework.GetDevice() );

    // エフェクトバイナリに nn::g3d::ResFile をバインドする
    g_pVfxSystem->GetResource( useSeparateBinaryId )->BindExternalG3dResFile( pG3dPrimitiveResFile );


    // 分割出力したシェーダバイナリをロード、バインドする
    void* shaderBinary = nullptr;
    nn::gfx::ResShaderFile* pResShaderFile = nullptr;
    void* computeShaderBinary = nullptr;
    nn::gfx::ResShaderFile* pResComputeShaderFile = nullptr;

    result = LoadShaderBinary( "Contents:/Shader.bnsh", &shaderBinary, &pResShaderFile );
    NN_ASSERT( result, "VfxShaderBinaryFile Load Failed." );

    result = LoadShaderBinary( "Contents:/ComputeShader.bnsh", &computeShaderBinary, &pResComputeShaderFile );
    NN_ASSERT( result, "VfxComputeShaderBinary Load Failed." );

    // エフェクトバイナリにロードしたシェーダ、コンピュートシェーダバイナリをバインドする
    g_pVfxSystem->GetResource( useSeparateBinaryId )->BindExternalResShaderFile( pResShaderFile, pResComputeShaderFile );


    int frame              = 0;
    int groupID            = 0;
    int emitterSetId       = 0;
    int emitterSetMaxCount = g_pVfxSystem->GetResource( useResidentTextureBinaryId )->GetEmitterSetCount();

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    MSG  msg;
#endif
    // メインループ
    for ( ; ; )
    {
        // テクスチャ取得と vsync 待ち
        g_GraphicsFramework.AcquireTexture(0);
        g_GraphicsFramework.WaitDisplaySync(0, nn::TimeSpan::FromSeconds(2));

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
        //------------------------------------------
        // Windows メッセージ処理
        //------------------------------------------
        if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE ) )
        {
            TranslateMessage( &msg );
            if ( msg.message == WM_QUIT )
            {
                break;
            }
            DispatchMessage( &msg );
        }
#endif

        // エフェクトの生成、破棄
        if ( ( frame % 300 ) == 0 )
        {
            if ( emitterSetId < emitterSetMaxCount )
            {
                g_pVfxSystem->KillEmitterSetGroup( groupID );
                for ( int i = 0; i < 2; i++ )
                {
                    nn::vfx::Handle emitterSetHandle;
                    g_pVfxSystem->CreateEmitterSetId( &emitterSetHandle, emitterSetId, i, groupID );
                    if ( emitterSetHandle.IsValid() )
                    {
                        NN_SDK_LOG( "%s\n", emitterSetHandle.GetEmitterSet()->GetName() );

                        nn::util::Matrix4x3fType matrix;
                        nn::util::Vector3fType pos = NN_UTIL_VECTOR_3F_INITIALIZER( 10.0f * i - 5.f, 0.0f, 0.0f );
                        nn::util::MatrixIdentity( &matrix );
                        nn::util::MatrixSetTranslate( &matrix, pos );
                        emitterSetHandle.GetEmitterSet()->SetMatrix( matrix );
                    }
                }
                emitterSetId++;
            }
            else
            {
                g_pVfxSystem->KillEmitterSetGroup( groupID );
                break;
            }
        }
        frame++;

        //------------------------------------------
        // VFXランタイムの計算処理
        //------------------------------------------
        g_pVfxSystem->BeginFrame();
        g_pVfxSystem->Calculate( groupID, 1.0f, true );

        g_GraphicsFramework.BeginFrame( 0 );
        {
            nn::gfx::CommandBuffer*    commandBuffer = g_GraphicsFramework.GetRootCommandBuffer( 0 );
            nn::gfx::ColorTargetView*  pColorTargetView = g_GraphicsFramework.GetColorTargetView();
            nn::gfx::DepthStencilView* pDepthStecilView = g_GraphicsFramework.GetDepthStencilView();

            commandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode );
            commandBuffer->ClearColor( pColorTargetView, 0.2f, 0.2f, 0.2f, 0.0f, nullptr );
            commandBuffer->ClearDepthStencil( pDepthStecilView, 1.0f, 0,
                nn::gfx::DepthStencilClearMode_DepthStencil, nullptr );

            commandBuffer->SetRenderTargets( 1, &pColorTargetView, pDepthStecilView );
            commandBuffer->SetViewportScissorState( g_GraphicsFramework.GetViewportScissorState() );

            //------------------------------------------
            // VFXランタイムの描画処理
            //------------------------------------------
            nn::vfx::ViewParam view;
            int processingIndex = 0;

            view.Set( viewMatrix, g_ProjectionMatrix, g_CamPosVec, g_NearZ, g_FarZ );
            g_pVfxSystem->SwapBuffer();
            g_pVfxSystem->SetViewParam( processingIndex, &view );
            g_pVfxSystem->AddSortBuffer( processingIndex, groupID );
            g_pVfxSystem->DrawSortBuffer( processingIndex, commandBuffer, true, nullptr );
        }
        g_GraphicsFramework.EndFrame( 0 );

        g_GraphicsFramework.ExecuteCommand( 0 );
        g_GraphicsFramework.QueuePresentTexture( 1 );
        g_GraphicsFramework.WaitGpuSync( 0, nn::TimeSpan::FromSeconds(2) );
    }

    //------------------------------------------
    // VFXランタイムの終了処理
    //------------------------------------------

    g_pVfxSystem->GetResource( useResidentTextureBinaryId )->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->ClearResource( &g_VfxHeap, useResidentTextureBinaryId );
    g_pVfxSystem->GetResource( useSeparateBinaryId )->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->ClearResource( &g_VfxHeap, useSeparateBinaryId );

    g_VfxHeap.Free( useResidentTextureBinary );
    g_VfxHeap.Free( useSeparateBinary );

    g_pVfxSystem->UnregisterTextureViewFromDescriptorPool( FreeDescriptorSlotForTexture, &g_GraphicsFramework );
    g_pVfxSystem->UnregisterSamplerFromDescriptorPool( FreeDescriptorSlotForSampler, &g_GraphicsFramework );
    delete g_pVfxSystem;

    // シェーダリソースの解放
    if ( shaderBinary )
    {
        pResComputeShaderFile->GetShaderContainer()->Finalize( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( computeShaderBinary );

        pResShaderFile->GetShaderContainer()->Finalize( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( shaderBinary );
    }

    // テクスチャリソースの解放
    if ( textureBinary )
    {
        pSeparateResTextureFile->Finalize( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( textureBinary );
    }

    // G3dリソースの解放
    if ( g3dPrimitiveBinary )
    {
        pG3dPrimitiveResFile->Cleanup( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( g3dPrimitiveBinary );
    }

    // 常駐テクスチャリソースの解放
    if ( residentTextureBinary )
    {
        pResidentResTextureFile->Finalize( g_GraphicsFramework.GetDevice() );
        g_VfxHeap.Free( residentTextureBinary );
    }

    nn::fs::Unmount( "Contents" );
    delete[] mountRomCacheBuffer;

    g_VfxHeap.Finalize();
    free( g_HeapPtr );

    return;
} // NOLINT(readability/fn_size)
