﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{UtilCompression.cpp,PageSampleUtilCompression}
 *
 * @brief
 *  圧縮ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleUtilCompression 圧縮
 * @tableofcontents
 *
 * @brief
 *  圧縮ライブラリを用いたサンプルプログラムの解説です。
 *
 * @section PageSampleUtilCompression_SectionBrief 概要
 *  ここでは、圧縮ライブラリを用いてデータの圧縮を行うサンプルプログラムの説明をします。
 *
 *  圧縮ライブラリの使い方については、
 *  @ref nn::util "ユーティリティライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleUtilCompression_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/UtilCompression
 *  Samples/Sources/Applications/UtilCompression @endlink 以下にあります。
 *
 * @section PageSampleUtilCompression_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleUtilCompression_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleUtilCompression_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleUtilCompression_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleUtilCompression_SectionDetail 解説
 *
 * @subsection PageSampleUtilCompression_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  UtilCompression.h
 *  @includelineno UtilCompression.h
 *
 *  UtilCompression.cpp
 *  @includelineno UtilCompression.cpp
 *
 * @subsection PageSampleUtilCompression_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムでは文字列 SampleData の gzip 形式での圧縮を行います。@n
 *  圧縮ライブラリが対応している他の圧縮形式には zlib 形式と raw deflate 形式があります。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 圧縮元データである SampleData のサイズを表示
 *  - 圧縮に必要なワークバッファ pWorkBuffer のメモリ領域確保
 *      - ワークバッファに必要なサイズは nn::util::CompressGzipWorkBufferSizeDefault の値から取得
 *  - 圧縮ライブラリの nn::util::CompressGzip() を呼び出して圧縮
 *      - pOutCompressedBuffer に圧縮データを格納
 *      - false が返ってきた場合は出力バッファサイズが不足
 *  - 圧縮データのサイズを表示
 *
 *  このサンプルプログラムでは圧縮オプションを指定せずに圧縮を行っていますが、
 *  圧縮ライブラリでは圧縮レベルと使用メモリレベルを指定することができます。@n
 *  gzip 形式で圧縮オプションを指定する場合は、必要なワークバッファのサイズを
 *  NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE() で取得して下さい。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  UtilCompression_OutputExample.txt
 *
 */


#include <algorithm>
#include <memory>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/util/util_Compression.h>

#include "UtilCompression.h"

//-----------------------------------------------------------------------------

namespace {

//
// オプション指定せず gzip 形式の圧縮を行うサンプル関数
//
void SampleCompressGzip(size_t* pOutCompressedDataSize, void* pOutCompressedBuffer,
                        size_t compressedBufferSize, const char* pSrcData, size_t srcDataSize)
{
    // gzip 形式圧縮に必要なワークバッファのサイズ
    const size_t workBufferSize = nn::util::CompressGzipWorkBufferSizeDefault;

    // ワークバッファをヒープから取得
    std::unique_ptr<char[]> pWorkBuffer(new char[workBufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(pWorkBuffer.get());

    // gzip 形式で圧縮
    bool isSuccess = nn::util::CompressGzip(pOutCompressedDataSize, pOutCompressedBuffer,
                compressedBufferSize, pSrcData, srcDataSize, pWorkBuffer.get(), workBufferSize);

    // ライブラリからの返り値をチェック
    NN_LOG("Compression (gzip): %s\n", (isSuccess) ? "OK." : "NG.");
    if(!isSuccess)
    {
        // ライブラリが false を返した場合は出力バッファのサイズが不足
        NN_LOG(" (the output buffer lacks in the size.)\n");
    }
}

} // namespace
//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    // 圧縮データを格納するバッファ（圧縮元の SampleData が小さいためスタックに確保）
    char pCompressedBuffer[SampleDataSize];
    size_t  compressedDataSize; // 圧縮データのサイズ

    // 圧縮前のデータサイズを表示
    NN_LOG("\nOriginal data size ( %d Bytes )\n", SampleDataSize);

    // gzip 形式での圧縮
    SampleCompressGzip(&compressedDataSize, pCompressedBuffer, SampleDataSize,
        SampleData, SampleDataSize);

    // 圧縮後のデータサイズを表示
    NN_LOG("Compressed data size ( %d Bytes )\n\n", compressedDataSize);
}

