﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UserMigration_ServerSequence.h"

#include <cstring>

#include <nn/nn_Result.h>
#include <nn/account/account_Api.h>
#include <nn/account/account_Result.h>
#include <nn/account/account_Selector.h>
#include <nn/migration/migration_UserMigrationApi.h>
#include <nn/migration/migration_Result.h>
#include <nn/settings/system/settings_SerialNumber.h>

#include "UserMigration.h"

namespace {
class SimpleServerAsyncSceneBase
    : public ServerSequence::SceneBase
{
protected:
    typedef ServerSequence::SceneBase Base;

private:
    nn::migration::UserMigrationServer& m_Server;
    nn::migration::AsyncContext(nn::migration::UserMigrationServer::*m_AsyncFunction)();
    char m_Label[128];
    bool m_IsSignaled;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SimpleServerAsyncSceneBase*>(data);
        obj.m_IsSignaled = true;
    }

    static bool HandleResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Server] HandleResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SimpleServerAsyncSceneBase*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        auto& components = *obj.GetComponentsPtr();
        components.ClearOption();
        components.AddOption("OK", obj.GetEndHandler());
        return false;
    }

protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void*,size_t)  = 0;

public:
    SimpleServerAsyncSceneBase(
        const char* label,
        nn::migration::UserMigrationServer& server,
        nn::migration::AsyncContext (nn::migration::UserMigrationServer::*asyncFunction)(),
        SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Server(server)
        , m_AsyncFunction(asyncFunction)
        , m_IsSignaled(false)
    {
        std::strncpy(m_Label, label, sizeof(m_Label));

        auto& components = *GetComponentsPtr();
        components.Reset(m_Label, nullptr, this);
        components.AddOption("Start", SignalHandler);
    }
    virtual void Update() NN_NOEXCEPT final NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        NNS_MIGRATION_LOG_LN("[Server] Start async function");
        SequenceBase::AsyncTask async = {(m_Server.*m_AsyncFunction)(), HandleResult, this, sizeof(*this)};
        PushAsyncProgress("Processing", std::move(async));
    }
};

class SceneStart final
    : public ServerSequence::SceneBase
{
private:
    typedef ServerSequence::SceneBase Base;
public:
    SceneStart() NN_NOEXCEPT
    {
        GetComponentsPtr()->Reset("Starting server", nullptr);
        EndScene(Base::SceneKind::CreateServer);
    }
};
class SceneEnd final
    : public ServerSequence::SceneBase
{
private:
    ServerSequence& m_Parent;

    static void Handler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneEnd*>(data);
        obj.m_Parent.RequestClose();
        obj.GetComponentsPtr()->ClearOption();
    }

public:
    explicit SceneEnd(ServerSequence& parent) NN_NOEXCEPT
        : m_Parent(parent)
    {
        auto& components = *GetComponentsPtr();
        components.Reset("Server sequence ended", nullptr, this);
        components.AddOption("Dispose server object", Handler);
    }
};
class SceneCreateServer final
    : public ServerSequence::SceneBase
{
private:
    typedef ServerSequence::SceneBase Base;
    nn::migration::UserMigrationServer& m_Server;
    void* m_Buffer;
    size_t m_BUfferSize;
    bool m_IsNewServer;
    bool m_IsSignaled;
    bool m_IsAborting;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneCreateServer*>(data);
        obj.m_IsSignaled = true;
    }

    static void AbortHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneCreateServer*>(data);
        obj.m_IsAborting = true;
    }

public:
    SceneCreateServer(nn::migration::UserMigrationServer* pServer, void* buffer, size_t bufferSize, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Server(*pServer)
        , m_Buffer(buffer)
        , m_BUfferSize(bufferSize)
        , m_IsSignaled(false)
        , m_IsAborting(false)
    {
        auto state = nn::migration::GetLastUserMigrationState();
        NN_ABORT_UNLESS(false
            || state == nn::migration::LastUserMigrationState::Unresumable
            || state == nn::migration::LastUserMigrationState::ServerResumable
            || state == nn::migration::LastUserMigrationState::ServerResumeSuggested);
        m_IsNewServer = (state == nn::migration::LastUserMigrationState::Unresumable);

        if (m_IsNewServer)
        {
            auto& components = *GetComponentsPtr();
            components.Reset("Select user to emigrate", "User account must be linked with a Nintendo Account.", this);
            components.AddOption("Select user and create new server", SignalHandler);
        }
        else
        {
            auto& components = *GetComponentsPtr();
            components.Reset(
                "Suspended server is found",
                state == nn::migration::LastUserMigrationState::ServerResumeSuggested
                    ? "state: LastUserMigrationState::ServerResumeSuggested"
                    : "state: LastUserMigrationState::ServerResumable",
                this);
            components.AddOption("Resume server", SignalHandler);
            components.AddOption("Abort ongoing emigration", AbortHandler);
        }
    }
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        if (m_IsAborting)
        {
            NNS_MIGRATION_LOG_LN("[Server] Aborting server");
            NN_SDK_ASSERT(!m_IsNewServer);
            nn::migration::CleanupUserMigration(m_Buffer, m_BUfferSize);
            PushError("This is not an error, but ongoing emigration is aborted.");
            return;
        }

        if (m_IsNewServer)
        {
            NNS_MIGRATION_LOG_LN("[Server] Opening psel");

            nn::account::UserSelectionSettings settings = {};
            settings.isNetworkServiceAccountRequired = true;

            nn::account::Uid uid;
            auto r = nn::account::ShowUserSelector(&uid, settings);
            if (!r.IsSuccess())
            {
                if (nn::account::ResultCancelledByUser::Includes(r))
                {
                    PushError("User selection is canceled.");
                    return;
                }
                NN_ABORT_UNLESS_RESULT_SUCCESS(r);
            }

            NNS_MIGRATION_LOG_LN("[Server] Creating server");
            nn::settings::system::SerialNumber sn;
            nn::settings::system::GetSerialNumber(&sn);
            nn::migration::UserMigrationServerProfile profile = {};
            std::strncpy(reinterpret_cast<char*>(profile.data), sn.string, sizeof(profile.data));
            m_Server = nn::migration::CreateUserMigrationServer(uid, profile, m_Buffer, m_BUfferSize);
        }
        else
        {
            NNS_MIGRATION_LOG_LN("[Server] Resumimg server");
            m_Server = nn::migration::ResumeUserMigrationServer(m_Buffer, m_BUfferSize);
        }

        auto uid = m_Server.GetUid();
        nn::account::Nickname nickname;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetNickname(&nickname, uid));
        char description[128];
        nn::util::SNPrintf(description, sizeof(description), "\"%s\"(%016llx_%016llx)", nickname.name, uid._data[0], uid._data[1]);

        NNS_MIGRATION_LOG_LN("[Server] User: %s", description);

        auto& components = *GetComponentsPtr();
        components.Reset("User to emigrate", description, this);
        components.AddOption("OK", Base::EndHandler<decltype(*this), Base::SceneKind::Prepare>);
    }
};

class ScenePrepare final
    : public SimpleServerAsyncSceneBase
{
private:
    nn::migration::UserMigrationServer& m_Server;
protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void* data, size_t dataSize)
    {
        if (m_Server.IsConnectionRequired())
        {
            return &Base::EndHandler<decltype(*this), Base::SceneKind::ProcessConnection>;
        }
        else
        {
            return &Base::EndHandler<decltype(*this), Base::SceneKind::Complete>;
        }
    }

public:
    ScenePrepare(nn::migration::UserMigrationServer& server, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : SimpleServerAsyncSceneBase("Preparing migration", server, &nn::migration::UserMigrationServer::PrepareAsync, dialogCaller)
        , m_Server(server)
    {
    }
};

class SceneProcessConnection final
    : public ServerSequence::SceneBase
{
private:
    typedef ServerSequence::SceneBase Base;
    nn::migration::UserMigrationServer& m_Server;
    bool m_IsSignaled;
    bool m_IsWaitStarted;
    bool m_IsAccepted;
    bool m_IsDeclined;

    static void SignalHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        obj.m_IsSignaled = true;
    }
    static void ConnectHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        obj.m_IsWaitStarted = true;
    }
    static void AcceptHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        obj.m_IsAccepted = true;
    }
    static void DeclineHandler(void* data, size_t dataSize) NN_NOEXCEPT
    {
        SignalHandler(data, dataSize);

        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        obj.m_IsDeclined = true;
    }

    static bool HandleConnectResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Server] HandleConnectionResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        nn::migration::UserMigrationClientProfile profile;
        obj.m_Server.GetClientProfile(&profile);
        char description[128];
        nn::util::SNPrintf(description, sizeof(description), "Device \"%s\"", profile.data);

        NNS_MIGRATION_LOG_LN("[Server] Client: %s", description);

        auto& components = *obj.GetComponentsPtr();
        components.Reset("Following client is connecting", description, &obj);
        components.AddOption("Accept connection", AcceptHandler);
        components.AddOption("Decline connection", DeclineHandler);
        return false;
    }

    static bool HandleAcceptResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Server] HandleAcceptResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        nn::migration::UserMigrationClientProfile profile;
        obj.m_Server.GetClientProfile(&profile);
        char description[128];
        nn::util::SNPrintf(description, sizeof(description), "\"%s\"", profile.data);

        auto& components = *obj.GetComponentsPtr();
        components.Reset("Connected with client device", description, &obj);
        components.AddOption("OK", Base::EndHandler<decltype(obj), Base::SceneKind::ProcessTransfer>);
        return false;
    }

    static bool HandleDelineResult(void* data, size_t dataSize, nn::Result r) NN_NOEXCEPT
    {
        NNS_MIGRATION_LOG_LN("[Server] HandleDelineResult: %08lx", r.GetInnerValueForDebug());

        NN_UNUSED(dataSize);
        auto& obj = *reinterpret_cast<SceneProcessConnection*>(data);
        if (obj.Base::HandleResult(r))
        {
            return true;
        }

        auto& components = *obj.GetComponentsPtr();
        components.ClearOption();
        obj.PushError("Connection declinature done");
        return true;
    }

public:
    SceneProcessConnection(nn::migration::UserMigrationServer& server, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : Base(dialogCaller)
        , m_Server(server)
        , m_IsSignaled(false)
        , m_IsWaitStarted(false)
        , m_IsAccepted(false)
        , m_IsDeclined(false)
    {
        auto& components = *GetComponentsPtr();
        components.Reset("Start waiting for connection", nullptr, this);
        components.AddOption("Start", ConnectHandler);
    }
    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        if (!m_IsSignaled)
        {
            return;
        }
        m_IsSignaled = false;

        if (m_IsWaitStarted)
        {
            NNS_MIGRATION_LOG_LN("[Server] Waiting for client to connect");
            m_IsWaitStarted = false;
            SequenceBase::AsyncTask async = {m_Server.WaitConnectionAsync(), HandleConnectResult, this, sizeof(*this)};
            PushAsyncProgress("Waiting", std::move(async));
            return;
        }
        else if (m_IsAccepted)
        {
            NNS_MIGRATION_LOG_LN("[Server] Connection will be accepted");
            m_IsAccepted = false;
            SequenceBase::AsyncTask async = {m_Server.AcceptConnectionAsync(), HandleAcceptResult, this, sizeof(*this)};
            PushAsyncProgress("Accepting", std::move(async));
            return;
        }
        else if (m_IsDeclined)
        {
            NNS_MIGRATION_LOG_LN("[Server] Connection will be declined");
            m_IsDeclined = false;
            SequenceBase::AsyncTask async = {m_Server.DeclineConnectionAsync(), HandleDelineResult, this, sizeof(*this)};
            PushAsyncProgress("Declining", std::move(async));
            return;
        }
    }
};

class SceneProcessTransfer final
    : public SimpleServerAsyncSceneBase
{
protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void* data, size_t dataSize)
    {
        return Base::EndHandler<decltype(*this), Base::SceneKind::Complete>;
    }
public:
    SceneProcessTransfer(nn::migration::UserMigrationServer& server, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : SimpleServerAsyncSceneBase("Processing data transfer request", server, &nn::migration::UserMigrationServer::ProcessTransferAsync, dialogCaller)
    {
    }
};

class SceneComplete final
    : public SimpleServerAsyncSceneBase
{
protected:
    virtual void (*GetEndHandler() NN_NOEXCEPT) (void* data, size_t dataSize)
    {
        return Base::EndHandler<decltype(*this), Base::SceneKind::End>;
    }
public:
    SceneComplete(nn::migration::UserMigrationServer& server, SequenceBase::DialogCaller& dialogCaller) NN_NOEXCEPT
        : SimpleServerAsyncSceneBase("Finalizing emigration", server, &nn::migration::UserMigrationServer::CompleteAsync, dialogCaller)
    {
    }
};

Mutex g_BufferLock = {NN_OS_MUTEX_INITIALIZER(false)};
std::aligned_storage<nn::migration::RequiredWorkBufferSizeForUserMigrationOperation, nn::os::MemoryPageSize>::type g_Buffer;

} // ~namespace <anonymous>

ServerSequence::ServerSequence(Window& window) NN_NOEXCEPT
    : SequenceBase(window)
    , m_CurrentScene(SceneKind::Start)
    , m_NextScene(SceneKind::Start)
    , m_IsLockAcquired(false)
{
    m_Scene.reset(new SceneStart());
    NotifyCurrentSceneUpdate();

    NNS_MIGRATION_LOG_LN("[Server] sequense created");
}
ServerSequence::~ServerSequence() NN_NOEXCEPT
{
    if (m_IsLockAcquired)
    {
        m_Server = decltype(m_Server)();
        g_BufferLock.Unlock();
    }

}
void ServerSequence::OnEntering(SceneKind next) NN_NOEXCEPT
{
    switch (next)
    {
    case SceneKind::Start:
        NN_ABORT("");

    case SceneKind::End:
        m_Scene.reset(new SceneEnd(*this));
        NNS_MIGRATION_LOG_LN("[Server] Scene entered: End");
        break;

    case SceneKind::CreateServer:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Start);
        {
            NN_ABORT_UNLESS(g_BufferLock.TryLock());
            m_IsLockAcquired = true;
            m_Scene.reset(new SceneCreateServer(&m_Server, &g_Buffer, sizeof(g_Buffer), GetDialogCallerRef()));
        }
        NNS_MIGRATION_LOG_LN("[Server] Scene entered: CreateServer");
        break;

    case SceneKind::Prepare:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::CreateServer);
        m_Scene.reset(new ScenePrepare(m_Server, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Server] Scene entered: Prepare");
        break;

    case SceneKind::ProcessConnection:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Prepare);
        m_Scene.reset(new SceneProcessConnection(m_Server, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Server] Scene entered: ProcessConnection");
        break;

    case SceneKind::ProcessTransfer:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::ProcessConnection);
        m_Scene.reset(new SceneProcessTransfer(m_Server, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Server] Scene entered: ProcessTransfer");
        break;

    case SceneKind::Complete:
        NN_ABORT_UNLESS(m_CurrentScene == SceneKind::Prepare || m_CurrentScene == SceneKind::ProcessTransfer);
        m_Scene.reset(new SceneComplete(m_Server, GetDialogCallerRef()));
        NNS_MIGRATION_LOG_LN("[Server] Scene entered: Complete");
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
    m_CurrentScene = next;
}
const Window::Components* ServerSequence::GetCurrentSceneComponentsPtr() const NN_NOEXCEPT
{
    return static_cast<const SceneBase*>(m_Scene.get())->GetComponentsPtr();
}
void ServerSequence::UpdateImpl() NN_NOEXCEPT
{
    if (IsErrorOccurred())
    {
        if (m_CurrentScene != SceneKind::End)
        {
            NNS_MIGRATION_LOG_LN("[Server] ServerSequence detects error");
        }
        m_NextScene = SceneKind::End;
    }

    if (m_NextScene != m_CurrentScene)
    {
        OnEntering(m_NextScene);
        NotifyCurrentSceneUpdate();
    }

    m_Scene->Update();
    if (m_Scene->IsEnded())
    {
        m_NextScene = m_Scene->GetNextScene();
    }
}
