﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>

#include "Ui2dSimple.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

// LoadToMemoryPool を true にすると、一つの MemoryPool にまとめてリソースを読み込む動作を確認できます。
// 読み込むリソースの数が多い場合、生成する MemoryPool の数が少なく抑えられるため初期化処理コストが軽減されます。
static const bool                  LoadToMemoryPool = false;
static nn::gfx::MemoryPool         g_ResourceMemoryPool;
static void*                       g_pResourceMemoryPoolBuffer = nullptr;

//------------------------------------------------------------------------------
// リソース読み込み
//------------------------------------------------------------------------------
void LoadResource(nn::gfx::Device* pDevice, nn::ui2d::ArcResourceAccessor* pArcResourceAccessor, nn::font::ResFont* pFont)
{
    // レイアウトアーカイブの読み込み
    {
        g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dDemo.arc", nn::ui2d::ArchiveResourceAlignment);
        {
            bool    result = pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
            NN_ASSERT(result);
        }
    }

    // フォントの読み込み
    {
        void* pFontResource = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool result = pFont->SetResource(pDevice, pFontResource);
        NN_ASSERT(result);
    }
}

//------------------------------------------------------------------------------
// リソース読み込み（一つの MemoryPool にまとめて読み込む）
//------------------------------------------------------------------------------
void LoadResourceToMemoryPool(nn::gfx::Device* pDevice, nn::ui2d::ArcResourceAccessor* pArcResourceAccessor, nn::font::ResFont* pFont,
    nn::gfx::MemoryPool* pMemoryPool, ptrdiff_t resourceMemoryPoolOffset, const size_t ResourceMemoryPoolSize)
{
    {
        ptrdiff_t startOffset = resourceMemoryPoolOffset;
        ptrdiff_t arcOffset;
        ptrdiff_t fntOffset;
        void* pFontResource = nullptr;
        {
            // マップして管理するメモリー領域へのポインタを取得します。
            void* pMemPoolBuffer = pMemoryPool->Map();

            // レイアウトアーカイブの読み込み
            {
                arcOffset = resourceMemoryPoolOffset;
                arcOffset = nn::util::align_up(arcOffset, nn::ui2d::ArchiveResourceAlignment);

                size_t fileSize = 0;
                g_pLayoutArchiveBinary = ReadFileToBuffer(&fileSize, "Contents:/Ui2dDemo.arc", pMemPoolBuffer, arcOffset);

                resourceMemoryPoolOffset = arcOffset + fileSize;
            }

            // フォントの読み込み
            {
                fntOffset = resourceMemoryPoolOffset;
                fntOffset = nn::util::align_up(fntOffset, nn::font::ResourceAlignment);

                size_t fileSize = 0;
                pFontResource = ReadFileToBuffer(&fileSize, "Contents:/sample.bffnt", pMemPoolBuffer, fntOffset);

                resourceMemoryPoolOffset += fntOffset + fileSize;
            }

            // 書き込んだ内容を CPU キャッシュからメモリへとフラッシュします。Unmap 前に呼び出します。
            pMemoryPool->FlushMappedRange(startOffset, resourceMemoryPoolOffset - startOffset);
            pMemoryPool->Unmap();
        }

        // メモリープールの内容が構築された後に初期化を行います。
        bool resultArcInit = pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".", pMemoryPool, arcOffset, ResourceMemoryPoolSize);
        NN_ASSERT(resultArcInit);
        bool resultFntInit = pFont->SetResource(pDevice, pFontResource, pMemoryPool, fntOffset, ResourceMemoryPoolSize);
        NN_ASSERT(resultFntInit);
    }
}

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeSimple()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサ・フォントの初期化
    {
        g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();
        g_pFont = AllocAndConstruct<nn::font::ResFont>();

        if (NN_STATIC_CONDITION(LoadToMemoryPool))
        {
            // リソース読み込み用メモリプールの初期化
            // メモリプールに、直接ファイル読み込みを行うには、MemoryPoolProperty_CpuCached を指定する必要があります。
            size_t ResourceMemoryPoolSize = 0;
            {
                nn::gfx::MemoryPool::InfoType info;
                info.SetDefault();
                info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuCached | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_ShaderCode);
                ResourceMemoryPoolSize = nn::util::align_up(4 * 1024 * 1024, nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(pDevice, info));
                g_pResourceMemoryPoolBuffer = Ui2dAllocateFunction(ResourceMemoryPoolSize, nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, info), NULL);
                NN_ASSERT_NOT_NULL(g_pResourceMemoryPoolBuffer);
                info.SetPoolMemory(g_pResourceMemoryPoolBuffer, ResourceMemoryPoolSize);
                g_ResourceMemoryPool.Initialize(pDevice, info);
            }

            ptrdiff_t resourceMemoryPoolOffset = 0;
            LoadResourceToMemoryPool(pDevice, g_pArcResourceAccessor, g_pFont, &g_ResourceMemoryPool, resourceMemoryPoolOffset, ResourceMemoryPoolSize);
        }
        else
        {
            LoadResource(pDevice, g_pArcResourceAccessor, g_pFont);
        }
    }

    // フォントの初期化
    {
        // フォントのテクスチャ用ディスクリプタスロットを登録
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
        // フォントをリソースアクセサーに登録
        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, "simple.bflyt");
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    // アニメーションの初期化
    {
        nn::ui2d::Animator* pAnimator = g_pLayout->CreateGroupAnimator(pDevice, "loop");
        pAnimator->PlayAuto(1.0f);

        // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
        g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
    }
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeSimple()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;

    {
        g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        void* pFontResource = g_pFont->RemoveResource(pDevice);
        if (NN_STATIC_CONDITION(!LoadToMemoryPool))
        {
            Ui2dDeallocateFunction(pFontResource, NULL);
        }
        g_pFont->Finalize(pDevice);
        DestructAndFree<nn::font::ResFont>(g_pFont);
        g_pFont = NULL;
    }

    {
        g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pArcResourceAccessor->Detach();
        g_pArcResourceAccessor->Finalize(pDevice);

        if (NN_STATIC_CONDITION(!LoadToMemoryPool))
        {
            // ArcResourceAccessor の Finalize でアクセスされるため Finalize 後に開放する。
            Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
            g_pLayoutArchiveBinary = NULL;
        }

        DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
        g_pArcResourceAccessor = NULL;
    }

    // リソース読み込み用メモリプールの解放
    if (NN_STATIC_CONDITION(LoadToMemoryPool))
    {
        g_ResourceMemoryPool.Finalize(pDevice);
        Ui2dDeallocateFunction(g_pResourceMemoryPoolBuffer, NULL);
        g_pResourceMemoryPoolBuffer = NULL;
    }
}
