﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>

#include "Ui2dMultiArcResource.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------

// ui2d レイアウト
static nn::ui2d::Layout*                    g_pLayout2 = NULL;
static nn::ui2d::Layout*                    g_pLayout3 = NULL;

// ui2d アーカイブリソースアクセッサ
static nn::ui2d::ArchiveHandle*             g_pArchiveHandle = NULL;
static nn::ui2d::ArchiveHandle*             g_pArchiveHandle2 = NULL;
static nn::ui2d::MultiArcResourceAccessor*  g_pMultiArcResourceAccessor1 = NULL;
static nn::ui2d::MultiArcResourceAccessor*  g_pMultiArcResourceAccessor2 = NULL;

// レイアウトアーカイブバイナリデータ
static void*                                g_pLayoutArchiveBinary2;

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeMultiArcResource()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pMultiArcResourceAccessor1 = AllocAndConstruct<nn::ui2d::MultiArcResourceAccessor>();
    g_pMultiArcResourceAccessor2 = AllocAndConstruct<nn::ui2d::MultiArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dDemo.arc", nn::ui2d::ArchiveResourceAlignment);
    g_pLayoutArchiveBinary2 = ReadFileWithAllocate("Contents:/Ui2dDemo2.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        // アーカイブリソースをg_pArchiveHandleに構築して、複数のMultiArcResourceAccessorで共有させる
        g_pArchiveHandle = AllocAndConstruct<nn::ui2d::ArchiveHandle>();
        g_pArchiveHandle2 = AllocAndConstruct<nn::ui2d::ArchiveHandle>();

        g_pArchiveHandle->Initialize(g_pLayoutArchiveBinary, ".");
        g_pArchiveHandle2->Initialize(g_pLayoutArchiveBinary2, ".");

        g_pMultiArcResourceAccessor1->Attach(g_pArchiveHandle);
        g_pMultiArcResourceAccessor1->Attach(g_pArchiveHandle2);
        // g_pArchiveHandle は Accessor1 と Accessor2 双方にアタッチします。共有参照されている状態です。
        g_pMultiArcResourceAccessor2->Attach(g_pArchiveHandle);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        g_pMultiArcResourceAccessor1->RegisterFont("sample.bffnt", g_pFont);
        g_pMultiArcResourceAccessor2->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();
    g_pLayout2 = AllocAndConstruct<nn::ui2d::Layout>();
    g_pLayout3 = AllocAndConstruct<nn::ui2d::Layout>();

    nn::ui2d::Layout::BuildOption   opt;
    opt.SetDefault();

    // 描画に必要なコンスタントバッファサイズを計算します。
    // 同時に表示するレイアウト分のバッファサイズを積算します。
    nn::ui2d::BuildResultInformation totalRequiredBufferInformation;
    totalRequiredBufferInformation.SetDefault();

    BuildLayoutData(totalRequiredBufferInformation, g_pLayout, opt, g_pMultiArcResourceAccessor1, "simple.bflyt");
    BuildLayoutData(totalRequiredBufferInformation, g_pLayout2, opt, g_pMultiArcResourceAccessor2, "simple.bflyt");
    BuildLayoutData(totalRequiredBufferInformation, g_pLayout3, opt, g_pMultiArcResourceAccessor1, "multiTexture.bflyt");

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用されるコンスタントバッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, totalRequiredBufferInformation, g_pUi2dConstantBuffer);

    // 表示位置・サイズの調整
    {
        nn::util::Float2 scale = NN_UTIL_FLOAT_2_INITIALIZER(0.2f, 0.2f);

        nn::util::Float2 t1 = NN_UTIL_FLOAT_2_INITIALIZER(-60.f, 50.f);
        g_pLayout->GetRootPane()->SetScale(scale);
        g_pLayout->GetRootPane()->SetTranslate(t1);

        nn::util::Float2 t2 = NN_UTIL_FLOAT_2_INITIALIZER(60.f, 50.f);
        g_pLayout2->GetRootPane()->SetScale(scale);
        g_pLayout2->GetRootPane()->SetTranslate(t2);

        nn::util::Float2 t3 = NN_UTIL_FLOAT_2_INITIALIZER(0.f, -50.f);
        g_pLayout3->GetRootPane()->SetScale(scale);
        g_pLayout3->GetRootPane()->SetTranslate(t3);
    }

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    // アニメーションの作成と再生
    nn::ui2d::GroupAnimator* pAnimator = g_pLayout->CreateGroupAnimator(pDevice, "loop");
    pAnimator->PlayAuto(1.0f);

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pMultiArcResourceAccessor1->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
    g_pMultiArcResourceAccessor2->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeMultiArcResource()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;

    if (g_pLayout2 != NULL)
    {
        g_pLayout2->Finalize(pDevice);
        DestructAndFree<nn::ui2d::Layout>(g_pLayout2);
        g_pLayout2 = NULL;
    }

    g_pLayout3->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout3);
    g_pLayout3 = NULL;

    g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
    void* pFontResource = g_pFont->RemoveResource(pDevice);
    Ui2dDeallocateFunction(pFontResource, NULL);
    g_pFont->Finalize(pDevice);
    DestructAndFree<nn::font::ResFont>(g_pFont);
    g_pFont = NULL;

    if (g_pMultiArcResourceAccessor2 != NULL)
    {
        g_pMultiArcResourceAccessor2->Detach(g_pArchiveHandle);
        g_pMultiArcResourceAccessor2->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pMultiArcResourceAccessor2->Finalize(pDevice);
        DestructAndFree<nn::ui2d::MultiArcResourceAccessor>(g_pMultiArcResourceAccessor2);
        g_pMultiArcResourceAccessor2 = NULL;
    }

    // g_pMultiArcResourceAccessor1はg_pArchiveHandleのDetachを行わずにFinalizeを呼ぶことで、
    // g_pArchiveHandleの解放が自動で行われます。
    g_pMultiArcResourceAccessor1->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
    g_pMultiArcResourceAccessor1->Finalize(pDevice);
    DestructAndFree<nn::ui2d::MultiArcResourceAccessor>(g_pMultiArcResourceAccessor1);
    g_pMultiArcResourceAccessor1 = NULL;

    g_pArchiveHandle->Finalize(pDevice);
    DestructAndFree<nn::ui2d::ArchiveHandle>(g_pArchiveHandle);
    g_pArchiveHandle = NULL;

    g_pArchiveHandle2->Finalize(pDevice);
    DestructAndFree<nn::ui2d::ArchiveHandle>(g_pArchiveHandle2);
    g_pArchiveHandle2 = NULL;

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
    Ui2dDeallocateFunction(g_pLayoutArchiveBinary2, NULL);
    g_pLayoutArchiveBinary = NULL;
    g_pLayoutArchiveBinary2 = NULL;
}

//------------------------------------------------------------------------------
// 計算処理
//------------------------------------------------------------------------------
void CalculateMultiArcResource(int frame)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // リソースアクセサの1つを3秒後に解放する
    if (frame == 60 * 3)
    {
        // Layoutの1つを解放する
        if (g_pLayout2 != NULL)
        {
            g_pLayout2->Finalize(pDevice);
            DestructAndFree<nn::ui2d::Layout>(g_pLayout2);
            g_pLayout2 = NULL;
        }

        // MultiArcResourceAccessorの1つを解放する
        if (g_pMultiArcResourceAccessor2 != NULL)
        {
            // 複数のリソースアクセサで、1つのアーカイブリソースを共有することができます。
            // 1つのアーカイブリソースを複数のリソースアクセサにAttachした場合、
            // 1つのリソースアクセサがFinalizeするとAttachしていたアーカイブリソースが自動的に解放されてしまうため、
            // 他のリソースアクセサに影響が及んでしまいます。
            // このように1つのリソースを別のアクセサでも使っている場合には、
            // リソースを解放したくないリソースアクセサからリソースをDetachすることでFinalize時に解放しないようにできます。
            g_pMultiArcResourceAccessor2->Detach(g_pArchiveHandle);

            // Detachしたので、g_pArchiveHandle内のリソースは
            // UnregisterTextureViewFromDescriptorPoolおよびFinalizeの影響を受けません。
            g_pMultiArcResourceAccessor2->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
            g_pMultiArcResourceAccessor2->Finalize(pDevice);

            DestructAndFree<nn::ui2d::MultiArcResourceAccessor>(g_pMultiArcResourceAccessor2);
            g_pMultiArcResourceAccessor2 = NULL;
        }
        NN_LOG("The second multiArcResourceAccessor was freed.\n");
    }
}

//------------------------------------------------------------------------------
// レイアウトの計算処理
//------------------------------------------------------------------------------
void CalculateMultiArcResourceLayout()
{
    if (g_pLayout2 != NULL)
    {
        g_pLayout2->Calculate(*g_pDrawInfo);
    }
    if (g_pLayout3 != NULL)
    {
        g_pLayout3->Calculate(*g_pDrawInfo);
    }
}

//------------------------------------------------------------------------------
// レイアウトの描画処理
//------------------------------------------------------------------------------
void DrawMultiArcResourceLayout(nn::gfx::CommandBuffer* pCmdBuffer)
{
    if (g_pLayout2 != NULL)
    {
        g_pLayout2->Draw(*g_pDrawInfo, *pCmdBuffer);
    }
    if (g_pLayout3 != NULL)
    {
        g_pLayout3->Draw(*g_pDrawInfo, *pCmdBuffer);
    }
}
