﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Ui2dMenu.h"

#include <nn/fs.h>
#include <nn/mem.h>
#include "Ui2dCommon.h"
#include "InputController.h"


enum Phase
{
    FadeIn,
    Calculate,
    WaitAnimation,
    FadeOut,
};

// ボタングループ
static nn::ui2d::ButtonGroup*               g_pButtonGroup;

// タッチもしくはクリックされているか
static bool                                 g_Hold;

static int                                  g_SelectCursor = 0;
static int                                  g_SelectCount = 0;
static Phase                                g_Phase = FadeIn;
static SampleMode                           g_SelectedSampleMode;
static nn::ui2d::Animator*                  g_Animator = NULL;

static int                                  g_ExitNumber = -1;

//------------------------------------------------------------------------------
//  ボタンが押されたときのコールバック
//------------------------------------------------------------------------------
static void LayoutButtonCallback(nn::ui2d::AnimButton* pButton, nn::ui2d::AnimButton::State prevState, nn::ui2d::AnimButton::State nextState, void* pParam)
{
    NN_UNUSED(pParam);

    if (prevState == nn::ui2d::AnimButton::State_OnIdle && nextState == nn::ui2d::AnimButton::State_Down)
    {
        nn::ui2d::Pane* pPane = pButton->GetHitPane()->GetParent();
        NN_ASSERT_NOT_NULL(pPane);
        const char* textChangePaneName = "L_parts_";
        if (strstr(pPane->GetName(), textChangePaneName) != NULL)
        {
            int number = atoi(pPane->GetName() + strlen(textChangePaneName));
            g_SelectedSampleMode = (SampleMode)number;
        }
        else if (strcmp(pPane->GetName(), "Exit_00") == 0)
        {
            g_SelectedSampleMode = (SampleMode)g_ExitNumber;
        }
    }
}

//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeMenu(const uint16_t** pMenuNames, int menuCount)
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Tutorial_Menu.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        // ctrlモジュールで用意されているボタンを作成するためのクラスを作成。ボタングループを引数として指定する。
        g_pButtonGroup = AllocAndConstruct<nn::ui2d::ButtonGroup>();
        nn::ui2d::DefaultControlCreator controlCreater(g_pButtonGroup);

        // 入力受付を変更する為にあたり判定を無効にする
        g_pButtonGroup->SetHitTestEnabled(false);

        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, &controlCreater, NULL, opt, "Tutorial_Menu.bflyt");
    }

    {
         //ボタンコールバックを設定する
        g_pButtonGroup->SetStateChangeCallbackAll(LayoutButtonCallback, NULL);
    }

    // 文字列の入れ替え
    nn::ui2d::Pane* pInOutPane = g_pLayout->GetRootPane()->FindPaneByName("N_InOut");
    NN_ASSERT_NOT_NULL(pInOutPane);
    nn::ui2d::PaneList::iterator endIter = pInOutPane->GetChildList().end();
    for (nn::ui2d::PaneList::iterator iter = pInOutPane->GetChildList().begin(); iter != endIter; ++iter)
    {
        const char* textChangePaneName = "L_parts_";
        if (strstr(iter->GetName(), textChangePaneName) == NULL)
        {
            continue;
        }
        int number = atoi(iter->GetName() + strlen(textChangePaneName));

        const uint16_t* pName = NULL;
        for (int i = 0; i < menuCount; i++)
        {
            if (number == i)
            {
                pName = *(pMenuNames + i);
            }
        }

        // メニュー項目以外のメニューバーを非表示にする
        if (pName == NULL)
        {
            iter->SetVisible(false);
            continue;
        }
        nn::ui2d::TextBox* pTextBox = static_cast<nn::ui2d::TextBox*>(iter->FindPaneByName("Text"));
        pTextBox->AllocateStringBuffer(g_GfxFramework.GetDevice(), 128);
        pTextBox->SetString(pName, 0);

        // 表示する文字数分フォント用のコンスタントバッファの領域を拡張しておく。
        buildResult.requiredFontConstantBufferSize += sizeof(nn::font::detail::VertexShaderCharAttribute) * 128;

        // フォント幅が変更される為、改行を無効にする
        pTextBox->SetWidthLimitEnabled(false);
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    // アニメーションの初期化
    {
        g_Animator = g_pLayout->CreateGroupAnimator(pDevice, "In");
        g_Animator->Play(nn::ui2d::Animator::PlayType::PlayType_OneShot, 1.0f);
    }

    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

    g_Hold = false;
    // 選択可能なメニュー数 ＋ 部品パーツEXIT
    g_SelectCount = menuCount + 1;
    g_Phase = FadeIn;
}

//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeMenu()
{
    // ボタンの破棄
    g_pButtonGroup->FreeAll();
    DestructAndFree<nn::ui2d::ButtonGroup>(g_pButtonGroup);
    g_pButtonGroup = NULL;

    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;
    g_Animator = NULL;

    {
        g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);

        void* pFontResource = g_pFont->RemoveResource(pDevice);
        Ui2dDeallocateFunction(pFontResource, NULL);
        g_pFont->Finalize(pDevice);
        DestructAndFree<nn::font::ResFont>(g_pFont);
        g_pFont = NULL;
    }

    {
        g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pArcResourceAccessor->Detach();
        g_pArcResourceAccessor->Finalize(pDevice);

        Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
        g_pLayoutArchiveBinary = NULL;

        DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
        g_pArcResourceAccessor = NULL;
    }
}

//------------------------------------------------------------------------------
//   カーソルの更新処理
//------------------------------------------------------------------------------
bool UpdateCursor(const nn::util::Float2& pos)
{
    bool bChangeMenu = false;

    // ボタンの当たり判定を更新
    // ユーザーがボタンを押したフレームにアニメーションを開始するためには、
    // この処理はレイアウトのAnimateメソッドを呼び出す前に行う必要がある。
    {
        bool isDown = false;

        if (GetPad().IsTriggered(Pad::BUTTON_UP))
        {
            g_SelectCursor--;
            if (g_SelectCursor < 0)
            {
                g_SelectCursor = g_SelectCount - 1;
            }
        }
        else if (GetPad().IsTriggered(Pad::BUTTON_DOWN))
        {
            g_SelectCursor++;
            if (g_SelectCursor >= g_SelectCount)
            {
                g_SelectCursor = 0;
            }
        }
        else if (GetPad().IsTriggered(Pad::BUTTON_A))
        {
            isDown = true;
        }
        else
        {
            nn::ui2d::ButtonList& buttonList = g_pButtonGroup->GetButtonList();
            nn::ui2d::ButtonList::iterator endIter = buttonList.end();
            for (nn::ui2d::ButtonList::iterator iter = buttonList.begin(); iter != endIter; ++iter)
            {
                // 部品パーツの部品・コントロール設定にある機能ペインの Hit に登録されている親ペイン( == 配置された部品パーツのペイン)が非表示状態のものは除外します。
                if (!iter->GetHitPane()->GetParent()->IsVisible())
                {
                    continue;
                }
                if (iter->IsActive() && iter->IsHit(pos))
                {
                    nn::ui2d::AnimButton* pHitButton = &(*iter);
                    nn::ui2d::Pane*pPane = pHitButton->GetHitPane()->GetParent();
                    NN_ASSERT_NOT_NULL(pPane);
                    if (strstr(pPane->GetName(), "L_parts_") != NULL)
                    {
                        int num = atoi(pPane->GetName() + strlen("L_parts_"));
                        g_SelectCursor = num;
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
                        // 実機の場合はタッチパネルなので、選択した段階で決定とする。
                        isDown = true;
#endif
                        break;
                    }
                    if (strcmp(pPane->GetName(), "Exit_00") == 0)
                    {
                        g_SelectCursor = g_SelectCount - 1;
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
                        // 実機の場合はタッチパネルなので、選択した段階で決定とする。
                        isDown = true;
#endif
                        break;
                    }
                }
            }
        }

        nn::ui2d::ButtonList& buttonList = g_pButtonGroup->GetButtonList();

        bool isExcludeDown = false;
        if (g_pButtonGroup->IsExcludeDown()) {
            // ボタンの中から、押されたときに他のボタンが押されるのを防ぐ設定になっていて、
            // かつ現在押されているボタンを探し、もしあったらこのフレームボタンには当たり
            // が発生しないようにする。
            nn::ui2d::ButtonList::iterator endIter = buttonList.end();
            for (nn::ui2d::ButtonList::iterator iter = buttonList.begin(); iter != endIter; ++iter)
            {
                if (iter->IsExcludeDown() && iter->IsDowning())
                {
                    isExcludeDown = true;
                    break;
                }
            }
        }

        if (!isExcludeDown)
        {
            int count = 0;
            nn::ui2d::ButtonList::iterator endIter = buttonList.end();
            for (nn::ui2d::ButtonList::iterator iter = buttonList.begin(); iter != endIter; ++iter)
            {
                // 部品パーツの部品・コントロール設定にある機能ペインの Hit に登録されている親ペイン( == 配置された部品パーツのペイン)が非表示状態のものは除外します。
                if (!iter->GetHitPane()->GetParent()->IsVisible())
                {
                    continue;
                }
                if (g_SelectCursor == count && iter->IsAcceptAction(nn::ui2d::ButtonBase::Action_On))
                {
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
                    // ボタン決定、又はメニューバー範囲内でのマウス決定ボタン
                    if (isDown == true || (g_Hold == true && iter->IsHit(pos)))
#else
                    if (isDown == true)
#endif
                    {
                        iter->Down();
                        bChangeMenu = true;
                    }
                    else
                    {
                        iter->On();
                    }
                }
                else
                {
                    iter->Off();
                }
                count++;
            }
        }

        // ボタン更新処理
        g_pButtonGroup->Update(&pos, false);
    }

    return bChangeMenu;
} // NOLINT(impl/function_size)

//------------------------------------------------------------------------------
// 計算処理
//------------------------------------------------------------------------------
void CalculateMenu()
{
    switch(g_Phase)
    {
    case FadeIn:
        if (g_Animator->IsEndFrame())
        {
            nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
            g_Animator = g_pLayout->CreateGroupAnimator(pDevice, "Wait");
            g_Animator->PlayAuto(1.0f);
            g_Phase = Calculate;
        }
        break;
    case Calculate:
    {
        float x = 0.0f; // タッチの X 座標
        float y = 0.0f; // タッチの Y 座標
        bool hold = false; // タッチされているか
        bool trigger = false; // タッチされた瞬間か

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
        x = static_cast<float>(GetMouseState().x);
        y = static_cast<float>(GetMouseState().y);
        if (GetMouseState().buttons.Test(nn::hid::MouseButton::Left::Index))
        {
            hold = true;
        }
#else
        nns::hid::TouchScreen* pTouchScreen = reinterpret_cast<nns::hid::TouchScreen*>(GetControllerManager().GetController(nns::hid::ControllerId_TouchScreen, 0));
        const std::vector<nns::hid::TouchScreen::TouchState>& states = pTouchScreen->GetTouchStates();
        for (size_t i = 0; i < states.size(); i++)
        {
            // 複数のタッチがあった場合は最初の 1 つだけを処理する
            const nns::hid::TouchScreen::TouchState& state = states[i];
            x = state.position.x;
            y = state.position.y;
            hold = true;
            break;
        }
#endif
        // タッチされた瞬間を判定
        if (!g_Hold && hold)
        {
            trigger = true;
        }
        g_Hold = hold;

        // タッチが返す位置は、(0,0)-(1280,720)の座標系なので、レイアウトの座標系に直す
        nn::util::Float2 pos;
        nn::font::Rectangle layoutRect = g_pLayout->GetLayoutRect();
        pos.x = layoutRect.GetWidth() / 2.0f * (2.0f * x / 1280.0f - 1.0f);
        pos.y = layoutRect.GetHeight() / 2.0f * (2.0f * y / 720.0f - 1.0f);


        if (UpdateCursor(pos) == true)
        {
            g_Phase = WaitAnimation;
        }
        break;
    }
    case WaitAnimation:
    {
        // ボタン更新処理
        nn::util::Float2 pos;
        pos.x = 0.0f;
        pos.y = 0.0f;
        g_pButtonGroup->Update(&pos, false);

        // 決定されたボタンと選択が外れたボタンがアニメーションしている場合は終了を待つ
        bool bExit = false;
        nn::ui2d::ButtonList& buttonList = g_pButtonGroup->GetButtonList();
        nn::ui2d::ButtonList::iterator endIter = buttonList.end();
        for (nn::ui2d::ButtonList::iterator iter = buttonList.begin(); iter != endIter; ++iter)
        {
            const nn::ui2d::AnimButton::State state = iter->GetState();
            if (state != nn::ui2d::AnimButton::State_OnIdle &&
                state != nn::ui2d::AnimButton::State_OffIdle &&
                state != nn::ui2d::AnimButton::State_DownIdle)
            {
                bExit = true;
                break;
            }

        }
        if (bExit == true)
        {
            break;
        }

        // フェードアウトアニメーションに変更
        {
            nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();
            g_Animator = g_pLayout->CreateGroupAnimator(pDevice, "Out");
            g_Animator->Play(nn::ui2d::Animator::PlayType::PlayType_OneShot, 1.0f);
            g_Phase = FadeOut;
        }
        break;
    }
    case FadeOut:
        if (!g_Animator->IsEndFrame())
        {
            break;
        }
        // 次のサンプルモードを更新
        if ((int)(g_SelectedSampleMode) == g_ExitNumber)
        {
            g_NextSampleMode = SampleMode_Exit;
        }
        else
        {
            g_NextSampleMode = g_SelectedSampleMode;
        }
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}
