﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

// 文字コードの型を調整するマクロ
#define FONT_SAMPLE_LITERAL(string) reinterpret_cast<const uint16_t*>(NN_CHAR16LITERAL(string))

// グラフィックスフレームワーク
extern nns::gfx::GraphicsFramework g_GfxFramework;

// サンプルの動作モードです。
enum SampleMode
{
    SampleMode_Simple,               // 最も単純なサンプル
    SampleMode_MultiArcResource,     // 複数のアーカイブリソースやリソースアクセサを扱うサンプル
    SampleMode_TextureChange,        // ペインのテクスチャを自ら読み込んだテクスチャに変更します
    SampleMode_UserShader,           // カスタマイズしたシェーダを Ui2d ライブラリに読み込ませるサンプル
    SampleMode_FontChange,           // フォントをプログラム上で差し替えるサンプル
    SampleMode_ComplexFont,          // 複合フォントを読み込むサンプル
    SampleMode_CannonButton,         // ボタンを扱うサンプル
    SampleMode_CapturePane,          // キャプチャペインを扱うサンプル
    SampleMode_AlignmentPane,        // 整列ペインを扱うサンプル
    SampleMode_Exit,                 // 終了
    SampleMode_CommandMenu,          // メニュー選択
};

// PrimitiveRenderer
extern nns::gfx::PrimitiveRenderer::Renderer*   g_pPrimitiveRenderer;

// ui2d 関係
extern nn::ui2d::Layout*                    g_pLayout;
extern nn::ui2d::ArcResourceAccessor*       g_pArcResourceAccessor;
extern void*                                g_pLayoutArchiveBinary;
extern nn::font::ResFont*                   g_pFont;
extern nn::ui2d::GraphicsResource*          g_pGraphicsResource;
extern nn::font::GpuBuffer*                 g_pUi2dConstantBuffer;
extern nn::ui2d::DrawInfo*                  g_pDrawInfo;
extern SampleMode                           g_NextSampleMode;

// アプリケーションヒープからメモリを確保する
void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData);

// アプリケーションヒープから確保したメモリを開放する
void Ui2dDeallocateFunction(void* ptr, void* pUserData);

//------------------------------------------------------------------------------
//  構築・破壊
//------------------------------------------------------------------------------
template <typename TObject>
TObject* AllocAndConstruct()
{
    return new(Ui2dAllocateFunction(sizeof(TObject), 4, NULL)) TObject();
}
template <typename TObject>
void DestructAndFree(TObject* ptr)
{
    ptr->TObject::~TObject();
    Ui2dDeallocateFunction(ptr, NULL);
}

// スロットへの登録と解除
bool RegisterSlotForTexture(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData);
bool RegisterSlotForSampler(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData);
void UnregisterSlotForTexture(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData);
void UnregisterSlotForSampler(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData);

// メモリを確保してファイルを読み込みます
void* ReadFileWithAllocate(const char* pFileName, size_t alignment);
void* ReadFileToBuffer(size_t* pFileSize, const char* pFileName, void* pBuffer, ptrdiff_t offset);

// PrimitiveRenderer の初期化処理
void InitializePrimitiveRenderer(nn::gfx::Device* pDevice);

// PrimitiveRenderer の終了処理
void FinalizePrimitiveRenderer(nn::gfx::Device* pDevice);

// ui2d 描画のためのバッファ関連初期化処理
void InitializeUi2dBuffers(nn::ui2d::DrawInfo& drawInfo, const nn::ui2d::BuildResultInformation& buildResult, nn::font::GpuBuffer* pUi2dConstantBuffer);

// GraphicsResource の初期化処理
void InitializeGraphicsResource();

// GraphicsResource の終了処理
void FinalizeGraphicsResource();

// レイアウトデータセットアップ処理
void BuildLayoutData(
    nn::ui2d::BuildResultInformation& totalRequiredBufferInfomation,
    nn::ui2d::Layout* pLayout,
    nn::ui2d::Layout::BuildOption& option,
    nn::ui2d::ResourceAccessor* pResourceAccessor,
    const char* pFileName);
