﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ui2d.h>
#include <nn/fs.h>
#include <nn/font/font_ScalableFontUtil.h>

#include "Ui2dCapturePane.h"

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------
// Lifetime_Layout 用のキャプチャテクスチャ用メモリサイズ
const size_t    StaticCaptureTextureMemorySize = 32 * 1024 * 1024;
// Lifetime_OneFrame 用のキャプチャテクスチャ用メモリサイズ
const size_t    DynamicCaptureTextureMemorySize = 32 * 1024 * 1024;

static ptrdiff_t                            g_OffsetToRenderTargetTextureMemoryPoolHead = NULL;
static ptrdiff_t                            g_OffsetToRenderTargetTextureMemoryPoolStatic = NULL;
static ptrdiff_t                            g_OffsetToRenderTargetTextureMemoryPoolDynamic = NULL;

static const char* g_pAnimPartsNames[] =
{
    "L_parts_00",
    "L_parts_01",
    "L_parts_02",
    "L_parts_03",
};

enum UserFilterShaderType
{
    UserFilterShaderType_HorizontalGaussianBlur,
    UserFilterShaderType_VerticalGaussianBlur,
    UserFilterShaderType_Max
};

// フォント関連
static nn::font::Font* g_pComplexFontTree = NULL;
static size_t g_FontBinarySize = 0;
static void* g_pFontBinary = NULL;
static void* g_pFontRawResource = NULL;
static nn::font::TextureCache* g_pTextureCache;

// ユーザーフィルターシェーダー関連
static void*                    g_pUserFilterShaderBinary = NULL;
static nn::gfx::ResShaderFile*  g_pResShaderFile = NULL;
static nn::gfx::ShaderCodeType  g_CodeType;
static int                      g_UserFilterShaderConstantBufferSlot[UserFilterShaderType_Max];
static int                      g_FrameIndex = 0;

struct TextureGpuRes
{
    nn::gfx::Texture*       pTexture;
    nn::gfx::TextureView*   pTextureView;
    nn::gfx::DescriptorSlot*    pSlot;
};

// 終了処理時に 2 フレーム分登録されることを考慮して最大サイズを決定する。
static const int DelayReleaseSlotMax = 32;
int             g_DelayReleaseCount = 0;
TextureGpuRes   g_DelayReleaseSlot[DelayReleaseSlotMax];

struct FilterTextureInfo
{
    TextureGpuRes   gpuResources;
    nn::gfx::ColorTargetView    colorTargetView;
    nn::gfx::ViewportScissorState   viewportScissorState;
};

static FilterTextureInfo        g_FilterTempPassTexture;

struct UserFilterShaderEffectParams
{
    float   weights[8];
    int     textureWidth;
    int     textureHeight;
    int     padding0[2];
};

static UserFilterShaderEffectParams g_UserFilterShaderEffectParams;

//------------------------------------------------------------------------------
nn::ui2d::RenderTargetTextureLifetime CreateRenderTargetTexture(nn::gfx::Texture** pTexture, nn::gfx::TextureView** pTextureView, nn::gfx::DescriptorSlot** pSlot, const nn::ui2d::Layout* pLayout, const nn::gfx::TextureInfo& info, void* pUserData, nn::ui2d::RenderTargetTextureLifetime lifetimeHint)
{
    NN_UNUSED(pLayout);

    NN_ASSERT_NOT_NULL(pTexture);
    NN_ASSERT_NOT_NULL(pTextureView);
    NN_ASSERT_NOT_NULL(pSlot);

    nns::gfx::GraphicsFramework* pFramework = static_cast<nns::gfx::GraphicsFramework*>(pUserData);
    nn::gfx::Device*    pDevice = pFramework->GetDevice();

    const size_t size = nn::gfx::Texture::CalculateMipDataSize(pDevice, info);
    const size_t mipDataAlignment = nn::gfx::Texture::CalculateMipDataAlignment(pDevice, info);

    // テクスチャの初期化
    // RenderTargetTextureLifetime_Layout は初回フレームだけキャプチャするテクスチャに設定されており
    // レイアウトインスタンスの初期化時と終了時にキャプチャテクスチャリソースの初期化と終了が行われます。
    // RenderTargetTextureLifetime_OneFrame はその他のキャプチャテクスチャに設定されており、毎フレーム描画直前に終了処理と初期化処理が行われます。
    // 本サンプルでは LifetimeHint 設定ごとにメモリを確保する領域を変えています。

    *pTexture = AllocAndConstruct<nn::gfx::Texture>();

    switch (lifetimeHint)
    {
    case nn::ui2d::RenderTargetTextureLifetime_Layout:
        // nn::gfx::ImageFormat_R8_Unorm のような 1 要素で中途半端なサイズのテクスチャを作成することがあるため、テクスチャ作成時のアライメントを調整する必要があります。
        g_OffsetToRenderTargetTextureMemoryPoolStatic = nn::util::align_up(g_OffsetToRenderTargetTextureMemoryPoolStatic, mipDataAlignment);
        (*pTexture)->Initialize(pDevice, info, pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), g_OffsetToRenderTargetTextureMemoryPoolStatic, size);
        g_OffsetToRenderTargetTextureMemoryPoolStatic += size;
        NN_ASSERT(g_OffsetToRenderTargetTextureMemoryPoolStatic - g_OffsetToRenderTargetTextureMemoryPoolHead < StaticCaptureTextureMemorySize,
            "Failed to allocate the static capture texture memory. The LayoutViewer can't preview over %d MB for static capture texture memory.",
            StaticCaptureTextureMemorySize / (1024 * 1024) / 2);
        break;
    case nn::ui2d::RenderTargetTextureLifetime_OneFrame:
        // nn::gfx::ImageFormat_R8_Unorm のような 1 要素で中途半端なサイズのテクスチャを作成することがあるため、テクスチャ作成時のアライメントを調整する必要があります。
        g_OffsetToRenderTargetTextureMemoryPoolDynamic = nn::util::align_up(g_OffsetToRenderTargetTextureMemoryPoolDynamic, mipDataAlignment);
        (*pTexture)->Initialize(pDevice, info, pFramework->GetMemoryPool(nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget), g_OffsetToRenderTargetTextureMemoryPoolDynamic, size);
        g_OffsetToRenderTargetTextureMemoryPoolDynamic += size;
        NN_ASSERT(g_OffsetToRenderTargetTextureMemoryPoolDynamic - g_OffsetToRenderTargetTextureMemoryPoolHead < StaticCaptureTextureMemorySize + DynamicCaptureTextureMemorySize,
            "Failed to allocate the dynamic capture texture memory. The LayoutViewer can't preview over %d MB for static capture texture memory.",
            DynamicCaptureTextureMemorySize / (1024 * 1024) / 2);
        break;
    default:
        break;
    }

    // TextureView
    {
        nn::gfx::TextureView::InfoType textureViewInfo;
        textureViewInfo.SetDefault();
        textureViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
        textureViewInfo.EditSubresourceRange().EditArrayRange().SetArrayLength(1);
        textureViewInfo.EditSubresourceRange().EditArrayRange().SetBaseArrayIndex(0);
        textureViewInfo.SetImageFormat(info.GetImageFormat());
        textureViewInfo.SetTexturePtr(*pTexture);

        *pTextureView = AllocAndConstruct<nn::gfx::TextureView>();
        (*pTextureView)->Initialize(pDevice, textureViewInfo);
    }

    *pSlot = AllocAndConstruct<nn::gfx::DescriptorSlot>();

    RegisterSlotForTexture(*pSlot, **pTextureView, pFramework);

    return lifetimeHint;
}

//------------------------------------------------------------------------------
void DestroyRenderTargetTexture(nn::gfx::Texture* pTexture, nn::gfx::TextureView* pTextureView, nn::gfx::DescriptorSlot* pSlot, const nn::ui2d::Layout* pLayout, void* pUserData, nn::ui2d::RenderTargetTextureLifetime lifetimeHint)
{
    NN_UNUSED(pLayout);
    NN_UNUSED(lifetimeHint);

    nns::gfx::GraphicsFramework* pFramework = static_cast<nns::gfx::GraphicsFramework*>(pUserData);
    nn::gfx::Device*    pDevice = pFramework->GetDevice();

    if (lifetimeHint == nn::ui2d::RenderTargetTextureLifetime_OneFrame)
    {
        // OneFrame の場合、CaptureTexture::Draw 内で毎フレームテクスチャが作り直される。
        // 作成しなおす際に 1 フレーム前に作成したテクスチャ関連リソースを破棄するが、コマンドバッファをダブルバッファリングしていると
        // GPU が描画コマンドを実行する際に必要となるリソースを開放することになる。
        // GPU が参照しなくなってから解放するため、一時的に開放待ち領域に登録しておき、GPU の描画完了後に実際に開放する。
        g_DelayReleaseSlot[g_DelayReleaseCount].pTexture = pTexture;
        g_DelayReleaseSlot[g_DelayReleaseCount].pTextureView = pTextureView;
        g_DelayReleaseSlot[g_DelayReleaseCount].pSlot = pSlot;
        ++g_DelayReleaseCount;
        NN_SDK_ASSERT(g_DelayReleaseCount <= DelayReleaseSlotMax);
    }
    else
    {
        UnregisterSlotForTexture(pSlot, *pTextureView, pUserData);
        pTextureView->Finalize(pDevice);
        pTexture->Finalize(pDevice);

        DestructAndFree(pSlot);
        DestructAndFree(pTextureView);
        DestructAndFree(pTexture);
    }
}

//------------------------------------------------------------------------------
void TextureGpuResourceDelayRelease()
{
    // 遅延開放スロットに登録された GPU リソースの開放処理を行い、遅延開放リストをクリアします。
    nn::gfx::Device*    pDevice = g_GfxFramework.GetDevice();
    for (int i = 0; i < g_DelayReleaseCount; ++i)
    {
        TextureGpuRes& info = g_DelayReleaseSlot[i];

        UnregisterSlotForTexture(info.pSlot, *info.pTextureView, &g_GfxFramework);
        info.pTextureView->Finalize(pDevice);
        info.pTexture->Finalize(pDevice);

        DestructAndFree(info.pSlot);
        DestructAndFree(info.pTextureView);
        DestructAndFree(info.pTexture);
    }
    g_DelayReleaseCount = 0;
}

//------------------------------------------------------------------------------

bool ApplyUserFilterShaderCallback(
    nn::gfx::CommandBuffer& commandBuffer,
    const nn::gfx::ColorTargetView* pColorTarget,
    const nn::gfx::DescriptorSlot& destTexture,
    const nn::gfx::DescriptorSlot& srcTexture,
    const nn::gfx::ViewportScissorState& viewportScissorState,
    nn::ui2d::Pane* pPane,
    void* pUserData)
{
    NN_UNUSED(destTexture);
    NN_UNUSED(pUserData);

    // キャプチャテクスチャのコピー処理直前に呼び出されるコールバックです。
    // 本サンプルでは特定の名前のキャプチャペインの時だけアプリケーション側で用意したピクセルシェーダーを設定しています。
    if (strcmp(pPane->GetName(), "C_UserFilterShader") == 0 ||
        strcmp(pPane->GetName(), "P_UserFilterShader_FB") == 0)
    {
        commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture);

        const nn::util::Uint8x4 white = { { 255, 255, 255, 255 } };

        g_pPrimitiveRenderer->SetColor(white);
        nn::util::Matrix4x3fType modelMatrix;
        nn::util::MatrixIdentity(&modelMatrix);
        g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Vector3fType camPos = { 0.0f, 0.0f, -1.0f };
        nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);
        g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

        nn::util::Matrix4x4fType projMat;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projMat, 1.0f, -1.0f, 1.0f, -1.0f, 0.0f, 300.0f);
        g_pPrimitiveRenderer->SetProjectionMatrix(&projMat);

        // コンスタントバッファのセットアップ
        nn::gfx::GpuAddress gpuAddr;
        g_pPrimitiveRenderer->SetUserConstantBuffer(&gpuAddr, &g_UserFilterShaderEffectParams, sizeof(UserFilterShaderEffectParams));

        // 横ブラー
        const nn::gfx::ColorTargetView* pTempPassColorTargetView = &g_FilterTempPassTexture.colorTargetView;
        commandBuffer.SetRenderTargets(1, &pTempPassColorTargetView, nullptr);
        g_pPrimitiveRenderer->SetBlendState(&commandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity);
        g_pPrimitiveRenderer->SetDepthStencilState(&commandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthNoWriteTest);
        commandBuffer.SetViewportScissorState(&g_FilterTempPassTexture.viewportScissorState);

        nn::gfx::Shader* pShader = g_pResShaderFile->GetShaderContainer()->GetResShaderVariation(UserFilterShaderType_HorizontalGaussianBlur)->GetResShaderProgram(g_CodeType)->GetShader();
        commandBuffer.SetConstantBuffer(g_UserFilterShaderConstantBufferSlot[UserFilterShaderType_HorizontalGaussianBlur], nn::gfx::ShaderStage_Pixel, gpuAddr, sizeof(UserFilterShaderEffectParams));
        g_pPrimitiveRenderer->SetUserPixelShader(pShader);

        g_pPrimitiveRenderer->DrawScreenQuad(
            &commandBuffer,
            srcTexture,
            g_pGraphicsResource->GetSamplerDescriptorSlot(nn::ui2d::TexWrap_Clamp, nn::ui2d::TexWrap_Clamp, nn::ui2d::TexFilter_Linear, nn::ui2d::TexFilter_Linear));

        commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture);

        // 縦ブラー
        commandBuffer.SetRenderTargets(1, &pColorTarget, nullptr);
        g_pPrimitiveRenderer->SetBlendState(&commandBuffer, nns::gfx::PrimitiveRenderer::BlendType_Opacity);
        g_pPrimitiveRenderer->SetDepthStencilState(&commandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType_DepthNoWriteTest);
        commandBuffer.SetViewportScissorState(&viewportScissorState);

        pShader = g_pResShaderFile->GetShaderContainer()->GetResShaderVariation(UserFilterShaderType_VerticalGaussianBlur)->GetResShaderProgram(g_CodeType)->GetShader();
        commandBuffer.SetConstantBuffer(g_UserFilterShaderConstantBufferSlot[UserFilterShaderType_VerticalGaussianBlur], nn::gfx::ShaderStage_Pixel, gpuAddr, sizeof(UserFilterShaderEffectParams));
        g_pPrimitiveRenderer->SetUserPixelShader(pShader);

        g_pPrimitiveRenderer->DrawScreenQuad(
            &commandBuffer,
            *g_FilterTempPassTexture.gpuResources.pSlot,
            g_pGraphicsResource->GetSamplerDescriptorSlot(nn::ui2d::TexWrap_Clamp, nn::ui2d::TexWrap_Clamp, nn::ui2d::TexFilter_Linear, nn::ui2d::TexFilter_Linear));

        commandBuffer.FlushMemory(nn::gfx::GpuAccess_ColorBuffer | nn::gfx::GpuAccess_Texture);

        return true;
    }

    // 戻り値として false を返すとキャプチャされたテクスチャがそのまま使用されます。。
    return false;
}

//------------------------------------------------------------------------------
// キャプチャテクスチャに適用するユーザーフィルターシェーダーを初期化する
//------------------------------------------------------------------------------
void InitializeUserFilterShader(nn::gfx::Device* pDevice)
{
    g_pUserFilterShaderBinary = ReadFileWithAllocate("Contents:/UserFilterShader.bnsh", 4096);

    // シェーダの読み込み
    {
        g_pResShaderFile = nn::gfx::ResShaderFile::ResCast(g_pUserFilterShaderBinary);

        nn::gfx::ResShaderContainer* pContainer = g_pResShaderFile->GetShaderContainer();
        NN_SDK_ASSERT_NOT_NULL(pContainer);
        pContainer->Initialize(pDevice);

        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation(0);
        NN_SDK_ASSERT_NOT_NULL(pVariation);

        g_CodeType = nn::ui2d::TryInitializeAndGetShaderCodeType(pDevice, pVariation);

        int variationCount = g_pResShaderFile->GetShaderContainer()->GetShaderVariationCount();
        NN_ASSERT(variationCount == UserFilterShaderType_Max);

        for (int i = 1; i < variationCount; ++i)
        {
            pVariation = pContainer->GetResShaderVariation(i);
            NN_SDK_ASSERT_NOT_NULL(pVariation);

            nn::gfx::ShaderInitializeResult shaderResult =  pVariation->GetResShaderProgram(g_CodeType)->Initialize(pDevice);
            NN_SDK_ASSERT(shaderResult == nn::gfx::ShaderInitializeResult_Success);
            NN_UNUSED(shaderResult);
        }

        for (int i = 0; i < variationCount; ++i)
        {
            nn::gfx::Shader* pPixelShader = g_pResShaderFile->GetShaderContainer()->GetResShaderVariation(i)->GetResShaderProgram(g_CodeType)->GetShader();
            g_UserFilterShaderConstantBufferSlot[i] = pPixelShader->GetInterfaceSlot(nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "UserEffectParams");
        }
    }

    // キャプチャテクスチャのサイズを計算するために一度グローバルマトリクスを更新します。
    g_pLayout->CalculateGlobalMatrix(*g_pDrawInfo);

    // キャプチャペインのポインタからキャプチャテクスチャを取得する。
    const nn::ui2d::Pane* pCaptureTarget = g_pLayout->GetRootPane()->FindPaneByName("C_UserFilterShader");
    const nn::ui2d::CaptureTexture* pCaptureTexture = g_pLayout->FindCaptureTextureByPanePtr(pCaptureTarget);
    float   x, y, width, height;

    // キャプチャ範囲を取得する。
    pCaptureTexture->GetCaptureAreaInfo(&x, &y, &width, &height);
    float textureScale = pCaptureTexture->GetTextureScale();

    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetWidth(static_cast<int>(width * textureScale));
    textureInfo.SetHeight(static_cast<int>(height * textureScale));
    textureInfo.SetImageFormat(pCaptureTexture->GetTextureFormat());
    textureInfo.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
    textureInfo.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
    textureInfo.SetMipCount(1);

    CreateRenderTargetTexture(
        &(g_FilterTempPassTexture.gpuResources.pTexture),
        &(g_FilterTempPassTexture.gpuResources.pTextureView),
        &(g_FilterTempPassTexture.gpuResources.pSlot),
        NULL,
        textureInfo,
        &g_GfxFramework,
        nn::ui2d::RenderTargetTextureLifetime_Layout);

    // ColorTargetView
    nn::gfx::ColorTargetView::InfoType targetViewInfo;
    targetViewInfo.SetDefault();
    targetViewInfo.SetImageDimension(nn::gfx::ImageDimension_2d);
    targetViewInfo.EditArrayRange().SetArrayLength(1);
    targetViewInfo.EditArrayRange().SetBaseArrayIndex(0);
    targetViewInfo.SetImageFormat(textureInfo.GetImageFormat());
    targetViewInfo.SetTexturePtr(g_FilterTempPassTexture.gpuResources.pTexture);
    g_FilterTempPassTexture.colorTargetView.Initialize(pDevice, targetViewInfo);

    // ViewportScissorState
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetOriginX(static_cast<float>(0));
        viewportInfo.SetOriginY(static_cast<float>(0));
        viewportInfo.SetWidth(static_cast<float>(textureInfo.GetWidth()));
        viewportInfo.SetHeight(static_cast<float>(textureInfo.GetHeight()));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetOriginX(0);
        scissorInfo.SetOriginY(0);
        scissorInfo.SetWidth(textureInfo.GetWidth());
        scissorInfo.SetHeight(textureInfo.GetHeight());
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);

    g_FilterTempPassTexture.viewportScissorState.Initialize(pDevice, info);

    float sum = 0.0f;

    for (int i = 0; i < 6; ++i)
    {
        float r = static_cast<float>(i) * 2.0f;
        if (i == 0) r = 3.0f;
        g_UserFilterShaderEffectParams.weights[i] = expf(-(r * r) / (2.0f * 5.0f * 5.0f));

        if (i != 0)
        {
            sum += g_UserFilterShaderEffectParams.weights[i] * 2.0f;
        }
        else
        {
            sum += g_UserFilterShaderEffectParams.weights[i];
        }
    }

    for (int i = 0; i < 6; ++i)
    {
        g_UserFilterShaderEffectParams.weights[i] /= sum;
    }

    g_UserFilterShaderEffectParams.textureWidth = textureInfo.GetWidth();
    g_UserFilterShaderEffectParams.textureHeight = textureInfo.GetHeight();
}

//------------------------------------------------------------------------------
// フォント名からフォントデータを探すコールバック関数
//------------------------------------------------------------------------------
static void* AcquireFontFunctionForComplexFont(size_t* pOutFontDataSize, const char* pFontName, nn::ui2d::ResType resType, void* pUserData)
{
    NN_UNUSED(pFontName);
    NN_UNUSED(resType);
    NN_UNUSED(pUserData);

    // 本サンプルでは一種類のフォント固定のデータを使用するため必ず同じデータを返します。
    if (g_pFontBinary != NULL)
    {
        *pOutFontDataSize = g_FontBinarySize;
        return g_pFontBinary;
    }
    else
    {
        nn::Result result;
        nn::fs::FileHandle fileHandle;
        int64_t fileSize;

        char path[256];
        strcpy(path, "Contents:/IconFont-Regular.otf");
        result = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        result = nn::fs::GetFileSize(&fileSize, fileHandle);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        void* pFontFileImage = Ui2dAllocateFunction(static_cast<size_t>(fileSize), nn::font::ResourceAlignment, NULL);
        g_pFontRawResource = pFontFileImage;

        size_t readSize;
        result = nn::fs::ReadFile(&readSize, fileHandle, 0, pFontFileImage, static_cast<size_t>(fileSize));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        g_FontBinarySize = static_cast<size_t>(fileSize);
        g_pFontBinary = pFontFileImage;

        nn::fs::CloseFile(fileHandle);

        NN_ASSERT_NOT_NULL(g_pFontBinary);
        *pOutFontDataSize = g_FontBinarySize;
        return g_pFontBinary;
    }
}

//------------------------------------------------------------------------------
// サンプルで使用するフォントの読み込みと初期化
//------------------------------------------------------------------------------
static void InitializeSampleFonts()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // ビットマップフォントの読み込み
    {
        void* pFontResource = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool result = g_pFont->SetResource(pDevice, pFontResource);
        NN_ASSERT(result);

        // フォントのテクスチャ用ディスクリプタスロットを登録
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
        // フォントをリソースアクセサーに登録
        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);

    }

    // 複合フォントの読み込み
    {
        const void* pFcpxData = NULL;
        int fontFaceHead;

        pFcpxData = g_pArcResourceAccessor->FindResourceByName(nn::ui2d::ResourceTypeComplexFont, "IconFont-Regular.bfcpx");

        nn::font::TextureCache::InitializeArg* pArg = static_cast<nn::font::TextureCache::InitializeArg*>(nn::ui2d::Layout::AllocateMemory(sizeof(nn::font::TextureCache::InitializeArg)));
        pArg->SetDefault();
        pArg->workMemorySize = 1024 * 1024;
        pArg->noPlotWorkMemorySize = 1024 * 1024;
        pArg->allocateFunction = nn::ui2d::Layout::GetAllocateFunction();
        pArg->textureCacheWidth = 1024;
        pArg->textureCacheHeight = 1024;

        // ユーティリティ関数を使って bfcpx から TextureCacheArg を構築する
        fontFaceHead = nn::ui2d::ComplexFontHelper::SetupTextureCacheArg(
            pArg,
            AcquireFontFunctionForComplexFont,
            g_pArcResourceAccessor,
            pFcpxData);

        // TextureCache を初期化する
        g_pTextureCache = AllocAndConstruct<nn::font::TextureCache>();
        g_pTextureCache->Initialize(pDevice, *pArg);
        g_pTextureCache->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);

        nn::ui2d::Layout::FreeMemory(pArg);

        // ユーティリティ関数を使って bfcpx からフォントを構築する
        g_pComplexFontTree = nn::ui2d::ComplexFontHelper::InitializeComplexFontTree(
            pDevice,
            RegisterSlotForTexture,
            &g_GfxFramework,
            g_pTextureCache,
            fontFaceHead,
            AcquireFontFunctionForComplexFont,
            g_pArcResourceAccessor,
            pFcpxData);

        // アーカイバに .fcpx として登録します。
        g_pArcResourceAccessor->RegisterFont("IconFont-Regular.fcpx", g_pComplexFontTree);
    }
}

//------------------------------------------------------------------------------
// サンプルで使用したフォントの終了処理
//------------------------------------------------------------------------------
static void FinalizeSampleFonts()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // 複合フォント
    {
        nn::ui2d::ComplexFontHelper::FinalizeComplexFontTree(
            pDevice,
            g_pComplexFontTree,
            UnregisterSlotForTexture,
            &g_GfxFramework);

        g_pComplexFontTree = NULL;

        g_pTextureCache->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pTextureCache->Finalize(pDevice, nn::ui2d::Layout::GetFreeFunction(), NULL);
        DestructAndFree<nn::font::TextureCache>(g_pTextureCache);
        g_pTextureCache = NULL;

        Ui2dDeallocateFunction(g_pFontRawResource, NULL);

        g_FontBinarySize = 0;
        g_pFontBinary = NULL;
        g_pFontRawResource = NULL;
    }

    // ビットマップフォント
    {
        g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        void* pFontResource = g_pFont->RemoveResource(pDevice);
        Ui2dDeallocateFunction(pFontResource, NULL);
        g_pFont->Finalize(pDevice);
        DestructAndFree<nn::font::ResFont>(g_pFont);
        g_pFont = NULL;
    }
}


//------------------------------------------------------------------------------
// 初期化
//------------------------------------------------------------------------------
void InitializeCapturePane()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // キャプチャテクスチャを使用するため、レイアウトからレンダーターゲットテクスチャリソースを作成・破棄するためのコールバック関数を登録
    nn::ui2d::Layout::SetRenderTargetTextureResourceManagementCallback(CreateRenderTargetTexture, DestroyRenderTargetTexture, &g_GfxFramework);

    // ユーザーフィルターで使用するために PrimitiveRenderer を初期化
    InitializePrimitiveRenderer(pDevice);

    // RenderTarget 用のメモリ領域をフレームワークから確保する
    {
        nn::gfx::MemoryPool::InfoType memorypoolInfo;

        memorypoolInfo.SetDefault();
        memorypoolInfo.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

        size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(pDevice, memorypoolInfo);

        // 16MB レンダーターゲットテクスチャ用にメモリを確保し、前半を初回だけキャプチャするテクスチャに、後半を動的にキャプチャするテクスチャように割り振る
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
        g_OffsetToRenderTargetTextureMemoryPoolHead = g_GfxFramework.AllocatePoolMemory(memoryPoolType, StaticCaptureTextureMemorySize + DynamicCaptureTextureMemorySize, alignment);
        g_OffsetToRenderTargetTextureMemoryPoolStatic = g_OffsetToRenderTargetTextureMemoryPoolHead;
        g_OffsetToRenderTargetTextureMemoryPoolDynamic = g_OffsetToRenderTargetTextureMemoryPoolHead + StaticCaptureTextureMemorySize;
    }

    // リソースアクセサ・フォントの初期化
    {
        g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();
        g_pFont = AllocAndConstruct<nn::font::ResFont>();

        // レイアウトアーカイブの読み込み
        {
            g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dCapturePane.arc", nn::ui2d::ArchiveResourceAlignment);
            {
                bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
                NN_ASSERT(result);
            }
        }

        // フォントの読み込み・初期化
        InitializeSampleFonts();
    }

    // レイアウトの初期化
    nn::ui2d::BuildResultInformation buildResult;
    {
        g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();

        nn::ui2d::Layout::BuildOption    opt;
        opt.SetDefault();
        buildResult.SetDefault();

        g_pLayout->BuildWithName(&buildResult, pDevice, g_pArcResourceAccessor, NULL, NULL, opt, "CapturePane.bflyt");
    }

    // グラフィックスリソースの設定
    InitializeGraphicsResource();

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    g_pDrawInfo = AllocAndConstruct<nn::ui2d::DrawInfo>();
    NN_ASSERT_NOT_NULL(g_pDrawInfo);
    g_pUi2dConstantBuffer = AllocAndConstruct<nn::font::GpuBuffer>();
    NN_ASSERT_NOT_NULL(g_pUi2dConstantBuffer);
    InitializeUi2dBuffers(*g_pDrawInfo, buildResult, g_pUi2dConstantBuffer);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(&projection, rect.left, rect.right, rect.bottom, rect.top, 0.0f, 300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(&view, pos, target, up);

        g_pDrawInfo->SetGraphicsResource(g_pGraphicsResource);
        g_pDrawInfo->SetProjectionMtx(projection);
        g_pDrawInfo->SetViewMtx(view);
    }

    // アニメーションの初期化
    {
        nn::ui2d::Animator* pAnimator = g_pLayout->CreateGroupAnimator(pDevice, "loop");
        pAnimator->PlayAuto(1.0f);

        int partsCount = sizeof(g_pAnimPartsNames) / sizeof(char*);
        for (int i = 0; i < partsCount; ++i)
        {
            nn::ui2d::Parts* pParts = g_pLayout->FindPartsPaneByName(g_pAnimPartsNames[i]);
            pAnimator = pParts->GetLayout()->CreateGroupAnimator(pDevice, "Tag_00");
            pAnimator->PlayAuto(1.0f);
        }

        // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
        g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture, &g_GfxFramework);
    }

    // 複合フォント用のテクスチャキャッシュの状態をを更新する
    nn::font::RegisterGlyphFromTextBoxRecursive(g_pLayout->GetRootPane());
    g_pTextureCache->UpdateTextureCache();
    g_pTextureCache->CompleteTextureCache();

    // キャプチャペインでキャプチャされるテクスチャを設定する
    // フレームバッファキャプチャが有効になっているキャプチャペインは、子ペインの描画前にこのメソッドで設定されたテクスチャから特定範囲をキャプチャします。
    g_pDrawInfo->SetFramebufferTexture(g_GfxFramework.GetColorBuffer(), g_GfxFramework.GetDisplayWidth(), g_GfxFramework.GetDisplayHeight());

    // キャプチャテクスチャに適用するユーザーフィルターシェーダーを初期化する
    InitializeUserFilterShader(pDevice);

    // キャプチャテクスチャのユーザーフィルターコールバックを設定します。
    // このコールバックでキャプチャのテクスチャに対してユーザーが任意のシェーダーを適用することができます。
    g_pDrawInfo->SetApplyCaptureTextureFilterCallback(ApplyUserFilterShaderCallback, NULL);
}



//------------------------------------------------------------------------------
// 解放
//------------------------------------------------------------------------------
void FinalizeCapturePane()
{
    nn::gfx::Device* pDevice = g_GfxFramework.GetDevice();

    g_pUi2dConstantBuffer->Finalize(pDevice, Ui2dDeallocateFunction, NULL);
    DestructAndFree<nn::font::GpuBuffer>(g_pUi2dConstantBuffer);
    g_pUi2dConstantBuffer = NULL;
    DestructAndFree<nn::ui2d::DrawInfo>(g_pDrawInfo);
    g_pDrawInfo = NULL;

    FinalizeGraphicsResource();

    g_pLayout->Finalize(pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);
    g_pLayout = NULL;

    TextureGpuResourceDelayRelease();

    {
        g_FilterTempPassTexture.viewportScissorState.Finalize(pDevice);
        g_FilterTempPassTexture.colorTargetView.Finalize(pDevice);
        DestroyRenderTargetTexture(
            g_FilterTempPassTexture.gpuResources.pTexture,
            g_FilterTempPassTexture.gpuResources.pTextureView,
            g_FilterTempPassTexture.gpuResources.pSlot,
            NULL,
            &g_GfxFramework,
            nn::ui2d::RenderTargetTextureLifetime_Layout);

        g_pResShaderFile->GetShaderContainer()->GetResShaderVariation(0)->GetResShaderProgram(g_CodeType)->Finalize(pDevice);
        g_pResShaderFile->GetShaderContainer()->Finalize(pDevice);
        Ui2dDeallocateFunction(g_pUserFilterShaderBinary, NULL);
    }

    // サンプルで使用したフォントの終了処理
    FinalizeSampleFonts();

    {
        g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture, &g_GfxFramework);
        g_pArcResourceAccessor->Detach();
        g_pArcResourceAccessor->Finalize(pDevice);

        // ArcResourceAccessor の Finalize でアクセスされるため Finalize 後に開放する。
        Ui2dDeallocateFunction(g_pLayoutArchiveBinary, NULL);
        g_pLayoutArchiveBinary = NULL;

        DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);
        g_pArcResourceAccessor = NULL;
    }

    // RenderTarget 用の領域をフレームワークに返却する
    if (g_OffsetToRenderTargetTextureMemoryPoolHead != NULL)
    {
        const nns::gfx::GraphicsFramework::MemoryPoolType memoryPoolType = nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget;
        g_GfxFramework.FreePoolMemory(memoryPoolType, g_OffsetToRenderTargetTextureMemoryPoolHead);
        g_OffsetToRenderTargetTextureMemoryPoolHead = NULL;
        g_OffsetToRenderTargetTextureMemoryPoolStatic = NULL;
        g_OffsetToRenderTargetTextureMemoryPoolDynamic = NULL;
    }

    FinalizePrimitiveRenderer(pDevice);
}

void CalculateCapturePane()
{
    // レンダーターゲットのテクスチャリソース遅延開放処理。
    TextureGpuResourceDelayRelease();

    // 動的なテクスチャ用の領域を毎フレームリセットする
    g_OffsetToRenderTargetTextureMemoryPoolDynamic = g_OffsetToRenderTargetTextureMemoryPoolHead + StaticCaptureTextureMemorySize;

    // PrimitiveRenderer の更新処理
    g_pPrimitiveRenderer->Update(g_FrameIndex % 2);
    ++g_FrameIndex;
}

// キャプチャテクスチャ更新前描画処理
void DrawCapturePaneSampleBeforeCaptureTexture(nn::gfx::CommandBuffer& commandBuffer)
{
    // ビューとプロジェクションを設定
    float radius = 20.f;
    float x = radius * sin(g_FrameIndex / 500.f);
    float z = radius * cos(g_FrameIndex / 500.f);

    g_pPrimitiveRenderer->SetDefaultParameters();

    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;
    nn::util::MatrixIdentity(&modelMatrix);
    nn::util::Vector3f translate;
    nn::util::VectorSet(&translate, 0.f, 0.f, 1.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);

    nn::util::Uint8x4 white = { { 255, 255, 255, 255 } };
    nn::util::Uint8x4 red = { { 255, 0, 0, 255 } };
    nn::util::Uint8x4 green = { { 0, 255, 0, 255 } };
    nn::util::Uint8x4 blue = { { 0, 0, 255, 255 } };

    nn::util::Vector3fType camPos = { x, 10.f, z };
    nn::util::Vector3fType camTarget = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType camUp = { 0.f, 1.f, 0.f };
    nn::util::MatrixLookAtRightHanded(&viewMatrix, camPos, camTarget, camUp);
    g_pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

    // プロジェクションを初期化
    const float fovy = nn::util::FloatPi / 3.0f;
    const float aspect = static_cast< float >(1280) / static_cast< float >(720);
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, fovy, aspect, 0.1f, 1000.f);
    g_pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

    // Depth Enable
    g_pPrimitiveRenderer->SetDepthStencilState(&commandBuffer, nns::gfx::PrimitiveRenderer::DepthStencilType::DepthStencilType_DepthWriteTest);

    // 軸を描画
    float interval = -10.f;
    nn::util::Vector3fType begin;
    nn::util::Vector3fType end;
    g_pPrimitiveRenderer->SetLineWidth(1.f);
    for (int i = 0; i < 21; i++)
    {

        nn::util::VectorSet(&begin, -10.f, 0.f, interval);
        nn::util::VectorSet(&end, 10.f, 0.f, interval);
        g_pPrimitiveRenderer->SetColor(white);
        g_pPrimitiveRenderer->DrawLine(&commandBuffer, begin, end);
        nn::util::VectorSet(&begin, interval, 0.f, -10.f);
        nn::util::VectorSet(&end, interval, 0.f, 10.f);
        g_pPrimitiveRenderer->DrawLine(&commandBuffer, begin, end);
        interval += 1.0f;
    }

    g_pPrimitiveRenderer->SetLineWidth(5.f);
    nn::util::Vector3fType zeroVector;
    nn::util::VectorSet(&zeroVector, 0.f, 0.f, 0.f);
    nn::util::Vector3fType axisPos;
    nn::util::VectorSet(&axisPos, 5.f, 0.f, 0.f);
    g_pPrimitiveRenderer->SetColor(red);
    g_pPrimitiveRenderer->DrawLine(&commandBuffer, zeroVector, axisPos);
    nn::util::VectorSet(&axisPos, 0.f, 0.f, 5.f);
    g_pPrimitiveRenderer->SetColor(blue);
    g_pPrimitiveRenderer->DrawLine(&commandBuffer, zeroVector, axisPos);
    nn::util::VectorSet(&axisPos, 0.f, 5.f, 0.f);
    g_pPrimitiveRenderer->SetColor(green);
    g_pPrimitiveRenderer->DrawLine(&commandBuffer, zeroVector, axisPos);

    nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
    nn::util::Vector3fType size = { 4.f, 4.f, 4.f };

    nn::util::VectorSet(&translate, 0.f, 0.f, 0.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    g_pPrimitiveRenderer->SetLineWidth(1.f);
    g_pPrimitiveRenderer->SetColor(white);
    g_pPrimitiveRenderer->DrawCube(&commandBuffer,
        nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
        center, size);

    nn::util::VectorSet(&translate, 8.f, 0.f, 0.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    g_pPrimitiveRenderer->SetColor(white);
    g_pPrimitiveRenderer->DrawCube(&commandBuffer,
        nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
        center, size);

    nn::util::VectorSet(&translate, -8.f, 0.f, 0.f);
    nn::util::MatrixSetAxisW(&modelMatrix, translate);
    g_pPrimitiveRenderer->SetModelMatrix(&modelMatrix);
    g_pPrimitiveRenderer->SetColor(white);
    g_pPrimitiveRenderer->DrawCube(&commandBuffer,
        nns::gfx::PrimitiveRenderer::Surface::Surface_Normal,
        center, size);
}

