﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/vi.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include "InputController.h"

static Pad                         g_Pad;
static nn::hid::MouseState         g_MouseState;
static nns::hid::ControllerManager g_ControllerManager;
nn::vi::NativeWindowHandle         g_WndHandle = NULL;
static int                         g_DisplayWidth;
static int                         g_DisplayHeight;

//--------------------------------------------------------------------------------------------------
//
void InitializeInputController(int width, int height, void* hwnd)
{
    g_DisplayWidth = width;
    g_DisplayHeight = height;
    g_WndHandle = reinterpret_cast<nn::vi::NativeWindowHandle>(hwnd);

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // DebugPad を初期化します。
    nn::hid::InitializeDebugPad();
#else
    nns::hid::util::SetControllerManagerWithDefault(&GetControllerManager());
#endif

    // パッド初期化
    GetPad().Initialize();

    // ジョイコンを使用します
    GetPad().InitializeNpad();

    GetPad().Reset();
}

void UpdateInputController()
{
    GetPad().Read();
}


//--------------------------------------------------------------------------------------------------
//
nn::hid::MouseState& GetMouseState() NN_NOEXCEPT
{
    return g_MouseState;
}


nns::hid::ControllerManager& GetControllerManager() NN_NOEXCEPT
{
    return g_ControllerManager;
}

//--------------------------------------------------------------------------------------------------
// Pad

Pad& GetPad() NN_NOEXCEPT
{
    return g_Pad;
}

void Pad::Initialize()
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // マウスを使用する
    nn::hid::InitializeMouse();

    // デバッグパッドにキーマッピングを設定する
    nn::settings::GetDebugPadKeyboardMap(&m_DebugPadKeyboardMap);
    m_DebugPadKeyboardMap.buttonA = nn::hid::KeyboardKey::A::Index;
    m_DebugPadKeyboardMap.buttonB = nn::hid::KeyboardKey::B::Index;
    m_DebugPadKeyboardMap.buttonX = nn::hid::KeyboardKey::X::Index;
    m_DebugPadKeyboardMap.buttonY = nn::hid::KeyboardKey::Y::Index;
    m_DebugPadKeyboardMap.buttonL = nn::hid::KeyboardKey::L::Index;
    m_DebugPadKeyboardMap.buttonR = nn::hid::KeyboardKey::R::Index;
    m_DebugPadKeyboardMap.buttonZL = nn::hid::KeyboardKey::Z::Index;
    m_DebugPadKeyboardMap.buttonZR = nn::hid::KeyboardKey::Z::Index;
    m_DebugPadKeyboardMap.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
    m_DebugPadKeyboardMap.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
    m_DebugPadKeyboardMap.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
    m_DebugPadKeyboardMap.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
    m_DebugPadKeyboardMap.buttonStart = nn::hid::KeyboardKey::Space::Index;
    m_DebugPadKeyboardMap.buttonSelect = nn::hid::KeyboardKey::Return::Index;
    nn::settings::SetDebugPadKeyboardMap(m_DebugPadKeyboardMap);
#endif
}

void Pad::InitializeNpad()
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    nn::hid::InitializeNpad();

    // 使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleJoyDual::Mask | nn::hid::NpadStyleHandheld::Mask);

    // 使用する Npad を設定
    m_NpadIds[0] = nn::hid::NpadId::No1;
    m_NpadIds[1] = nn::hid::NpadId::Handheld;

    nn::hid::SetSupportedNpadIdType(m_NpadIds, NPADID_NUM);

    m_InitNpad = true;
#endif
}

bool Pad::Reset()
{
    if (!Read() || !Read()) // m_Last* を初期化するため2回空読みする。
    {
        NN_LOG("Pad::Read Failed.\n");
        return false;
    }
    m_AnalogStick.leftX = m_AnalogStick.leftY = m_AnalogStick.rightX = m_AnalogStick.rightY = 0.0f;
    m_Button = 0;

    return true;
}

bool Pad::Read()
{
    nn::Bit32 button = 0;
    Stick stick = { 0, 0, 0, 0 };
    bool IsAnalogStickLData = false;
    bool IsAnalogStickRData = false;
    bool IsEnableDebugPad = true;

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // マウス操作
    {
        nn::hid::MouseState& state = g_MouseState;
        nn::hid::GetMouseState(&state);

        if (0 <= state.x && state.x <= g_DisplayWidth && 0 <= state.y && state.y <= g_DisplayHeight)
        {
            static const int8_t RADIUS = 127;
            float invRadius = 1.0f / std::min(g_DisplayWidth, g_DisplayHeight);
            float rx = invRadius * ((state.x << 1) - g_DisplayWidth);
            float ry = invRadius * (g_DisplayHeight - (state.y << 1));
            float lengthSq = rx * rx + ry * ry;
            if (lengthSq > 1.0f)
            {
                float rcpLen = 1.0f / std::sqrt(lengthSq);
                rx *= rcpLen;
                ry *= rcpLen;
            }
            if (state.buttons.Test<nn::hid::MouseButton::Left>())
            {
                stick.leftX += static_cast<int8_t>(rx * RADIUS);
                stick.leftY += static_cast<int8_t>(ry * RADIUS);
                IsAnalogStickLData = true;
            }
            if (state.buttons.Test<nn::hid::MouseButton::Right>())
            {
                stick.rightX += static_cast<int8_t>(rx * RADIUS);
                stick.rightY += static_cast<int8_t>(ry * RADIUS);
                IsAnalogStickRData = true;
            }
        }
    }
#endif

    g_ControllerManager.Update();

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
    // ウインドウフォーカスが無い場合はデバッグパッドの入力を無効にする
    HWND hWnd = GetForegroundWindow();
    if (!(hWnd && hWnd == g_WndHandle))
    {
        IsEnableDebugPad = false;
    }
#endif

    if (IsEnableDebugPad)
    {
        nn::hid::DebugPadState state;
        nn::hid::GetDebugPadState(&state);
        button |= state.buttons.Test<nn::hid::DebugPadButton::Start>() ? BUTTON_START : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Left>() ? BUTTON_LEFT : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Up>() ? BUTTON_UP : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Right>() ? BUTTON_RIGHT : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Down>() ? BUTTON_DOWN : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::A>() ? BUTTON_A : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::B>() ? BUTTON_B : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::X>() ? BUTTON_X : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Y>() ? BUTTON_Y : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::R>() ? TRIGGER_R : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::L>() ? TRIGGER_L : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::ZL>() ? TRIGGER_Z : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::ZR>() ? TRIGGER_Z : 0;
        button |= state.buttons.Test<nn::hid::DebugPadButton::Select>() ? BUTTON_A : 0;

        if (!IsAnalogStickLData)
        {
            stick.leftX = static_cast<int8_t>(state.analogStickL.x >> 8);
            stick.leftY = static_cast<int8_t>(state.analogStickL.y >> 8);
        }
        if (!IsAnalogStickRData)
        {
            stick.rightX = static_cast<int8_t>(state.analogStickR.x >> 8);
            stick.rightY = static_cast<int8_t>(state.analogStickR.y >> 8);
        }
    }

    ReadNpad(&button, &stick);

    static const int STICK_THRESHOLD = 32;
    button |= (m_Stick.leftX < -STICK_THRESHOLD) ? STICK_L_LEFT : 0;
    button |= (m_Stick.leftX > STICK_THRESHOLD) ? STICK_L_RIGHT : 0;
    button |= (m_Stick.leftY < -STICK_THRESHOLD) ? STICK_L_DOWN : 0;
    button |= (m_Stick.leftY > STICK_THRESHOLD) ? STICK_L_UP : 0;
    button |= (m_Stick.rightX < -STICK_THRESHOLD) ? STICK_R_LEFT : 0;
    button |= (m_Stick.rightX > STICK_THRESHOLD) ? STICK_R_RIGHT : 0;
    button |= (m_Stick.rightY < -STICK_THRESHOLD) ? STICK_R_DOWN : 0;
    button |= (m_Stick.rightY > STICK_THRESHOLD) ? STICK_R_UP : 0;

    m_LastButton = m_Button;
    m_Button = button;
    m_LastStick = m_Stick;
    m_Stick = stick;

    m_AnalogStick.leftX +=
        static_cast<float>(m_Stick.leftX - m_LastStick.leftX) / 56.0f;
    m_AnalogStick.leftY +=
        static_cast<float>(m_Stick.leftY - m_LastStick.leftY) / 56.0f;
    m_AnalogStick.rightX +=
        static_cast<float>(m_Stick.rightX - m_LastStick.rightX) / 56.0f;
    m_AnalogStick.rightY +=
        static_cast<float>(m_Stick.rightY - m_LastStick.rightY) / 56.0f;

    nn::Bit32 changed = m_Button ^ m_LastButton;
    m_Triggered = changed & m_Button;
    m_Released = changed & m_LastButton;

    return true;
}

void Pad::ReadNpad(nn::Bit32* pButton, Stick* pStick)
{
#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    if (!m_InitNpad)
    {
        return;
    }

    nn::hid::NpadButtonSet npadState[NPADID_NUM];
    nn::hid::AnalogStickState npadAnalogStickL;
    nn::hid::AnalogStickState npadAnalogStickR;
    nn::Bit32 result = GetNpadState(npadState, &npadAnalogStickL, &npadAnalogStickR);
    if (result & NpadState_Buttons)
    {
        for (int idxNPad = 0; idxNPad < NPADID_NUM; ++idxNPad)
        {
            nn::hid::NpadButtonSet* pNpadState = &npadState[idxNPad];
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::A>() ? BUTTON_A : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::B>() ? BUTTON_B : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::X>() ? BUTTON_X : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Y>() ? BUTTON_Y : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Plus>() ? BUTTON_START : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Left>() ? BUTTON_LEFT : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Up>() ? BUTTON_UP : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Right>() ? BUTTON_RIGHT : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::Down>() ? BUTTON_DOWN : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::R>() ? TRIGGER_R : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::L>() ? TRIGGER_L : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::ZR>() ? TRIGGER_Z : 0;
            *pButton |= pNpadState->Test<nn::hid::NpadJoyButton::ZL>() ? TRIGGER_Z : 0;
        }
    }

    if (result & NpadState_AnalogStickL)
    {
        pStick->leftX = static_cast<int8_t>(npadAnalogStickL.x >> 8);
        pStick->leftY = static_cast<int8_t>(npadAnalogStickL.y >> 8);
    }
    if (result & NpadState_AnalogStickR)
    {
        pStick->rightX = static_cast<int8_t>(npadAnalogStickR.x >> 8);
        pStick->rightY = static_cast<int8_t>(npadAnalogStickR.y >> 8);
    }
#else
    NN_UNUSED(pButton);
    NN_UNUSED(pStick);
#endif
}

nn::Bit32 Pad::GetNpadState(nn::hid::NpadButtonSet* pNpadState,
    nn::hid::AnalogStickState* pAnalogStickL,
    nn::hid::AnalogStickState* pAnalogStickR)
{
    nn::Bit32 result = 0;

#if !defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    pAnalogStickL->x = 0;
    pAnalogStickL->y = 0;
    pAnalogStickR->x = 0;
    pAnalogStickR->y = 0;

    for (int idxNpad = 0; idxNpad < NPADID_NUM; ++idxNpad)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadIds[idxNpad]);

        pNpadState[idxNpad].Reset();

        // ジョイコン操作が有効な場合
        if (style.Test<nn::hid::NpadStyleJoyDual>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::NpadJoyDualState joyDualState;
            nn::hid::GetNpadState(&joyDualState, m_NpadIds[idxNpad]);
            pNpadState[idxNpad] = joyDualState.buttons;
            result |= NpadState_Buttons;

            // analogStickの確認
            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLDown>())
            {
                *pAnalogStickL = joyDualState.analogStickL;
                result |= NpadState_AnalogStickL;
            }

            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRDown>())
            {
                *pAnalogStickR = joyDualState.analogStickR;
                result |= NpadState_AnalogStickR;
            }
        }

        // 携帯機コントローラー操作が有効な場合
        if (style.Test<nn::hid::NpadStyleHandheld>() == true)
        {
            //最新のNpadのステートを取得
            nn::hid::NpadHandheldState handheldState;
            nn::hid::GetNpadState(&handheldState, m_NpadIds[idxNpad]);
            pNpadState[idxNpad] = handheldState.buttons;
            result |= NpadState_Buttons;

            // analogStickの確認
            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickLDown>())
            {
                *pAnalogStickL = handheldState.analogStickL;
                result |= NpadState_AnalogStickL;
            }

            if (pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRRight>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRUp>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRLeft>() ||
                pNpadState[idxNpad].Test< nn::hid::NpadJoyButton::StickRDown>())
            {
                *pAnalogStickR = handheldState.analogStickR;
                result |= NpadState_AnalogStickR;
            }
        }
    }
#else
    NN_UNUSED(pNpadState);
    NN_UNUSED(pAnalogStickL);
    NN_UNUSED(pAnalogStickR);
#endif

    return result;
}
