﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TimeStandardUserSystemClock.cpp,PageSampleTimeStandardUserSystemClock}
 *
 * @brief
 *  nn::time::StandardUserSystemClock のサンプルプログラム
 */

/**
 * @page PageSampleTimeStandardUserSystemClock StandardUserSystemClock
 * @tableofcontents
 *
 * @brief
 *  nn::time::StandardUserSystemClock を扱うサンプルプログラムの解説です。
 *
 * @section TimeStandardUserSystemClock_SectionBrief 概要
 *  ここでは、 nn::time::StandardUserSystemClock を扱うサンプルプログラムの解説をします。
 *  このクロックはデバイスが持つ現在時刻を指し示し、人間の理解可能な日時表現に変換できる値を取得することができます。
 *
 *  @ref nn::time::StandardUserSystemClock "StandardUserSystemClock の関数リファレンス" も併せて参照して下さい。
 *
 * @section TimeStandardUserSystemClock_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/TimeStandardUserSystemClock Samples/Sources/Applications/TimeStandardUserSystemClock @endlink 以下にあります。
 *
 * @section TimeStandardUserSystemClock_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section TimeStandardUserSystemClock_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section TimeStandardUserSystemClock_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section TimeStandardUserSystemClock_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section TimeStandardUserSystemClock_SectionDetail 解説
 *
 * @subsection TimeStandardUserSystemClock_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  TimeStandardUserSystemClock.cpp
 *  @includelineno TimeStandardUserSystemClock.cpp
 *
 * @subsection TimeStandardUserSystemClock_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - TIME ライブラリの初期化
 * - POSIX 時間 ( nn::time::PosixTime ) を取得し、人間が理解できる日時 ( nn::time::CalendarTime ) へ変換を行うサンプル
 * - 時刻が操作されたかを検知するサンプル
 * - TIME ライブラリの終了
 *
 * 時刻が操作されたかを検知する機能においては、
 * nn::time::SystemClockContext の値をセーブデータ等で永続化することで
 * アプリケーションのシャットダウンや本体の電源断をまたいだ操作の検知を行うことができます。
 *
 * nn::Result を返す API において、事前条件を満たしていれば必ず成功するものは
 * @ref NN_ABORT_UNLESS_RESULT_SUCCESS を利用してハンドリングしています。
 *
 * このサンプルの実行結果を以下に示します。
 * ただし、 CalendarTime の値は実行するたびに異なるものが表示されます。
 *
 * @verbinclude TimeStandardUserSystemClock_ExampleOutput.txt
 *
 */

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

extern "C" void nnMain()
{
    // TIME ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Initialize() );
    NN_LOG("nn::time::Initialize\n");

    // POSIX 標準の経過秒数を取得し、人間が理解できる日時へ変換を行うサンプルです。
    NN_LOG("---- Sample of getting current posix time, and converting to calendar date and hour.\n");
    {
        // 協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数で現在時刻を取得します。
        nn::time::PosixTime posixTime;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );

        // PosixTime を CalendarTime へ変換します。
        // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用します。
        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
        NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
            calendarAdditionalInfo.timeZone.standardTimeName,
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);
    }


    // 2つの SystemClockContext を比較して時刻が操作されたかを検知するサンプルです。
    NN_LOG("---- Sample of check if nn::time::StandardUserSystemClock is operated.\n");
    {
        // 比較する元となる SystemClockContext を取得します。
        nn::time::SystemClockContext oldSystemClockContext;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&oldSystemClockContext) );

        // 上記の SystemClockContext の値をセーブデータ等に保存して永続化し、次回起動時に読み出すことで、
        // ここでアプリケーションのシャットダウンや本体の電源断が発生したとしても、
        // 時計の操作の有無を判定することができます。

        // 比較対象となる現在の SystemClockContext を取得します。
        nn::time::SystemClockContext latestSystemClockContext;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetSystemClockContext(&latestSystemClockContext) );

        // 2つの SystemClockContext を比較して時計操作が行われたかを確認できます。
        if(oldSystemClockContext == latestSystemClockContext)
        {
            // 時計の操作がされていない場合ここに来ます。
            NN_LOG("StandardUserSystemClock was not operated.\n");
        }
        else
        {
            // 時計が操作された場合にここに来ます。
            NN_LOG("StandardUserSystemClock was operated.\n");
        }
    }

    // TIME ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
    NN_LOG("nn::time::Finalize\n");
}

