﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{TimeClaimPeriodicBenefit.cpp,PageSampleTimeClaimPeriodicBenefitWithUserSystemClock}
 *
 * @brief
 *  ユーザー時計( @ref nn::time::StandardUserSystemClock )の時刻を使って、
 *  定期的な利益を得られるかどうかを判定するサンプルプログラム
 */

/**
 * @page PageSampleTimeClaimPeriodicBenefitWithUserSystemClock ClaimPeriodicBenefitWithUserSystemClock
 * @tableofcontents
 *
 * @brief
 *  ユーザー時計( @ref nn::time::StandardUserSystemClock )の時刻を使って、
 *  時計の操作を考慮した上で定期的な利益を得られるかどうかを判定するサンプルプログラムの解説です。
 *
 * @section ClaimPeriodicBenefitWithUserSystemClock_SectionBrief 概要
 *  ここでは、 nn::time::ClaimPeriodicBenefitWithUserSystemClock 関数を利用して、
 *  毎日日付が変わると利益を得られる、毎週特定の曜日にだけイベントが発生する、
 *  といった日時を基準に定期的に発動するイベントを、ユーザー時計の時刻で管理するサンプルプログラムの解説をします。
 *
 *  @ref nn::time::ClaimPeriodicBenefitWithUserSystemClock "ClaimPeriodicBenefitWithUserSystemClock の関数リファレンス"
 *  も併せて参照して下さい。
 *
 * @section ClaimPeriodicBenefitWithUserSystemClock_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/TimeClaimPeriodicBenefit Samples/Sources/Applications/TimeClaimPeriodicBenefit @endlink 以下にあります。
 *
 * @section TimeClaimPeriodicBenefitWithUserSystemClock_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section TimeClaimPeriodicBenefitWithUserSystemClock_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section TimeClaimPeriodicBenefitWithUserSystemClock_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section TimeClaimPeriodicBenefitWithUserSystemClock_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section TimeClaimPeriodicBenefitWithUserSystemClock_SectionDetail 解説
 *
 * @subsection TimeClaimPeriodicBenefitWithUserSystemClock_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  TimeClaimPeriodicBenefit.cpp
 *  @includelineno TimeClaimPeriodicBenefit.cpp
 *
 * @subsection TimeClaimPeriodicBenefitWithUserSystemClock_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - 時刻ライブラリの初期化
 * - 定期的な利益の受け取り判定(3回)
 * - 時刻ライブラリの終了
 *
 * このサンプルでは、ユーザー時計の時刻の「分」が変わるたびに、新しく利益を得られる定期イベントを管理しています。
 * @n
 * サンプルとして短い実行時間でライブラリの機能を確認できるように、
 * ユーザー時計の時刻の「分」が変わるたびに新しく利益を受け取る実装をしていますが、
 * 実際のアプリケーションでは「日付が変わるたびに新しく利益を得ることができる」「毎週水曜日に1度だけ利益を得ることができる」
 * といったような、「分」の切り替わりよりも大きな期間でのユースケースを想定しています。
 *
 * ユーザー時計の時刻に関係なく、「前回の利益受け取りから○○時間が経過することで新しく利益を得ることができる」といった実装は、
 * nn::time::CalculateSpanBetween() や nn::time::GetSpanBetween() で前回の利益受け取りからの経過時間を計算し、
 * 一定時間以上が経過しているかどうかを判定してください。
 * nn::time::GetSpanBetween() を使った経過時間の算出については @subpage PageSampleTimeStandardSteadyClock サンプル
 * を用意しているのでご参照ください。
 * @n
 * 前回の利益受け取りからの経過時間での判定のほうが、本サンプルよりも実装、挙動共に簡潔であり、
 * 実装コスト、テストコストの両方が少なく済むので、アプリケーションの企画の段階で一度検討してみることをお勧めします。
 *
 * nn::Result を返す API において、事前条件を満たしていれば必ず成功するものは
 * @ref NN_ABORT_UNLESS_RESULT_SUCCESS を利用してハンドリングしています。
 *
 * このサンプルの実行結果を以下に示します。
 * ただし、日時の値は実行するたびに異なるものが表示されます。
 * @verbinclude TimeClaimPeriodicBenefit_ExampleOutput1.txt
 *
 * また、利益を得られるかどうかの判定が「分」を跨いだ場合には、
 * 以下のように連続で "I can receive benefit!" が表示されることがあります。
 * @verbinclude TimeClaimPeriodicBenefit_ExampleOutput2.txt
 *
 */

#include <nn/time.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_ThreadApi.h>

// 1分に1度(ユーザー時計の時刻の分が変わるたびに)利益を受け取るための日時確認コールバッククラス
class PerMinuteBenefitCheckCallback : public nn::time::CheckCalendarTimeCallback
{
public:
    virtual bool IsSameTiming(
        const nn::time::CalendarTime& lhs,
        const nn::time::CalendarTime& rhs) const NN_NOEXCEPT NN_OVERRIDE
    {
        // この関数では2つの日時が、同じ利益の受け取りタイミングかどうかを判定します

        /*
        // 例えば、日付が変わることで1度だけ利益を得られるのであれば、以下のように日付(年月日)が同じであれば true を返します。
        return
            lhs.year == rhs.year &&
            lhs.month == rhs.month &&
            lhs.day == rhs.day;
         */

        /*
        // 例えば、毎日深夜4時を過ぎることで、新しく利益の受け取りが可能になるのであれば、以下のように実装します。
        nn::time::CalendarTime lhsCalendar = lhs - nn::TimeSpan::FromHours(4);
        nn::time::CalendarTime rhsCalendar = rhs - nn::TimeSpan::FromHours(4);
        // 4時間さかのぼった日時の、年月日が同じであれば同じ利益の受け取りタイミング
        return
            lhsCalendar.year == rhsCalendar.year &&
            lhsCalendar.month == rhsCalendar.month &&
            lhsCalendar.day == rhsCalendar.day;
         */

        // 本サンプルでは、年月日時分が同じであれば同じ受け取りタイミングなので true を返します。
        return
            lhs.year == rhs.year &&
            lhs.month == rhs.month &&
            lhs.day == rhs.day &&
            lhs.hour == rhs.hour &&
            lhs.minute == rhs.minute;
    }

    virtual bool IsMatched(
        const nn::time::CalendarTime& calendarTime) const NN_NOEXCEPT NN_OVERRIDE
    {
        // この関数では与えられた CalendarTime が、利益を受け取ることができる日時として適切かを判定します。

        /*
        // 例えば、水曜日にだけ利益を得られるのであれば、以下のように calendarTime が水曜日のときにだけ true を返します。
        nn::time::DayOfWeek dayOfWeek = nn::time::GetDayOfWeek(calendarTime.year, calendarTime.month, calendarTime.day);
        return dayOfWeek == nn::time::DayOfWeek_Wednesday;
         */

        // 本サンプルの利益受け取りは、日時に関係なくいつでも受け取ることができる可能性があるので、何もせず true を返します。
        NN_UNUSED(calendarTime);
        return true;
    }
};

extern "C" void nnMain()
{
    // 時刻ライブラリの初期化
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());
    NN_LOG("nn::time::Initialize\n");

    NN_LOG("---- Sample of claim periodic benefit with nn::time::StandardUserSystemClock.\n");

    // nn::time::ClaimPeriodicBenefitWithUserSystemClock() 用の PeriodicBenefitClaimContext
    // (この PeriodicBenefitClaimContext は、扱いたい定期イベントの種類の数だけインスタンスが必要です。
    // 初回の判定時には、コンストラクタで初期化されたままのオブジェクトを ClaimPeriodicBenefitWithUserSystemClock() へ渡してください。)
    nn::time::PeriodicBenefitClaimContext handoverContext;

    // ペナルティの最大期間
    // (時計操作時に課す利益を得られないペナルティ期間の最大値です。
    // このサンプルでは、1分に1度(時刻の分が変わるたびに)利益を受け取れることを考慮し、ペナルティ最大期間を1分としています。
    // 詳細は API リファレンスを参照してください。)
    const nn::TimeSpan MaxPenaltyTimeSpan = nn::TimeSpan::FromMinutes(1);

    // 時計を進めた場合に無視できる時計の操作量
    // (時計を進めたとしても、この値以下の操作量である場合は時計操作がなかったとして扱われます。
    // このサンプルでは、挙動の簡潔さを重視して 0 としています。
    // 詳細は API リファレンスを参照してください。)
    const nn::TimeSpan AcceptableOperationTimeSpan(0);

    // 日時確認用のクラス
    // (nn::time::CheckCalendarTimeCallback クラスを継承し、内部実装する必要があります。
    // APIリファレンスや、このサンプルでの実装を参照してください。)
    PerMinuteBenefitCheckCallback perMinuteBenefitCheckCallback;

    const int TryCount = 3;
    for(int i = 1 ; i <= TryCount ; i++)
    {
        // 利益受け取りを判定する現在の ClockSnapshot を作成
        nn::time::ClockSnapshot currentSnapshot;
        nn::time::ClockSnapshot::CreateWithStandardSystemClock(&currentSnapshot);
        const auto c = currentSnapshot.GetStandardUserSystemClockCalendarTime();
        NN_LOG("[Claim Benefit (%d/%d) at %04d/%02d/%02d %02d:%02d:%02d]\n",
                i, TryCount,
                c.year, c.month, c.day, c.hour, c.minute, c.second);

        auto backup = handoverContext; // ClaimPeriodicBenefitWithUserSystemClock()実行後の handoverContext 更新検知用

        auto result = nn::time::ClaimPeriodicBenefitWithUserSystemClock(
            &handoverContext,
            currentSnapshot,
            MaxPenaltyTimeSpan,
            AcceptableOperationTimeSpan,
            &perMinuteBenefitCheckCallback);

        if(result == nn::time::PeriodicBenefitClaimResult_Success)
        {
            // 定期的な利益を得ることができます。
            // アプリケーションの企画にそったアイテムやポイントの付与や、
            // 何らかの定期イベント発生をトリガーさせることを想定しています。
            NN_LOG("    I can receive benefit!\n");
        }
        else if(result == nn::time::PeriodicBenefitClaimResult_NotReceivableTiming)
        {
            // 前回利益を得たタイミングから、新しく利益を得られるタイミングになっていないため、利益を得られません。
            NN_LOG("    nn::time::ClaimPeriodicBenefitWithUserSystemClock() failed.(nn::time::PeriodicBenefitClaimResult_NotReceivableTiming)\n");
        }
        else if(result == nn::time::PeriodicBenefitClaimResult_Penalty)
        {
            // 時計操作によって利益を得られないペナルティ期間が発生しています。
            // ペナルティ期間の開始は、時計を操作した瞬間です。
            NN_LOG("    nn::time::ClaimPeriodicBenefitWithUserSystemClock() failed.(nn::time::PeriodicBenefitClaimResult_Penalty)\n");
        }
        else
        {
            // 上記以外の Result は発生しません
            NN_ABORT("Unexpected result(%d)", result);
        }

        // 上記の handoverContext をセーブデータ等に保存して永続化し、次回の判定へ引き継ぐことで、
        // アプリケーションのシャットダウンや本体の電源断を跨いで、定期的な利益の受け取りを管理することができます。
        // このとき、nn::time::ClaimPeriodicBenefitWithUserSystemClock() の結果の成否(resultの成否)に依らず、
        // handoverContext を次回の判定へ引き継ぐ必要がある点に注意してください。
        //
        // また、アプリケーション側の利益の付与結果の永続化を行う場合には、
        // データの不整合を防ぐために、判定用の PeriodicBenefitClaimContext (handoverContext) と、
        // アプリケーション側の利益の付与結果とを、不可分に永続化することを強く推奨します。
        //
        // PeriodicBenefitClaimContext は C++ 標準の TriviallyCopyable の要件を満たしており、 std::memcpy() が可能なクラスなので、
        // そのままセーブデータ等に永続化が可能です。
        //
        // 推奨される実装の疑似コードは以下のようになります
        /*
        bool isCommitRequired = false;
        if(backup != handoverContext) // resultの成否に依らず、更新があれば永続化が必要です
        {
            WriteContext(handoverContext); // handoverContext を永続化
            isCommitRequired = true;
        }

        // ここでアプリケーションのシャットダウンや電源断が発生しても、不整合は発生しません

        if(result.IsSuccess())
        {
            WriteBenefit(...); // アプリケーション側の利益付与を永続化
            isCommitRequired = true;
        }

        // ここでアプリケーションのシャットダウンや電源断が発生しても、不整合は発生しません

        if(isCommitRequired)
        {
            CommitSaveData(); // 上記を同時に実セーブデータへ反映
        }
         */
        NN_UNUSED(backup);

        NN_LOG("    Waiting 3 seconds.\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        // このスリープの間に、HOMEボタンを押して DevMenu もしくは HOME メニューを表示し、
        // そこで時刻やタイムゾーンの設定を変更することで、以降の判定を時計操作検知側へ分岐させることができます。
        //
        // 時計操作後、サンプルアプリケーションに1分以内に戻ることで、ペナルティが発生して利益を得られないことを確認できます。
        // 時計操作後、サンプルアプリケーションに戻らず1分以上を(DevMenu 上もしくは HOME メニュー上で)経過させることで
        // (すなわち時計操作後に MaxPenaltyTimeSpan 以上の時間が過ぎ去ることで)、再び利益を得られることを確認できます。
    }

    // 時刻ライブラリの終了
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::Finalize() );
    NN_LOG("nn::time::Finalize\n");
}

