﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Capture.h"

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>

#include "ApplicationHeap.h"


Capture::Capture() NN_NOEXCEPT
    : m_pApplicationHeap( nullptr )
    , m_pGraphicsSystem( nullptr )
    , m_pTexturePoolMemory( nullptr )
{
}

void Capture::Initialize(
    ApplicationHeap* pApplicationHeap,
    nns::gfx::GraphicsFramework* pGraphicsSystem) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pApplicationHeap);
    NN_ASSERT_NOT_NULL(pGraphicsSystem);

    m_pApplicationHeap = pApplicationHeap;

    m_pGraphicsSystem = pGraphicsSystem;

    // メモリプールの用意
    InitializeGfxMemoryPool();

    // テクスチャの用意
    InitializeGfxTexture();

    // テクスチャビューの用意(テクスチャを紐付け)
    InitializeGfxTextureView();

    // テクスチャビュースロットをデスクリプタプールに登録
    {
        m_TextureSlot = m_pGraphicsSystem->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        m_pGraphicsSystem->SetTextureViewToDescriptorPool( m_TextureSlot, &g_TextureView );
    }

    // サンプラの準備
    {
        m_SamplerSlot = m_pGraphicsSystem->AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
    }

    // バッファの作成
    InitializeGfxBuffer();
}

void Capture::InitializeGfxMemoryPool()
{
    // メモリプールの用意
    {
        const size_t MemoryPoolSize = 64 * 1024 * 1024;

        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();
        info.SetMemoryPoolProperty(
            nn::gfx::MemoryPoolProperty_CpuUncached |
            nn::gfx::MemoryPoolProperty_GpuCached );

        const size_t aligment =
            nn::gfx::MemoryPool::GetPoolMemoryAlignment(
                m_pGraphicsSystem->GetDevice(),
                info );

        m_pMemoryPool =
            m_pApplicationHeap->Allocate( MemoryPoolSize, aligment );

        info.SetPoolMemory( m_pMemoryPool, MemoryPoolSize );

        m_MemoryPool.Initialize( m_pGraphicsSystem->GetDevice(), info );

        m_pMemoryPoolOffset = 0;

        m_pMemoryPoolCommandOffset = 0;
    }

    // テクスチャ用メモリプールの用意
    {
        const size_t MemoryPoolSizeInvisible = 32 * 1024 * 1024;

        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();
        info.SetMemoryPoolProperty(
            nn::gfx::MemoryPoolProperty_CpuInvisible |
            nn::gfx::MemoryPoolProperty_GpuCached |
            nn::gfx::MemoryPoolProperty_Compressible );

        const size_t aligment =
            nn::gfx::MemoryPool::GetPoolMemoryAlignment(
                m_pGraphicsSystem->GetDevice(),
                info );

        m_pTexturePoolMemory =
            m_pApplicationHeap->Allocate( MemoryPoolSizeInvisible, aligment );

        info.SetPoolMemory( m_pTexturePoolMemory, MemoryPoolSizeInvisible );

        m_TextureMemoryPool.Initialize( m_pGraphicsSystem->GetDevice(), info );

        m_pTextureMemoryPoolOffset = 0;
    }
}

void Capture::InitializeGfxTexture()
{
    nn::gfx::ImageFormat format = nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;

    nn::gfx::Texture::InfoType textureInfo;
    textureInfo.SetDefault();
    textureInfo.SetGpuAccessFlags( nn::gfx::GpuAccess_Texture );
    textureInfo.SetImageStorageDimension( nn::gfx::ImageStorageDimension_2d );
    textureInfo.SetMipCount( 1 );
    textureInfo.SetTileMode( nn::gfx::TileMode_Linear );
    textureInfo.SetWidth( 1280 );
    textureInfo.SetHeight( 720 );
    textureInfo.SetImageFormat( format );

    {
        //
        m_pTextureMemoryPoolOffset = nn::util::align_up(
            m_pTextureMemoryPoolOffset,
            nn::gfx::Texture::CalculateMipDataAlignment(
                m_pGraphicsSystem->GetDevice(),
                textureInfo ) );
        //
        size_t size = nn::gfx::Texture::CalculateMipDataSize(
            m_pGraphicsSystem->GetDevice(),
            textureInfo );
        //
        m_Texture.Initialize(
            m_pGraphicsSystem->GetDevice(),
            textureInfo,
            &m_TextureMemoryPool,
            m_pTextureMemoryPoolOffset,
            size );
        //
        m_pTextureMemoryPoolOffset += size;
    }
}

void Capture::InitializeGfxTextureView()
{
    nn::gfx::ImageFormat format = nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb;

    nn::gfx::TextureView::InfoType viewInfo;
    viewInfo.SetDefault();
    viewInfo.SetImageDimension( nn::gfx::ImageDimension_2d );
    viewInfo.SetImageFormat( format );
    viewInfo.SetTexturePtr( &m_Texture );
    // ChannelMapping はデフォルト値 RGBA

    g_TextureView.Initialize( m_pGraphicsSystem->GetDevice(), viewInfo );
}

void Capture::InitializeGfxBuffer()
{
    nn::gfx::Buffer::InfoType info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_Read );
    info.SetSize( 1280 * 720 * 4 );

    {
        m_pMemoryPoolOffset = nn::util::align_up(
            m_pMemoryPoolOffset,
            nn::gfx::Buffer::GetBufferAlignment(
                m_pGraphicsSystem->GetDevice(),
                info ) );
        m_Buffer.Initialize(
            m_pGraphicsSystem->GetDevice(),
            info,
            &m_MemoryPool,
            m_pMemoryPoolOffset,
            info.GetSize() );
        m_pMemoryPoolOffset += info.GetSize();
    }
}

void Capture::Finalize() NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( m_pApplicationHeap );
    NN_ASSERT_NOT_NULL( m_pGraphicsSystem );

    m_Buffer.Finalize( m_pGraphicsSystem->GetDevice() );
    g_TextureView.Finalize( m_pGraphicsSystem->GetDevice() );
    m_Texture.Finalize( m_pGraphicsSystem->GetDevice() );
    m_TextureMemoryPool.Finalize( m_pGraphicsSystem->GetDevice() );
    m_pApplicationHeap->Deallocate( m_pTexturePoolMemory );
    m_MemoryPool.Finalize( m_pGraphicsSystem->GetDevice() );
    m_pApplicationHeap->Deallocate( m_pMemoryPool );
}

void Capture::Set( void* pBuf )
{
    NN_ASSERT_NOT_NULL( pBuf );

    {
        uint16_t* pDest = m_Buffer.Map< uint16_t >();
        memcpy( pDest, pBuf, 1280 * 720 * 4 );

        m_Buffer.FlushMappedRange( 0, 1280 * 720 * 4 );
        m_Buffer.Unmap();
    }
}

void Capture::Draw( nns::gfx::PrimitiveRenderer::Renderer* pRenderer, int bufferIndex ) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL( m_pGraphicsSystem );

    {
        nn::gfx::TextureCopyRegion dstRegion;
        dstRegion.SetDefault();
        dstRegion.SetWidth( m_pGraphicsSystem->GetDisplayWidth() );
        dstRegion.SetHeight( m_pGraphicsSystem->GetDisplayHeight() );
        m_pGraphicsSystem->GetRootCommandBuffer( bufferIndex )->CopyBufferToImage(
            &m_Texture,
            dstRegion,
            &m_Buffer,
            0 );
    }

    {
        nn::gfx::DescriptorSlot textureDescriptor;
        nn::gfx::DescriptorPool* pTextureDescriptorPool = m_pGraphicsSystem->GetDescriptorPool( nn::gfx::DescriptorPoolType_TextureView );
        pTextureDescriptorPool->GetDescriptorSlot(
            &textureDescriptor,
            m_TextureSlot );
        nn::gfx::DescriptorSlot samplerDescriptor;
        nn::gfx::DescriptorPool* pSamplerDescriptorPool = m_pGraphicsSystem->GetDescriptorPool( nn::gfx::DescriptorPoolType_Sampler );
        pSamplerDescriptorPool->GetDescriptorSlot(
            &samplerDescriptor,
            m_SamplerSlot - 1 );

        pRenderer->DrawScreenQuad(
            m_pGraphicsSystem->GetRootCommandBuffer( bufferIndex ),
            textureDescriptor,
            samplerDescriptor );
    }
}
