﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{SpySimple.cpp,PageSampleSpySimple}
 *
 * @brief
 *  nn::spy ライブラリの基本機能の使い方を示すサンプルプログラム
 */

/**
 * @page PageSampleSpySimple SpySimple 基本機能
 * @tableofcontents
 *
 * @brief
 *  nn::spy ライブラリの基本機能のサンプルプログラムの解説です。
 *
 * @section PageSampleSpySimple_SectionBrief 概要
 *  nn::spy ライブラリを使ってアプリケーションの状態をモニタするもっともシンプルなサンプルです。
 *
 * @section PageSampleSpySimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/SpySimple Samples/Sources/Applications/SpySimple @endlink 以下にあります。
 *
 * @section PageSampleSpySimple_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleSpySimple_SectionHowToOperate 操作方法
 *  サンプルを起動するとコンソールに操作方法が表示されます。
 *
 * @section PageSampleSpySimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *  PC 上で Spy ツールを起動して、サンプルプログラムと接続します。
 *
 * @section PageSampleSpySimple_SectionDetail 解説
 *
 * @subsection PageSampleSpySimple_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  SpySimple.cpp
 *  @includelineno SpySimple.cpp
 *
 * @subsection PageSampleSpySimple_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 * - nn::spy ライブラリの初期化 (InitializeSpy())
 *   - nn::spy::SpyController の初期化
 *   - nn::spy::PlotFloat の初期化とアタッチ
 *   - nn::spy::PlotState の初期化とアタッチ
 *   - nn::spy::Marker のアタッチ
 * - メインループ (Mainloop())
 *   - 操作に応じて Spy ツールに情報を送信します。
 * - nn::spy ライブラリの終了処理 (FinalizeSpy())
 *
 */

#include <algorithm>
#include <climits>
#include <cmath>

#include <nn/nn_Assert.h>
#include <nns/nns_Log.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/htcs.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/settings.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/util/util_FormatString.h>

#include <nn/spy.h>

#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic ignored "-Wunused-const-variable"
#pragma clang diagnostic ignored "-Wunused-function"
#endif

//-----------------------------------------------------------------------------

namespace {

    const size_t HeapSize = 16 * 1024 * 1024;
    const size_t SpyControllerDataBufferLength = 1024 * 1024;

    char g_Heap[HeapSize];
    nn::mem::StandardAllocator g_Allocator;

    const nn::hid::NpadIdType g_NpadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::Handheld,
    };

    const int NpadIdCountMax = sizeof(g_NpadIds) / sizeof(g_NpadIds[0]);

    nn::hid::DebugPadButtonSet g_PadButtonSet;
    nn::hid::DebugPadButtonSet g_PadLastButtonSet;

    nn::spy::SpyController g_SpyController;
    nn::spy::PlotFloat g_PlotFloat;
    nn::spy::PlotState g_PlotState;
    nn::spy::Marker g_Marker;

    int g_AppFrameCount;
    int g_LogCount;
    int g_PlotStateCount;
    int g_MarkerCount;

    double GetRandom()
    {
        static uint32_t s_Seed = 1;
        s_Seed = s_Seed * 1566083941 + 1;
        return (1.0 / (std::numeric_limits<uint32_t>::max() + 1.0)) * s_Seed;
    }

    template <typename T, T max>
    T GetRandom()
    {
        return std::min(max, static_cast<T>(GetRandom() * (max + 1)));
    }

    void* Alloc(size_t size)
    {
        return g_Allocator.Allocate(size);
    }

    void Free(void* addr, size_t size)
    {
        NN_UNUSED(size);
        g_Allocator.Free(addr);
    }

    void InitializeHid()
    {
        nn::hid::InitializeDebugPad();

        //キーボードのキーを DebugPad のボタンに割り当てます。
        nn::settings::DebugPadKeyboardMap map;
        nn::settings::GetDebugPadKeyboardMap(&map);
        map.buttonA = nn::hid::KeyboardKey::A::Index;
        map.buttonB = nn::hid::KeyboardKey::B::Index;
        map.buttonX = nn::hid::KeyboardKey::X::Index;
        map.buttonY = nn::hid::KeyboardKey::Y::Index;
        map.buttonL = nn::hid::KeyboardKey::L::Index;
        map.buttonR = nn::hid::KeyboardKey::R::Index;
        map.buttonLeft = nn::hid::KeyboardKey::LeftArrow::Index;
        map.buttonRight = nn::hid::KeyboardKey::RightArrow::Index;
        map.buttonUp = nn::hid::KeyboardKey::UpArrow::Index;
        map.buttonDown = nn::hid::KeyboardKey::DownArrow::Index;
        map.buttonStart = nn::hid::KeyboardKey::Space::Index;
        nn::settings::SetDebugPadKeyboardMap(map);

        nn::hid::InitializeNpad();

        nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleFullKey::Mask | nn::hid::NpadStyleHandheld::Mask);
        nn::hid::SetSupportedNpadIdType(g_NpadIds, NpadIdCountMax);
    }

    void InitializeSpy()
    {
        nn::htcs::Initialize(Alloc, Free);

#if defined(NN_BUILD_CONFIG_SPY_ENABLED)
        // nn::spy::SpyController の初期化
        nn::spy::SpyController::InitializeArg initializeArg;
        initializeArg.dataBufferLength = SpyControllerDataBufferLength;
        size_t memorySize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
        void* pMemory = Alloc(memorySize);
        NN_ABORT_UNLESS_NOT_NULL(pMemory);
        g_SpyController.Initialize(initializeArg, pMemory, memorySize);

        // Spy ツールと通信を開始
        nn::spy::SpyController::OpenArg openArg;
        bool openResult = g_SpyController.Open(openArg);
        NN_ABORT_UNLESS(openResult);
#endif

        // nn::spy::PlotFloat の初期化とアタッチ
        g_PlotFloat.SetName("PlotFloat");
        g_PlotFloat.SetRange(0.0, 255.0);
        g_SpyController.GetPlotModule().AttachItem(g_PlotFloat);

        // nn::spy::PlotState の初期化とアタッチ
        g_PlotState.SetName("PlotState");
        g_SpyController.GetPlotModule().AttachItem(g_PlotState);

        // nn::spy::Marker のアタッチ
        g_SpyController.GetMarkerModule().AttachMarker(g_Marker);

        g_AppFrameCount = 0;

        g_LogCount = 0;
        g_PlotStateCount = 0;
        g_MarkerCount = 0;
    }

    void FinalizeSpy()
    {
        g_SpyController.Finalize();
        nn::htcs::Finalize();
    }

    // Npad のボタン状態を DebugPad のボタン状態に変換します。
    const nn::hid::DebugPadButtonSet MapNpadButtonSet(const nn::hid::NpadButtonSet& buttons)
    {
        static const struct {
            nn::hid::NpadButtonSet from;
            nn::hid::DebugPadButtonSet to;
        } buttonMapList[] = {
            {
                nn::hid::NpadButton::A::Mask,
                nn::hid::DebugPadButton::A::Mask,
            },
            {
                nn::hid::NpadButton::B::Mask,
                nn::hid::DebugPadButton::B::Mask,
            },
            {
                nn::hid::NpadButton::X::Mask,
                nn::hid::DebugPadButton::X::Mask,
            },
            {
                nn::hid::NpadButton::Y::Mask,
                nn::hid::DebugPadButton::Y::Mask,
            },
            {
                nn::hid::NpadButton::L::Mask,
                nn::hid::DebugPadButton::L::Mask,
            },
            {
                nn::hid::NpadButton::R::Mask,
                nn::hid::DebugPadButton::R::Mask,
            },
            {
                nn::hid::NpadButton::Left::Mask,
                nn::hid::DebugPadButton::Left::Mask,
            },
            {
                nn::hid::NpadButton::Right::Mask,
                nn::hid::DebugPadButton::Right::Mask,
            },
            {
                nn::hid::NpadButton::Up::Mask,
                nn::hid::DebugPadButton::Up::Mask,
            },
            {
                nn::hid::NpadButton::Down::Mask,
                nn::hid::DebugPadButton::Down::Mask,
            },
            {
                nn::hid::NpadButton::Plus::Mask,
                nn::hid::DebugPadButton::Start::Mask,
            },
        };

        nn::hid::DebugPadButtonSet result = {};

        for (const auto& buttonMap : buttonMapList)
        {
            if ((buttons & buttonMap.from) == buttonMap.from)
            {
                result |= buttonMap.to;
            }
        }

        return result;
    }

    // デバッグパッドを更新します。
    nn::hid::DebugPadButtonSet UpdateDebugPad()
    {
        g_PadLastButtonSet = g_PadButtonSet;

        {
            nn::hid::DebugPadState state;
            nn::hid::GetDebugPadState(&state);
            g_PadButtonSet = state.buttons;
        }

        for (auto id : g_NpadIds)
        {
            const auto styles = nn::hid::GetNpadStyleSet(id);

            if (styles.Test<nn::hid::NpadStyleFullKey>())
            {
                nn::hid::NpadFullKeyState state;
                nn::hid::GetNpadState(&state, id);
                g_PadButtonSet |= MapNpadButtonSet(state.buttons);
            }

            if (styles.Test<nn::hid::NpadStyleHandheld>())
            {
                nn::hid::NpadHandheldState state;
                nn::hid::GetNpadState(&state, id);
                g_PadButtonSet |= MapNpadButtonSet(state.buttons);
            }
        }

        return ~g_PadLastButtonSet & g_PadButtonSet;
    }

    void Mainloop()
    {
#if defined(NN_BUILD_CONFIG_SPY_ENABLED)
        NNS_LOG(
            "-----------------------------\n"
            " SpySimple:\n"
            "\n"
            " [A]            send Log.\n"
            " [B]            send PlotFloat.\n"
            " [X]            send PlotState.\n"
            " [Y]            send Marker.\n"
            " [Start][Space] exit.\n"
            "-----------------------------\n"
            );
#else
        NNS_LOG(
            "-----------------------------\n"
            " SpySimple:\n"
            "\n"
            " Warning!\n"
            "   nn::spy is disabled.\n"
            "   nn::spy is only available on Debug or Develop build.\n"
            "\n"
            " [Start][Space] exit.\n"
            "-----------------------------\n"
            );
#endif

        while (NN_STATIC_CONDITION(true))
        {
            // フレームの開始時間を記録します
            g_SpyController.SetCurrentApplicationFrame(g_AppFrameCount++);

            nn::hid::DebugPadButtonSet triggered = UpdateDebugPad();

            if (triggered.Test<nn::hid::DebugPadButton::A>())
            {
                // Log を送信します

                g_SpyController.GetLogModule().WriteFormat("Log %d", g_LogCount);

                NNS_LOG("[A] Log %d\n", g_LogCount);
                g_LogCount++;
            }

            if (triggered.Test<nn::hid::DebugPadButton::B>())
            {
                // PlotFloat を送信します

                double value = GetRandom() * 255;

                g_PlotFloat.PushValue(value);

                NNS_LOG("[B] PlotFloat %lf\n", value);
            }

            if (triggered.Test<nn::hid::DebugPadButton::X>())
            {
                // PlotState を送信します

                char stateValue[256];
                nn::util::SNPrintf(stateValue, sizeof(stateValue), "State %d", g_PlotStateCount);
                uint8_t red = 128 + GetRandom<uint8_t, 127>();
                uint8_t green = 128 + GetRandom<uint8_t, 127>();
                uint8_t blue = 128 + GetRandom<uint8_t, 127>();

                g_PlotState.PushValue(stateValue, red, green, blue);

                NNS_LOG("[X] %s\n", stateValue);
                g_PlotStateCount++;
            }

            if (triggered.Test<nn::hid::DebugPadButton::Y>())
            {
                // Marker を送信します

                char markerDescription[256];
                nn::util::SNPrintf(markerDescription, sizeof(markerDescription), "Marker %d", g_MarkerCount);
                uint8_t red = 128 + GetRandom<uint8_t, 127>();
                uint8_t green = 128 + GetRandom<uint8_t, 127>();
                uint8_t blue = 128 + GetRandom<uint8_t, 127>();

                g_Marker.PushValue(markerDescription, red, green, blue);

                NNS_LOG("[Y] %s\n", markerDescription);
                g_MarkerCount++;
            }

            if (triggered.Test<nn::hid::DebugPadButton::Start>())
            {
                NNS_LOG("[Start] Exit.\n");
                return;
            }

            g_SpyController.GetDebugModule().PushDataBufferUsage();

            // Vsync の代わり
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
    }

} // namespace {anonymous}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    g_Allocator.Initialize(g_Heap, sizeof(g_Heap));

    InitializeHid();
    InitializeSpy();

    Mainloop();

    FinalizeSpy();
}

