﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * nn::spy::PlotModule の実験です。
 */

#include <algorithm>
#include <climits>
#include <cmath>

#include <nn/nn_Assert.h>
#include <nns/nns_Log.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/htcs.h>
#include <nn/mem.h>
#include <nn/os.h>
#include <nn/settings.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/util/util_FormatString.h>

#include <nn/spy.h>

#include "SpySandbox.h"

#if defined(NN_BUILD_CONFIG_COMPILER_CLANG)
#pragma clang diagnostic ignored "-Wunused-const-variable"
#pragma clang diagnostic ignored "-Wunused-function"
#endif

namespace {

    const size_t HeapSize = 16 * 1024 * 1024;
    const size_t SpyControllerDataBufferLength = 1024 * 1024;
    const int NodeCount = 100;

    char g_Heap[HeapSize];
    nn::mem::StandardAllocator g_Allocator;

    nn::spy::SpyController g_SpyController;
    nn::spy::PlotNode g_PlotNode[NodeCount];
    nn::spy::PlotFloat g_PlotFloat[NodeCount];
    nn::spy::PlotState g_PlotState[NodeCount];
    nn::spy::Marker g_Marker;

    uint32_t g_Seed = 1;

    int g_AppFrameCount;
    int g_LogCount;
    int g_PlotStateCount;
    int g_MarkerCount;


    double GetRandom()
    {
        g_Seed = g_Seed * 1566083941 + 1;
        return (1.0 / (std::numeric_limits<uint32_t>::max() + 1.0)) * g_Seed;
    }

    template <typename T, T max>
    T GetRandom()
    {
        return std::min(max, static_cast<T>(GetRandom() * (max + 1)));
    }

    void* Alloc(size_t size)
    {
        return g_Allocator.Allocate(size);
    }

    void Free(void* addr, size_t size)
    {
        NN_UNUSED(size);
        g_Allocator.Free(addr);
    }

    void InitializeSpy()
    {
#if defined(NN_BUILD_CONFIG_HTC_ENABLED)
        nn::htcs::Initialize(Alloc, Free);
#endif

        // nn::spy::SpyController の初期化
        nn::spy::SpyController::InitializeArg initializeArg;
        initializeArg.dataBufferLength = SpyControllerDataBufferLength;
        size_t memorySize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
        void* pMemory = Alloc(memorySize);
        NN_ABORT_UNLESS(memorySize == 0 || pMemory != nullptr);
        g_SpyController.Initialize(initializeArg, pMemory, memorySize);

        // Spy ツールと通信を開始
        nn::spy::SpyController::OpenArg openArg;
        bool openResult = g_SpyController.Open(openArg);
        NN_ABORT_UNLESS(openResult);

        // nn::spy::PlotNode の初期化とアタッチ
        for (int i = 0; i < NodeCount; ++i)
        {
            char name[256];
            nn::util::SNPrintf(name, sizeof(name), "Node %d", i);
            g_PlotNode[i].SetName(name);
            g_SpyController.GetPlotModule().AttachItem(g_PlotNode[i]);
        }

        // nn::spy::PlotFloat の初期化とアタッチ
        for (int i = 0; i < NodeCount; ++i)
        {
            g_PlotFloat[i].SetName("PlotFloat");
            g_PlotFloat[i].SetRange(0.0, 255.0);
            g_PlotFloat[i].SetParent(&g_PlotNode[i]);
            g_SpyController.GetPlotModule().AttachItem(g_PlotFloat[i]);
        }

        // nn::spy::PlotState の初期化とアタッチ
        for (int i = 0; i < NodeCount; ++i)
        {
            g_PlotState[i].SetName("PlotState");
            g_PlotState[i].SetParent(&g_PlotNode[i]);
            g_SpyController.GetPlotModule().AttachItem(g_PlotState[i]);
        }

        // nn::spy::Marker のアタッチ
        g_SpyController.GetMarkerModule().AttachMarker(g_Marker);

        g_AppFrameCount = 0;

        g_LogCount = 0;
        g_PlotStateCount = 0;
        g_MarkerCount = 0;
    }

    void FinalizeSpy()
    {
        g_SpyController.Finalize();
#if defined(NN_BUILD_CONFIG_HTC_ENABLED)
        nn::htcs::Finalize();
#endif
    }

    void Mainloop()
    {
#if defined(NN_BUILD_CONFIG_SPY_ENABLED)
        NNS_LOG(
            "-----------------------------\n"
            " PlotModule:\n"
            "\n"
            " [A]              send Log.\n"
            " [B]              send PlotFloat.\n"
            " [X]              send PlotState.\n"
            " [Y]              send Marker.\n"
            " [Up] + [+/Start][Space]  exit.\n"
            "-----------------------------\n"
            );
#else
        NNS_LOG(
            "-----------------------------\n"
            " PlotModule:\n"
            "\n"
            " Warning!\n"
            "   nn::spy is disabled.\n"
            "   nn::spy is only available on Debug or Develop build.\n"
            "\n"
            " [Up] + [+/Start][Space]  exit.\n"
            "-----------------------------\n"
            );
#endif

        while (NN_STATIC_CONDITION(true))
        {
            // フレームの開始時間を記録します
            g_SpyController.SetCurrentApplicationFrame(g_AppFrameCount++);

            SpySandbox::UpdatePadStatus();

            nn::hid::DebugPadButtonSet triggered = SpySandbox::GetButtonDown();
            nn::hid::DebugPadButtonSet buttonState = SpySandbox::GetButtonState();

            if (triggered.Test<nn::hid::DebugPadButton::A>())
            {
                // Log を送信します

                g_SpyController.GetLogModule().WriteFormat("Log %d", g_LogCount);

                NNS_LOG("[A] Log %d\n", g_LogCount);
                g_LogCount++;
            }

            if (triggered.Test<nn::hid::DebugPadButton::B>())
            {
                // PlotFloat を送信します

                double value = GetRandom() * 255;

                for (int i = 0; i < NodeCount; ++i)
                {
                    g_PlotFloat[i].PushValue(value);
                }

                NNS_LOG("[B] PlotFloat %lf\n", value);
            }

            if (triggered.Test<nn::hid::DebugPadButton::X>())
            {
                // PlotState を送信します

                char stateValue[256];
                nn::util::SNPrintf(stateValue, sizeof(stateValue), "State %d", g_PlotStateCount);
                uint8_t red = 128 + GetRandom<uint8_t, 127>();
                uint8_t green = 128 + GetRandom<uint8_t, 127>();
                uint8_t blue = 128 + GetRandom<uint8_t, 127>();

                for (int i = 0; i < NodeCount; ++i)
                {
                    g_PlotState[i].PushValue(stateValue, red, green, blue);
                }

                NNS_LOG("[X] %s\n", stateValue);
                g_PlotStateCount++;
            }

            if (triggered.Test<nn::hid::DebugPadButton::Y>())
            {
                // Marker を送信します

                char markerDescription[256];
                nn::util::SNPrintf(markerDescription, sizeof(markerDescription), "Marker %d", g_MarkerCount);
                uint8_t red = 128 + GetRandom<uint8_t, 127>();
                uint8_t green = 128 + GetRandom<uint8_t, 127>();
                uint8_t blue = 128 + GetRandom<uint8_t, 127>();

                g_Marker.PushValue(markerDescription, red, green, blue);

                NNS_LOG("[Y] %s\n", markerDescription);
                g_MarkerCount++;
            }

            if (buttonState.Test<nn::hid::DebugPadButton::Start>() &&
                buttonState.Test<nn::hid::DebugPadButton::Up>())
            {
                NNS_LOG("[Up]+[Start] Exit.\n");
                return;
            }

            // データバッファ使用量を送ります
            g_SpyController.GetDebugModule().PushDataBufferUsage();

            // Vsync の代わり
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
        }
    }

} // namespace {anonymous}

void PlotModule()
{
    g_Allocator.Initialize(g_Heap, sizeof(g_Heap));

    InitializeSpy();

    Mainloop();

    FinalizeSpy();
}
