﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nns/nns_Log.h>
#include <nn/atk.h>
#include <nn/mem/mem_StandardAllocator.h>
#include <nn/htcs.h>
#include <nn/spy.h>
#include <nn/spy/atk/spy_AtkSpyModule.h>
#include "AtkSoundProfile/Common.fsid"
#include "SpySandbox.h"

namespace
{
    const int MemoryHeapSize = 64 * 1024 * 1024;
    const int SoundHeapSize  =  4 * 1024 * 1024;
    const int MemoryPoolSize =  4 * 1024 * 1024;
    const int SoundHandleCount = 2;

    class EffectVolumeDown : public nn::atk::EffectAux
    {
    public:
        virtual bool Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            NNS_LOG("EffectVolumeDown: Initialize().\n");
            return true;
        }
        virtual void Finalize() NN_NOEXCEPT NN_OVERRIDE
        {
            NNS_LOG("EffectVolumeDown: Finalize().\n");
        }
    protected:
        virtual void UpdateSamples(int32_t* pSamples, const UpdateSamplesArg& arg) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ABORT_UNLESS_EQUAL(arg.readSampleCount % (arg.sampleCountPerAudioFrame * arg.channelCount), 0);
            int readAudioFrameCount = arg.readSampleCount / (arg.sampleCountPerAudioFrame * arg.channelCount);
            NN_ABORT_UNLESS_LESS_EQUAL(readAudioFrameCount, arg.audioFrameCount);

            for (auto frame = 0; frame < readAudioFrameCount; ++frame)
            {
                int32_t* bufferBase = pSamples + frame * arg.sampleCountPerAudioFrame * arg.channelCount;
                for (auto ch = 0; ch < arg.channelCount; ++ch)
                {
                    int32_t* pTargetSample = bufferBase + ch * arg.sampleCountPerAudioFrame;
                    for (auto i = 0; i < arg.sampleCountPerAudioFrame; ++i)
                    {
                        pTargetSample[i] /= 2;
                    }
                }
            }
        }
    };

    NN_STATIC_ASSERT(MemoryPoolSize % nn::audio::MemoryPoolType::SizeGranularity == 0);

    char g_MemoryHeap[MemoryHeapSize + SoundHeapSize];
    nn::mem::StandardAllocator g_Allocator;
    nn::mem::StandardAllocator g_MemoryPoolAllocator;

    nn::audio::MemoryPoolType g_MemoryPool;

    #define MOUNT_NAME  "asset"
    const char* SoundArchiveFilePath = MOUNT_NAME ":/AtkSoundProfile/Common.bfsar";

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;
    nn::atk::SoundHandle        g_SoundHandle[SoundHandleCount];

    EffectVolumeDown g_EffectVolumeDown;
    bool g_IsEffectClearing;
    bool g_IsEffectChanging;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForMemoryPool;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;
    void* g_pMemoryForLabelString;
    void* g_pMemoryForOpusDecoder;
    void* g_pMemoryForEffectAux;

    // 波形出力
    void* g_CircularBufferSinkBuffer;
    size_t g_CircularBufferSinkBufferSize;

    // Spy 関連
    const size_t SpyControllerDataBufferLength = 5 * 1024 * 1024;
    nn::spy::SpyController g_SpyController;
    nn::spy::atk::AtkSpyModule g_AtkSpyModule;

    nn::spy::PlotState g_TaskThreadState("AudioProcess/atk TaskThread");
    nn::spy::PlotState g_SoundThreadState("AudioProcess/atk SoundThread");
    nn::spy::PlotState g_SoundThreadDetailState("AudioProcess/DSP");

    void* g_pMemoryForSpyController;
    void* g_pMemoryForAtkSpyModule;

    void* g_MountRomCacheBuffer = nullptr;
}

void* Allocate(size_t size, size_t alignment) NN_NOEXCEPT
{
    void* pMemory = g_Allocator.Allocate( size, alignment );
    NN_ABORT_UNLESS( size == 0 || pMemory != nullptr );
    return pMemory;
}

void* Allocate(size_t size) NN_NOEXCEPT
{
    void* pMemory = g_Allocator.Allocate( size );
    NN_ABORT_UNLESS( size == 0 || pMemory != nullptr );
    return pMemory;
}

void Free(void* pMemory) NN_NOEXCEPT
{
    if( pMemory == nullptr )
    {
        return;
    }
    g_Allocator.Free( pMemory );
    pMemory = nullptr;
}

void Free(void* pMemory, size_t size) NN_NOEXCEPT
{
    NN_UNUSED( size );
    Free( pMemory );
}

void* AllocateFromMemoryPool(size_t size, size_t alignment) NN_NOEXCEPT
{
    void* pMemory = g_MemoryPoolAllocator.Allocate( size, alignment );
    NN_ABORT_UNLESS( size == 0 || pMemory != nullptr );
    return pMemory;
}

void* AllocateFromMemoryPool(size_t size) NN_NOEXCEPT
{
    void* pMemory = g_MemoryPoolAllocator.Allocate( size );
    NN_ABORT_UNLESS( size == 0 || pMemory != nullptr );
    return pMemory;
}

void FreeToMemoryPool(void* pMemory) NN_NOEXCEPT
{
    if( pMemory == nullptr )
    {
        return;
    }
    g_MemoryPoolAllocator.Free( pMemory );
}

void ClearEffect()
{
    nn::atk::SoundSystem::ClearEffect(nn::atk::AuxBus_A);
    FreeToMemoryPool(g_pMemoryForEffectAux);
    g_pMemoryForEffectAux = nullptr;
}

bool PrepareEffectRemove()
{
    g_EffectVolumeDown.SetEnabled(false);
    return true;
}

bool CheckEffectRemovable()
{
    return g_EffectVolumeDown.IsRemovable();
}

//  初期化です
void Initialize()
{
    //  atk 以外の初期化
    g_Allocator.Initialize( g_MemoryHeap, sizeof(g_MemoryHeap) );

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    g_MountRomCacheBuffer = Allocate(cacheSize);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom( MOUNT_NAME, g_MountRomCacheBuffer, cacheSize )
    );

#if defined(NN_BUILD_CONFIG_HTC_ENABLED)
    nn::htcs::Initialize(Allocate, Free);
#endif

    {
        // nn::spy::SpyController の初期化
        nn::spy::SpyController::InitializeArg initializeArg;
        initializeArg.dataBufferLength = SpyControllerDataBufferLength;
        size_t memorySize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
        g_pMemoryForSpyController = Allocate(memorySize);
        g_SpyController.Initialize(initializeArg, g_pMemoryForSpyController, memorySize);
    }

    // SoundSystem の初期化
    bool isSuccess = true;
    nn::atk::SoundSystem::SoundSystemParam param;
    param.enableProfiler = true;
    param.enableCircularBufferSink = true;
    size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize( param, reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ), memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess );

    // MemoryPool の初期化
    g_pMemoryForMemoryPool = Allocate( MemoryPoolSize, nn::audio::MemoryPoolType::AddressAlignment );
    g_MemoryPoolAllocator.Initialize( g_pMemoryForMemoryPool, MemoryPoolSize );
    nn::atk::SoundSystem::AttachMemoryPool( &g_MemoryPool, g_pMemoryForMemoryPool, MemoryPoolSize );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess );

    // SoundArchive の初期化
    isSuccess = g_SoundArchive.Open( SoundArchiveFilePath );
    NN_ABORT_UNLESS( isSuccess );

    // SoundArchive のパラメータ情報をメモリにロード
    size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = Allocate( infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess );

    // SoundDataManager の初期化
    size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize( &g_SoundArchive, g_pMemoryForSoundDataManager, memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess );

    // SoundArchivePlayer の初期化
    size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize );
    size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = AllocateFromMemoryPool( memSizeForStreamBuffer, nn::audio::BufferAlignSize );
    isSuccess = g_SoundArchivePlayer.Initialize( &g_SoundArchive, &g_SoundDataManager, g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer, g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess );

    // 文字列テーブルの初期化
    size_t memSizeForLabelString = g_SoundArchive.GetLabelStringDataSize();
    g_pMemoryForLabelString = Allocate(memSizeForLabelString);
    isSuccess = g_SoundArchive.LoadLabelStringData(g_pMemoryForLabelString, memSizeForLabelString);
    NN_ABORT_UNLESS( isSuccess );

    //  LoadData
    isSuccess = g_SoundDataManager.LoadData( SEQ_MARIOKART, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SEQ_MARIOKART) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_YOSHI, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_YOSHI) failed." );

    // AtkSpyModule の初期化
    nn::spy::atk::AtkSpyModule::InitializeArg atkSpyModuleInitializeArg;
    atkSpyModuleInitializeArg.isPerformanceMetricsEnabled = true;
    atkSpyModuleInitializeArg.isWaveformEnabled = true;
    atkSpyModuleInitializeArg.isAtkProfilesEnabled = true;
    size_t memorySizeForAtkSpyModule = nn::spy::atk::AtkSpyModule::GetRequiredMemorySize( atkSpyModuleInitializeArg );
    g_pMemoryForAtkSpyModule = Allocate( memorySizeForAtkSpyModule );
    g_AtkSpyModule.Initialize( atkSpyModuleInitializeArg, g_pMemoryForAtkSpyModule, memorySizeForAtkSpyModule );
    g_SpyController.RegisterModule( g_AtkSpyModule );

    {
        // Spy ツールと通信を開始
        nn::spy::SpyController::OpenArg openArg;
        bool openResult = g_SpyController.Open(openArg);
        NN_ABORT_UNLESS(openResult);
    }

    // OpusDecoder の初期化
    int decoderCount = 2;
    std::size_t memSizeForOpusDecoder = nn::atk::GetRequiredOpusDecoderBufferSize(decoderCount);
    g_pMemoryForOpusDecoder = Allocate(memSizeForOpusDecoder);
    nn::atk::InitializeOpusDecoder(g_pMemoryForOpusDecoder, memSizeForOpusDecoder, decoderCount);

    // 波形取得用のバッファを確保
    g_CircularBufferSinkBufferSize = nn::atk::SoundSystem::GetCircularBufferSinkBufferSize();
    g_CircularBufferSinkBuffer = g_Allocator.Allocate(g_CircularBufferSinkBufferSize);

    {
        g_EffectVolumeDown.SetAudioFrameCount(5);
        g_EffectVolumeDown.SetChannelCount(2);
        g_EffectVolumeDown.SetEnabled(true);

        size_t auxBufferSize = nn::atk::SoundSystem::GetRequiredEffectAuxBufferSize(&g_EffectVolumeDown);
        // エフェクト用のバッファはメモリプール管理されているヒープから確保する必要があります。
        g_pMemoryForEffectAux = AllocateFromMemoryPool(auxBufferSize, nn::audio::BufferAlignSize);
        NN_ABORT_UNLESS_NOT_NULL(g_pMemoryForEffectAux);

        isSuccess = nn::atk::SoundSystem::AppendEffect(nn::atk::AuxBus_A, &g_EffectVolumeDown, g_pMemoryForEffectAux, auxBufferSize);
        NNS_LOG("Aux    %d (bufferSize = %zu)\n", isSuccess, auxBufferSize);

        g_IsEffectClearing = false;
        g_IsEffectChanging = false;
    }
}

//  終了処理です
void Finalize()
{
    // エフェクトの解放処理
    if (!g_IsEffectClearing)
    {
        g_IsEffectClearing = PrepareEffectRemove();
    }
    if (g_IsEffectClearing)
    {
        // エフェクトが削除可能になるまで待つ
        for (;;)
        {
            if (CheckEffectRemovable())
            {
                ClearEffect();
                g_IsEffectClearing = false;
                break;
            }
        }
    }

    Free( g_CircularBufferSinkBuffer );

    g_SpyController.Close();

    g_SpyController.GetPlotModule().DetachItem(g_TaskThreadState);
    g_SpyController.GetPlotModule().DetachItem(g_SoundThreadState);
    g_SpyController.GetPlotModule().DetachItem(g_SoundThreadDetailState);

    g_SpyController.UnregisterModule( g_AtkSpyModule );
    g_AtkSpyModule.Finalize();

    Free( g_pMemoryForAtkSpyModule );

    g_SoundArchivePlayer.Finalize();
    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);
    nn::atk::FinalizeOpusDecoder();
    nn::atk::SoundSystem::Finalize();

    Free(g_pMemoryForOpusDecoder);
    Free( g_pMemoryForLabelString );
    FreeToMemoryPool( g_pMemoryForStreamBuffer );
    Free( g_pMemoryForSoundArchivePlayer );
    Free( g_pMemoryForSoundDataManager );
    Free( g_pMemoryForInfoBlock );
    Free( g_pMemoryForSoundHeap );
    g_MemoryPoolAllocator.Finalize();
    Free( g_pMemoryForMemoryPool );
    Free( g_pMemoryForSoundSystem );

    g_SpyController.Finalize();

#if defined(NN_BUILD_CONFIG_HTC_ENABLED)
    nn::htcs::Finalize();
#endif

    Free( g_pMemoryForSpyController );

    //  atk 以外の終了処理
    nn::fs::Unmount( MOUNT_NAME );
    Free( g_MountRomCacheBuffer );

    g_Allocator.Finalize();
}

//  使い方を表示します
void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]                StartSound SEQ  (SEQ_MARIOKART)\n");
    NNS_LOG("[X]                StartSound WSD  (SE_YOSHI)\n");
    NNS_LOG("[Y]                StartSound STRM (STRM_MARIOKART)\n");
    NNS_LOG("[B]                Stop Sound\n");
    NNS_LOG("[L]                Print Usage\n");
    NNS_LOG("[R]                Enable / Disable Push\n");
    NNS_LOG("[Up] + [+/Start][Space]   Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

//  サウンドを再生します
void PlayWithStartSound(nn::atk::SoundArchive::ItemId soundId, const char* label, int index)
{
    g_SoundHandle[index].Stop( 0 );

    bool result = g_SoundArchivePlayer.StartSound( &g_SoundHandle[index], soundId ).IsSuccess();
    NNS_LOG("StartSound(%s) ... (%d)\n", label, result);
}

//  更新します
void Update(const nn::hid::DebugPadButtonSet& button)
{
    if ( button.Test<nn::hid::DebugPadButton::A>() )
    {
        PlayWithStartSound(SEQ_MARIOKART, "SEQ_MARIOKART", 0);
    }
    if ( button.Test<nn::hid::DebugPadButton::X>() )
    {
        PlayWithStartSound( SE_YOSHI, "SE_YOSHI", 0 );
    }
    if ( button.Test<nn::hid::DebugPadButton::Y>() )
    {
        PlayWithStartSound(STRM_MARIOKART, "STRM_MARIOKART", 0);
        PlayWithStartSound(STRM_MARIOKART_OPUS, "STRM_MARIOKART_OPUS", 1);
    }
    if ( button.Test<nn::hid::DebugPadButton::B>() )
    {
        for (int i = 0; i < SoundHandleCount; i++)
        {
            g_SoundHandle[i].Stop(0);
        }
    }
    if ( button.Test<nn::hid::DebugPadButton::L>() )
    {
        PrintUsage();
    }
    if ( button.Test<nn::hid::DebugPadButton::R>() )
    {
        NNS_LOG("nn::spy::atk::AtkSpyModule::SetLogEnabled(%s)\n", !g_AtkSpyModule.IsLogEnabled() ? "true" : "false");
        g_AtkSpyModule.SetLogEnabled(!g_AtkSpyModule.IsLogEnabled());

        NNS_LOG("nn::spy::atk::AtkSpyModule::SetSoundStateEnabled(%s)\n", !g_AtkSpyModule.IsSoundStateEnabled() ? "true" : "false");
        g_AtkSpyModule.SetSoundStateEnabled(!g_AtkSpyModule.IsSoundStateEnabled());

        NNS_LOG("nn::spy::atk::AtkSpyModule::SetPerformanceMetricsEnabled(%s)\n", !g_AtkSpyModule.IsPerformanceMetricsEnabled() ? "true" : "false");
        g_AtkSpyModule.SetPerformanceMetricsEnabled(!g_AtkSpyModule.IsPerformanceMetricsEnabled());
    }

    g_SoundArchivePlayer.Update();
}

//  AtkSpyModule のプロファイラ機能を試します
void AtkSoundProfile()
{
    Initialize();

    PrintUsage();

    for(;;)
    {
        g_SpyController.SetCurrentApplicationFrame(g_SpyController.GetCurrentApplicationFrame() + 1);

        SpySandbox::UpdatePadStatus();
        const nn::hid::DebugPadButtonSet button = SpySandbox::GetButtonDown();
        const nn::hid::DebugPadButtonSet buttonState = SpySandbox::GetButtonState();
        Update( button );

        //  終了判定
        if( buttonState.Test<nn::hid::DebugPadButton::Start>() &&
            buttonState.Test<nn::hid::DebugPadButton::Up>() )
        {
            break;
        }

        g_AtkSpyModule.PushSoundState(g_SoundArchivePlayer);
        g_AtkSpyModule.PushSequenceVariable(g_SoundArchivePlayer);
        g_AtkSpyModule.PushPerformanceMetrics();
        g_AtkSpyModule.PushStreamSoundInfo(g_SoundArchivePlayer);

        if (g_AtkSpyModule.IsWaveformEnabled())
        {
            // nn::atk は nn::audio::RequestUpdateAudioRenderer() の
            // 完了時刻を提供しないので、波形データ取得の直前の時刻で代用します。
            nn::os::Tick tick = nn::os::GetSystemTick();

            // 波形データを取得します。
            size_t readBytes = nn::atk::SoundSystem::ReadCircularBufferSink(
                g_CircularBufferSinkBuffer,
                g_CircularBufferSinkBufferSize);

            // 波形データを送信します。
            g_AtkSpyModule.PushWaveform(
                g_CircularBufferSinkBuffer,
                readBytes,
                tick);
        }

        g_AtkSpyModule.PushAtkProfiles(g_SoundArchivePlayer);

        // データバッファ使用量を送ります
        g_SpyController.GetDebugModule().PushDataBufferUsage();

        //  Vsync の代わり
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 16 ) );
    }

    Finalize();
}
