﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{AtkSound3D.cpp,PageSampleAtkSound3D}
 *
 * @brief
 *  nn::atk ライブラリの Sound3D の機能を示すサンプルプログラム
 */

/**
 * @page PageSampleAtkSound3D Sound3D を用いた再生
 * @tableofcontents
 *
 * @brief
 *  nn::atk ライブラリの Sound3D に関する機能のサンプルプログラムの解説です。
 *
 * @section PageSampleAtkSound3D_SectionBrief 概要
 *  Atk ライブラリの Sound3D を用いて音声を再生するサンプルです。
 *  ドップラー効果や 3D Pan などの立体的な空間での音響を再現します。
 *
 * @section PageSampleAtkSound3D_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/AtkSound3D Samples/Sources/Applications/AtkSound3D @endlink 以下にあります。
 *
 * @section PageSampleAtkSound3D_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleAtkSound3D_SectionHowToOperate 操作方法
 *  コンソールに操作方法が表示されます。
 *
 * @section PageSampleAtkSound3D_SectionPrecaution 注意事項
 *  このサンプルプログラムは画面上に何も表示されません。実行中のログはコンソールに出力されます。
 *
 * @section PageSampleAtkSound3D_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleAtkSound3D_SectionDetail 解説
 *
 * @subsection PageSampleAtkSound3D_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  AtkSound3D.cpp
 *  @includelineno AtkSound3D.cpp
 *
 * @subsection PageSampleAtkSound3D_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの Atk に関連している処理の流れは以下の通りです（AtkSimple と違う部分には追加要素と書かれています）。
 *
 *  - Atk ライブラリの初期化 (InitializeAtk())
 *    - SoundArchivePlayer を除く AtkSimple で実行している初期化処理
 *    - Sound3DManager の初期化（追加要素）
 *    - SoundArchivePlayer の初期化
 *    - Sound3DListener の初期化（追加要素）
 *    - Sound3DActor の初期化（追加要素）
 *  - SoundHeap にウェーブサウンドの再生に必要な情報をロード (LoadData())
 *  - メインループでの処理
 *    - キー操作による処理
 *      - サウンドの再生・停止処理を実行
 *      - 適用する 3D フィルターの種類を変更（追加要素）
 *      - Sound3DActor の移動（追加要素）
 *    - Sound3DActor の位置を更新（追加要素）
 *    - SoundArchivePlayer の更新
 *  - Atk ライブラリの終了処理 (FinalizeAtk())
 *
 *  キー操作を行うことで、Sound3DActor を移動させることができます。
 *  Sound3DActor を移動させることで、立体的な音響の変化を確認することができます。
 *  立体的な音響の変化は SoundMaker で設定されたパラメータによって調整することができます。
 *  このサンプルではドップラー効果の ON/OFF, フィルター処理の ON/OFF をそれぞれ設定した
 *  4 つのサウンドを再生することができます。
 *
 */

#include "Common.fsid"

#include <nns/nns_Log.h>
#include <nn/atk.h>
#include <nn/htcs.h>
#include <nn/spy.h>

#include <nn/util/util_Matrix.h>
#include <nn/util/util_Arithmetic.h>

#include <nns/atk/atk_SampleCommon.h>

#include "SpySampleBModule.h"

namespace
{
    const char Title[] = "AtkSound3D";
    const char ArchiveRelativePath[] = "Common.bfsar";

    const int SoundHeapSize = 4 * 1024 * 1024;
    const size_t SpyControllerDataBufferLength = 1024 * 1024;

    nn::atk::SoundHeap          g_SoundHeap;
    nn::atk::FsSoundArchive     g_SoundArchive;
    nn::atk::SoundArchivePlayer g_SoundArchivePlayer;
    nn::atk::SoundDataManager   g_SoundDataManager;

    nn::audio::MemoryPoolType   g_MemoryPool;

    nn::spy::SpyController g_SpyController;
    SpySampleBModule g_SpySampleBModule;

    void* g_pMemoryForSoundSystem;
    void* g_pMemoryForSoundHeap;
    void* g_pMemoryForInfoBlock;
    void* g_pMemoryForSoundDataManager;
    void* g_pMemoryForSoundArchivePlayer;
    void* g_pMemoryForStreamBuffer;
    void* g_WorkBufferForSpyController = nullptr;

    int g_AppFrameCount = 0;


    nn::atk::SoundHandle        g_SoundHandle;

    //  Sound3D 用の変数
    nn::atk::Sound3DManager g_Sound3DManager;
    NN_ALIGNAS(16) nn::atk::Sound3DListener g_Sound3DListener;
    NN_ALIGNAS(16) nn::atk::Sound3DActor g_Sound3DActor;

    const float ControlPadStep = 0.2f;
    NN_ALIGNAS(16) nn::util::Vector3fType g_ActorPos;
    int g_FilterType;
    void* g_pMemoryForSound3DManager;
}

void Initialize()
{
    nns::atk::InitializeHeap();
    nns::atk::InitializeFileSystem();
    nns::atk::InitializeHidDevices();
}

void Finalize()
{
    nns::atk::FinalizeHidDevices();
    nns::atk::FinalizeFileSystem();
    nns::atk::FinalizeHeap();
}

void InitializeHtcs()
{
    nn::htcs::Initialize(nns::atk::Allocate, nns::atk::Free);
}

void FinalizeHtcs()
{
    nn::htcs::Finalize();
}

void InitializeSpy()
{
    // nn::spy::SpyController を初期化します。
    nn::spy::SpyController::InitializeArg initializeArg;
    initializeArg.dataBufferLength = SpyControllerDataBufferLength;

    size_t bufferSize = nn::spy::SpyController::GetRequiredMemorySize(initializeArg);
    g_WorkBufferForSpyController = nns::atk::Allocate(bufferSize);
    NN_ABORT_UNLESS(bufferSize == 0 || g_WorkBufferForSpyController != nullptr);

    g_SpyController.Initialize(initializeArg, g_WorkBufferForSpyController, bufferSize);

    g_SpyController.RegisterModule(g_SpySampleBModule);

    g_AppFrameCount = 0;
}

void FinalizeSpy()
{
    g_SpyController.UnregisterModule(g_SpySampleBModule);

    g_SpyController.Finalize();

    if (g_WorkBufferForSpyController)
    {
        nns::atk::Free(g_WorkBufferForSpyController);
        g_WorkBufferForSpyController = nullptr;
    }
}

// Spy ツールと通信を開始します。
void StartSpy()
{
    nn::spy::SpyController::OpenArg openArg;
    bool openResult = g_SpyController.Open(openArg);
    NN_ABORT_UNLESS(openResult);
}

// Spy ツールとの通信を終了します。
void StopSpy()
{
    g_SpyController.Close();
}

void CalcListenerMatrix( nn::util::Matrix4x3fType* mtx )
{
    // リスナーの位置
    nn::util::Vector3fType pos;
    nn::util::VectorSet( &pos, 0.0f, 0.0f, -3.0f );

    // リスナーの Up 方向のベクトル
    nn::util::Vector3fType upVec;
    nn::util::VectorSet( &upVec, 0.0f, 1.0f, 0.0f );

    // リスナーの方向ベクトル
    nn::util::Vector3fType direction;
    float directionSin;
    float directionCos;
    nn::util::SinCosEst( &directionSin, &directionCos, 0.0f );
    nn::util::VectorSet( &direction, -directionSin, 0.0f, -directionCos );

    // リスナー行列生成
    nn::util::Vector3fType target;
    nn::util::VectorAdd( &target, pos, direction );
    nn::util::MatrixLookAtRightHanded( mtx, pos, target, upVec );
}

void InitializeAtk()
{
    bool isSuccess = true;

    // SoundSystem の初期化
    nn::atk::SoundSystem::SoundSystemParam param;
    std::size_t memSizeForSoundSystem = nn::atk::SoundSystem::GetRequiredMemSize( param );
    g_pMemoryForSoundSystem = nns::atk::Allocate( memSizeForSoundSystem, nn::atk::SoundSystem::WorkMemoryAlignSize );
    isSuccess = nn::atk::SoundSystem::Initialize(
        param,
        reinterpret_cast<uintptr_t>( g_pMemoryForSoundSystem ),
        memSizeForSoundSystem );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundSystem" );

    // SoundHeap の初期化
    g_pMemoryForSoundHeap = nns::atk::Allocate( SoundHeapSize );
    isSuccess = g_SoundHeap.Create( g_pMemoryForSoundHeap, SoundHeapSize );
    NN_ABORT_UNLESS( isSuccess, "cannot create SoundHeap" );

    // SoundArchive の初期化
    const char* archiveAbsolutePath = nns::atk::GetAbsolutePath(ArchiveRelativePath);
    isSuccess = g_SoundArchive.Open(archiveAbsolutePath);
    NN_ABORT_UNLESS( isSuccess, "cannot open SoundArchive(%s)\n", archiveAbsolutePath );

    // SoundArchive のパラメータ情報をメモリにロード
    std::size_t infoBlockSize = g_SoundArchive.GetHeaderSize();
    g_pMemoryForInfoBlock = nns::atk::Allocate(  infoBlockSize, nn::atk::FsSoundArchive::BufferAlignSize );
    isSuccess = g_SoundArchive.LoadHeader( g_pMemoryForInfoBlock, infoBlockSize );
    NN_ABORT_UNLESS( isSuccess, "cannot load InfoBlock" );

    // SoundDataManager の初期化
    std::size_t memSizeForSoundDataManager = g_SoundDataManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundDataManager = nns::atk::Allocate( memSizeForSoundDataManager, nn::atk::SoundDataManager::BufferAlignSize );
    isSuccess = g_SoundDataManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSoundDataManager,
        memSizeForSoundDataManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundDataManager" );

    // Sound3DManager の初期化
    std::size_t memSizeForSound3DManager = g_Sound3DManager.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSound3DManager = nns::atk::Allocate( memSizeForSound3DManager );
    isSuccess = g_Sound3DManager.Initialize(
        &g_SoundArchive,
        g_pMemoryForSound3DManager,
        memSizeForSound3DManager );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize Sound3DManager" );

    g_FilterType= nn::atk::BiquadFilterType_None;
    g_Sound3DManager.SetBiquadFilterType( g_FilterType );
    g_Sound3DManager.SetMaxPriorityReduction( 32 );
    g_Sound3DManager.SetSonicVelocity( 340.0f / 60.0f );

    // SoundArchivePlayer で用いるストリームバッファの初期化
    // ストリームバッファはメモリプール管理されているヒープから確保する必要があります。
    std::size_t memSizeForStreamBuffer = g_SoundArchivePlayer.GetRequiredStreamBufferSize( &g_SoundArchive );
    g_pMemoryForStreamBuffer = nns::atk::AllocateForMemoryPool(memSizeForStreamBuffer);

    // 専用のヒープをメモリプールにアタッチ
    nn::atk::SoundSystem::AttachMemoryPool(&g_MemoryPool, nns::atk::GetPoolHeapAddress(), nns::atk::GetPoolHeapSize());

    // SoundArchivePlayer の初期化
    std::size_t memSizeForSoundArchivePlayer = g_SoundArchivePlayer.GetRequiredMemSize( &g_SoundArchive );
    g_pMemoryForSoundArchivePlayer = nns::atk::Allocate( memSizeForSoundArchivePlayer, nn::atk::SoundArchivePlayer::BufferAlignSize);
    isSuccess = g_SoundArchivePlayer.Initialize(
        &g_SoundArchive,
        &g_SoundDataManager,
        g_pMemoryForSoundArchivePlayer, memSizeForSoundArchivePlayer,
        g_pMemoryForStreamBuffer, memSizeForStreamBuffer );
    NN_ABORT_UNLESS( isSuccess, "cannot initialize SoundArchivePlayer" );

    // Sound3DListener の初期化
    nn::util::Matrix4x3fType listenerMtx;
    nn::util::Vector3fType zeroVelocity;
    CalcListenerMatrix( &listenerMtx );
    nn::util::VectorSet( &zeroVelocity, 0.0f, 0.0f, 0.0f );

    g_Sound3DListener.SetMatrix( listenerMtx );
    g_Sound3DListener.SetVelocity( zeroVelocity );
    g_Sound3DListener.SetMaxVolumeDistance( 5.0f );
    g_Sound3DListener.SetUnitDistance( 5.0f );
    g_Sound3DListener.SetInteriorSize( 5.0f );
    g_Sound3DManager.AddListener( &g_Sound3DListener );

    // Sound3DActor の初期化
    g_Sound3DActor.Initialize( &g_SoundArchivePlayer, &g_Sound3DManager );
    nn::util::VectorSet( &g_ActorPos, 0.0f, 0.0f, 0.0f );
    g_Sound3DActor.SetPosition( g_ActorPos );
    g_Sound3DActor.SetVelocity( zeroVelocity );
}

void FinalizeAtk()
{
    g_Sound3DActor.Finalize();
    //g_Sound3DListener の終了処理は不要

    g_SoundArchivePlayer.Finalize();

    g_Sound3DManager.Finalize();

    // 専用のヒープをメモリプールからデタッチ
    nn::atk::SoundSystem::DetachMemoryPool(&g_MemoryPool);

    g_SoundDataManager.Finalize();
    g_SoundArchive.Close();
    g_SoundHeap.Destroy();
    nn::atk::SoundSystem::Finalize();

    nns::atk::Free(g_pMemoryForSound3DManager);

    nns::atk::FreeForMemoryPool(g_pMemoryForStreamBuffer);
    nns::atk::Free(g_pMemoryForSoundArchivePlayer);
    nns::atk::Free(g_pMemoryForSoundDataManager);
    nns::atk::Free(g_pMemoryForInfoBlock);
    nns::atk::Free(g_pMemoryForSoundHeap);
    nns::atk::Free(g_pMemoryForSoundSystem);
}

void LoadData()
{
    bool isSuccess = true;

    isSuccess = g_SoundDataManager.LoadData( SE_IDLE, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_IDLE_DOPPLER, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE_DOPPLER) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_IDLE_FILTER, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE_FILTER) failed." );

    isSuccess = g_SoundDataManager.LoadData( SE_IDLE_DOPPLER_FILTER, &g_SoundHeap );
    NN_ABORT_UNLESS( isSuccess, "LoadData(SE_IDLE_DOPPLER_FILTER) failed." );
}

void StopSound()
{
    g_SoundHandle.Stop( 6 );
}

void PlaySound3D(nn::atk::SoundArchive::ItemId soundId, const char* debugLabelName)
{
    StopSound();
    bool result = g_Sound3DActor.StartSound( &g_SoundHandle, soundId ).IsSuccess();
    NNS_LOG("PlaySound(%s) ... (%d)\n", debugLabelName, result);
}

void PrintUsage()
{
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("%s Sample\n", Title);
    NNS_LOG("--------------------------------------------------\n");
    NNS_LOG("[A]              Play Sound (Doppler ON , Filter ON )\n");
    NNS_LOG("[B]              Play Sound (Doppler ON , Filter OFF)\n");
    NNS_LOG("[X]              Play Sound (Doppler OFF, Filter ON )\n");
    NNS_LOG("[Y]              Play Sound (Doppler OFF, Filter OFF)\n");
    NNS_LOG("[R+A]            Change Filter Type\n");
    NNS_LOG("[R+B]            Stop Sound\n");
    NNS_LOG("[R+X]            Reset Actor Position\n");
    NNS_LOG("[RIGHT/LEFT]     Move Actor Position.x\n");
    NNS_LOG("[UP/DOWN]        Move Actor Position.z\n");
    NNS_LOG("[L]              Print Usage\n");
    NNS_LOG("[Start][Space]   Exit Application\n");
    NNS_LOG("--------------------------------------------------\n");
}

const char* GetFilterTypeName( int type )
{
    const char* filterTypeString[]=
    {
        "NONE",
        "LPF"
    };
    if ( type < 0 || type >= sizeof(filterTypeString) / sizeof(filterTypeString[0]) )
    {
        return "INVALID";
    }
    return filterTypeString[type];
}

void PrintState()
{
    nn::util::Float3 actorPos;
    nn::util::VectorStore(&actorPos, g_ActorPos);

    NNS_LOG("ActorPos( %4.1f, %4.1f, %4.1f ), FilterType= %s\n", actorPos.v[0], actorPos.v[1], actorPos.v[2], GetFilterTypeName(g_FilterType));
}

void MoveActor( float x, float y, float z)
{
    nn::util::Vector3fType movement;
    nn::util::VectorSet( &movement, x, y, z );
    nn::util::VectorAdd( &g_ActorPos, g_ActorPos, movement );
    PrintState();
}

void ResetActorPosition()
{
    nn::util::VectorSet( &g_ActorPos, 0.0f, 0.0f, 0.0f );
    g_Sound3DActor.SetPosition( g_ActorPos );

    nn::util::Vector3fType zeroVelocity;
    nn::util::VectorSet( &zeroVelocity, 0.0f, 0.0f, 0.0f );
    g_Sound3DActor.SetVelocity( zeroVelocity );

    PrintState();
}

void ChangeFilterType()
{
    if ( g_FilterType == nn::atk::BiquadFilterType_None )
    {
        g_FilterType = nn::atk::BiquadFilterType_LowPassFilter;
    }
    else
    {
        g_FilterType = nn::atk::BiquadFilterType_None;
    }

    g_Sound3DManager.SetBiquadFilterType( g_FilterType );
    PrintState();
}

bool UpdateAtk()
{
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::R >() )
    {
        // FilterType の変更
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
        {
            ChangeFilterType();
        }

        // StopSound
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
        {
            StopSound();
        }

        // Sound3DActor の座標をリセット
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
        {
            ResetActorPosition();
        }
    }
    else
    {
        // PlaySound
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::A >() )
        {
            PlaySound3D( SE_IDLE_DOPPLER_FILTER, "SE_IDLE_DOPPLER_FILTER");
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::B >() )
        {
            PlaySound3D( SE_IDLE_DOPPLER, "SE_IDLE_DOPPLER");
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::X >() )
        {
            PlaySound3D( SE_IDLE_FILTER, "SE_IDLE_FILTER");
        }
        if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Y >() )
        {
            PlaySound3D( SE_IDLE, "SE_IDLE");
        }
    }

    // Sound3DActor の座標変更
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Left >() )
    {
        MoveActor( -ControlPadStep, 0.0f, 0.0f );
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Right >() )
    {
        MoveActor(  ControlPadStep, 0.0f, 0.0f );
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Up >() )
    {
        MoveActor( 0.0f, 0.0f, -ControlPadStep );
    }
    if ( nns::atk::IsHold< ::nn::hid::DebugPadButton::Down >() )
    {
        MoveActor( 0.0f, 0.0f,  ControlPadStep );
    }

    // Print Usage
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::L >() )
    {
        PrintUsage();
    }

    // Exit
    if ( nns::atk::IsTrigger< ::nn::hid::DebugPadButton::Start >() )
    {
        return false;
    }

    g_Sound3DActor.SetPosition( g_ActorPos );
    g_SoundArchivePlayer.Update();

    return true;
}

extern "C" void nnMain()
{
    Initialize();
    InitializeHtcs();
    InitializeAtk();

    // Spy の初期化して開始します。
    InitializeSpy();
    StartSpy();

    LoadData();

    PrintUsage();
    PrintState();

    for ( ;; )
    {
        // アプリケーションフレームの開始時間を記録します。
        g_SpyController.SetCurrentApplicationFrame(g_AppFrameCount++);

        nns::atk::UpdateHidDevices();

        if (!UpdateAtk())
        {
            break;
        }

        nn::util::Float3 actorPos;
        nn::util::VectorStore(&actorPos, g_ActorPos);
        g_SpySampleBModule.PushActorPosition(actorPos.v[0], actorPos.v[2]);

        g_SpyController.GetDebugModule().PushDataBufferUsage();

        // Vsync の代わり
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // Spy を停止して、終了します。
    StopSpy();
    FinalizeSpy();

    FinalizeAtk();
    FinalizeHtcs();
    Finalize();
}

