﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include "MulticastReceive.h"

namespace nns {
namespace socket {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const size_t     MulticastReceive::ReadBufferSize = 4096;

//***************************************************************
//*
//* MulticastReceive
//*
//***************************************************************

MulticastReceive::MulticastReceive() NN_NOEXCEPT : m_ReceivedTotalBytes(0),
                                                   m_UdpRecvSocket(-1),
                                                   m_MulticastUdpPort(0),
                                                   m_pReceivedData(nullptr),
                                                   m_IsInitialized(false)
{
    memset(&m_MulitcastGroupAddress, 0, sizeof(m_MulitcastGroupAddress));
    memset(&m_MreqAddress, 0, sizeof(m_MreqAddress));
}

MulticastReceive::~MulticastReceive() NN_NOEXCEPT
{
    if (m_IsInitialized == true)
    {
        Finalize();
    }

    memset(&m_MulitcastGroupAddress, 0, sizeof(m_MulitcastGroupAddress));
    memset(&m_MreqAddress, 0, sizeof(m_MreqAddress));
    m_MulticastUdpPort = 0;
}

uint64_t MulticastReceive::GetTotalBytesRecieved()
{
    return(m_ReceivedTotalBytes);
}

int MulticastReceive::Initialize(const char* pMulitcastGroupAddress, const uint16_t multicastUdpPort, const char* pNetworkInterface) NN_NOEXCEPT
{
    nn::socket::SockAddrIn sockAddr = { 0 };
    nn::socket::TimeVal     myTime = { 0 };
    size_t             ipAddrSize = 0;
    int                udpSocket = -1;
    int                rc = 0;
    int                ret = 0;

    do
    {
        // if already initialized
        if (m_IsInitialized == true)
        {
            NN_LOG("Object is already initialized!\n");
            break;
        }

        // Did we get a Multicast Group Address?
        if (pMulitcastGroupAddress == nullptr)
        {
            NN_LOG("Multicast Group Address IP Address (char) is required\n");
            break;
        }

        ipAddrSize = strnlen(pMulitcastGroupAddress, (size_t) (sizeof(m_MulitcastGroupAddress) - 1));
        if (ipAddrSize < 1 || ipAddrSize >= sizeof(m_MulitcastGroupAddress) - 1)
        {
            NN_LOG("Multicast Group Address IP Address (size_t) has illegal size [%ld]\n", ipAddrSize);
            break;
        }

        // Save Multicast Group Address
        memset(m_MulitcastGroupAddress, 0, sizeof(m_MulitcastGroupAddress));
        memcpy(m_MulitcastGroupAddress, pMulitcastGroupAddress, ipAddrSize);

        // Initialize Socket Addresses
        memset(&sockAddr, 0, sizeof(sockAddr));
        memset(&m_MreqAddress, 0, sizeof(m_MreqAddress));

        // Save the Multicast Group Port
        m_MulticastUdpPort = multicastUdpPort;

        // Create a UDP Socket
        udpSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Ip);
        if(udpSocket < 0)
        {
            NN_LOG("nn::socket::Socket(): Failed - (errno: %d)\n", nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // Establish UDP Source Address
        sockAddr.sin_family      = nn::socket::Family::Af_Inet;
        sockAddr.sin_port        = nn::socket::InetHtons(m_MulticastUdpPort);
        sockAddr.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);

        // Socket read calls should block up to 1 second
        myTime.tv_sec  = 1;
        myTime.tv_usec = 0;

        // UDP Socket - Don't block forever on read() calls
        rc = nn::socket::SetSockOpt(udpSocket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvTimeo, (void *) &myTime, sizeof(myTime));
        if (rc < 0)
        {
            NN_LOG( "nn::socket::SetSockOpt(): (nn::socket::Option::So_RcvTimeo) - (errno: %d)\n", nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // Bind UDP Socket
        rc = nn::socket::Bind(udpSocket, (struct sockaddr *) &sockAddr, sizeof(sockAddr));
        if (rc < 0)
        {
            NN_LOG( "nn::socket::Bind(): Failed (errno: %d)\n", nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // Set Multicast Group IP Address
        rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, m_MulitcastGroupAddress, &m_MreqAddress.imr_multiaddr.S_addr);
        if (rc != 1)
        {
            NN_LOG( "nn::socket::InetPton(): Failed to set Multicast Group Address: [%s] (errno: %d)\n",
                                               m_MulitcastGroupAddress, nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // Set the Multicast Group outbound Interface
        if (pNetworkInterface != nullptr)
        {
            // Translate the Multicast IP Network Interface address
            rc = nn::socket::InetPton(nn::socket::Family::Af_Inet, pNetworkInterface, &m_MreqAddress.imr_interface.S_addr);
            if (rc != 1)
            {
                NN_LOG( "nn::socket::InetPton(): Failed to set Multicast IP Network Interface address: [%s] (errno: %d)\n",
                                                  pNetworkInterface, nn::socket::GetLastError() );
                ret = -1;
                break;
            }
        }
        else
        {
            m_MreqAddress.imr_interface.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
        }

        // JOIN the Multicast Group
        rc = nn::socket::SetSockOpt(udpSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Add_Membership, &m_MreqAddress, sizeof(m_MreqAddress));
        if (rc < 0)
        {
            NN_LOG( "nn::socket::SetSockOpt: (nn::socket::Option::Ip_Add_Membership) failed - (errno: %d)\n", nn::socket::GetLastError() );
            ret = -1;
            break;
        }

        // Object has been initialized
        m_IsInitialized = true;

        // Allocate Receive Buffer
        m_pReceivedData = new char[ReadBufferSize];
        if (m_pReceivedData == nullptr)
        {
            NN_LOG( "new failed - allocating [%d] bytes of memory!\n", ReadBufferSize);
            ret = -1;
            break;
        }

        // Initialize the Receive Buffer memory
        memset(m_pReceivedData, 0, ReadBufferSize);

        // Save created socket
        m_UdpRecvSocket = udpSocket;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (ret != 0)
    {
        if (m_IsInitialized == true)
        {
            rc = nn::socket::SetSockOpt(udpSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Drop_Membership, &m_MreqAddress, sizeof(m_MreqAddress));
            if (rc < 0)
            {
                NN_LOG( "nn::socket::SetSockOpt: (nn::socket::Option::Ip_Drop_Membership) failed - (errno: %d)\n", nn::socket::GetLastError() );
                // Fall thru
            }
        }

        if (udpSocket > -1)
        {
            nn::socket::Close(udpSocket);
            udpSocket = -1;
            m_UdpRecvSocket = -1;
        }

        if (m_pReceivedData != nullptr)
        {
            delete [] m_pReceivedData;
            m_pReceivedData = nullptr;
        }

        // NOT initialized
        m_IsInitialized = false;
    }

    return(ret);

}    // NOLINT(impl/function_size)

int MulticastReceive::Finalize() NN_NOEXCEPT
{
    int   ret = 0;

    if (m_IsInitialized == true)
    {
        if (m_UdpRecvSocket > -1)
        {
            // DROP from the Multicast Group
            ret = nn::socket::SetSockOpt(m_UdpRecvSocket, nn::socket::Level::Sol_Ip, nn::socket::Option::Ip_Drop_Membership, &m_MreqAddress, sizeof(m_MreqAddress));
            if (ret < 0)
            {
                NN_LOG( "nn::socket::SetSockOpt: (nn::socket::Option::Ip_Drop_Membership) failed - (errno: %d)\n", nn::socket::GetLastError() );
                // Fall thru
            }

            // Close the Socket
            nn::socket::Close(m_UdpRecvSocket);
            m_UdpRecvSocket = -1;
        }

        if (m_pReceivedData != nullptr)
        {
            delete [] m_pReceivedData;
            m_pReceivedData = nullptr;
        }
        m_IsInitialized = false;
    }

    return(ret);
}

int MulticastReceive::ReceiveMessages(const int inNumOfSeconds) NN_NOEXCEPT
{
    nn::socket::SockAddrIn  recvAddr = { 0 };
    nn::socket::SockLenT    recvAddrLen = 0;
    ssize_t              rc1 = -1;
    int                  ret = 0;
    nn::socket::Errno    myError = nn::socket::Errno::ESuccess;
    int                  timeLeft = -1;

    do
    {
        if (m_IsInitialized == false)
        {
            NN_LOG("Object is not initialized!\n");
            ret = -1;
            break;
        }

        NN_LOG( "[BEGIN] - Reading UDP packets from Multicast Group [%s:%d] for the next [%d] seconds\n",
                m_MulitcastGroupAddress, m_MulticastUdpPort, inNumOfSeconds);

        for(timeLeft = inNumOfSeconds; timeLeft > 0; timeLeft--)
        {
            // Read from Multicast Group
            recvAddrLen = sizeof(recvAddr);
            rc1 = nn::socket::RecvFrom(m_UdpRecvSocket, m_pReceivedData, ReadBufferSize, nn::socket::MsgFlag::Msg_None, (nn::socket::SockAddr *) &recvAddr, &recvAddrLen);
            if (rc1 < 0)
            {
                myError = nn::socket::GetLastError();
                if (myError == nn::socket::Errno::EAgain || myError == nn::socket::Errno::EWouldBlock)
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
                    continue;
                }

                NN_LOG( "Recvfrom: Failed to Read UPNP Response (Multicast Recieve) - Errno: %d\n", myError );
                ret = -1;
                break;
            }

            // Save total bytes
            m_ReceivedTotalBytes = m_ReceivedTotalBytes + rc1;

            // Display data read from multicast group
            NN_LOG( "Multicast Group [%s:%d] RecvFrom: Source IP: [%s:%d] - Len [%lu], Data [%.*s]\n",
                    m_MulitcastGroupAddress, m_MulticastUdpPort,
                    nn::socket::InetNtoa(recvAddr.sin_addr), nn::socket::InetNtohs(recvAddr.sin_port ),
                    rc1, rc1, m_pReceivedData);
        }

        NN_LOG( "[END] - Finished Reading UDP packets from Multicast Group [%s:%d]\n",
                m_MulitcastGroupAddress, m_MulticastUdpPort);

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

}}   // Namespace nns / socket
