﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include "MulticastReceive.h"
#include "MulticastSend.h"

/**
 * @examplesource{MulticastBasic.cpp,PageSampleMulticastBasic}
 *
 * @brief
 * Send and receive UDP datagrams from an IP multicast group
 */

/**
 * @page PageSampleMulticastBasic MulticastBasic
 * @tableofcontents
 *
 * @brief
 *  Documentation about the Sample program which will send and receive UDP datagrams to an IP multicast group
 *
 * @section PageSampleMulticastBasic_SectionBrief Overview
 *  First this sample initializes network connectivity by using the NIFM library.  Then the sample will initialize
 *  the Socket library.  If the sample was called with the '-receive' command line option then the sample will receieve UDP
 *  datagrams from the IP multicast group for 60 seconds.  If the sample was called with the '-send' command line option then
 *  the sample will send UDP datagrams with the value "Hello!" to the IP multicast group for 60 seconds.  It is possible
 *  to receive (0 to many) UDP datagrams from the IP multicast group based on what network traffic is currently being
 *  sent to the multicast group.  When sending datagrams the sample will pause (1 second) between each send.
 *
 * @section PageSampleMulticastBasic_SectionFileStructure File Structure
 *  This sample program can be found at
 *  @link ../../../Samples/Sources/Applications/SocketMulticast/Basic Samples/Sources/Applications/SocketMulticast/Basic @endlink
 *
 * @section PageSampleMulticastBasic_SectionNecessaryEnvironment System Requirements
 *  It is required to import the network setting by SettingManager prior to running this sample.
 *  Please refer @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} for further information.
 *
 * @section PageSampleMulticastBasic_SectionHowToOperate Operation Procedure
 *  The sample can be run with either a '-receive' or '-send' command line option.  If no command line argument is specified then
 *  the sample will default to the "-send" command line option.  If the '-receive' command line option is selected then the
 *  sample will receive UDP datagrams from the IP multicast group for the next 60 seconds.  If the '-send' command line option is
 *  specified then the sample will send UDP datagrams to the IP multicast group for the next 60 seconds.
 *
 *  By "default" the sample will pick the first network interface defined to the host it is running on that is multicast capable
 *  to use for sending and receiving UDP datagrams to or from the IP multicast group.  However, it is possible that the host that the
 *  sample is running on has more than one multicast capable interface defined to it.  Use the "-interface" command line option
 *  to specify the IPv4 address of the network interface to send and receive UDP datagrams on.  For example if a network interface
 *  has been assigned the IPv4 address of "10.0.0.1" - you can specify the command line option as "-interface 10.0.0.1".
 *
 * @section PageSampleMulticastBasic_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleMulticastBasic_SectionHowToExecute Execution Procedure
 *  Build the Visual Studio Solution in the desired configuration and run it.
 *
 * @section PageSampleMulticastBasic_SectionDetail Detail
 *  Here is the sequence of this sample program.
 *
 *  - Sends a request to use the network to the NIFM library
 *  - Initializes the socket library
 *  - If the sample was called with the '-receive' command line option then the sample will join the IP multicast group.  The sample will then
 *    receieve UDP datagrams from the IP multicast group for the next 60 seconds.  NOTE: That if there are no UDP datagrams to
 *    recieve from the IP multcast group then no data will be received by the sample.
 *  - If the sample was called with the '-send' command line option then the sample will send UDP datagrams addressed to the IP multicast
 *    group for the next 60 seconds.  There is a 1 second pause between each UDP datagram sent to the IP multicast group.
 *  - Sample finishes.
 *
 *  Please set "IpMulticastUdpAddress" to a legal IPv4 multicast address value if you would like to send or recieve UDP
 *  datagrams to an IPv4 multicast group that is different from 224.0.0.150.  By default this sample will send and receive UDP
 *  datagrams to/from IPv4 multicast group 224.0.0.150.
 *
 *  Please set "IpMulticastUdpPort" to a legal IPv4 multicast UDP port value if you would like to send or recieve UDP
 *  datagrams to an IPv4 multicast group UDP port that is different from UDP port 9000.  By default this sample will send and
 *  receive UDP datagrams to/from IPv4 multicast UDP port 9000.
 *
 * @subsection PageSampleMulticastBasic_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  MulticastBasic.cpp
 *  @includelineno MulticastBasic.cpp

 * @subsection PageSampleMulticastBasic_SectionExpectedOutput Expected Output
 *
 * @verbinclude MulticastBasic_SendOutputExample.txt
 * @verbinclude MulticastBasic_ReceiveOutputExample.txt
 */

namespace nns {
namespace socket {

// Constants
const char*                          IpMulticastUdpAddress = "224.0.0.150";      //<! IP Address supporting UDP/IP Multicast group
const uint16_t                       IpMulticastUdpPort = 9000;                  //<! UDP port supporting UDP/IP Multicast group
const int                            MaximumNumberOfSecondsToRun = 60;           //<! How long the Multicast Sender and Receiver will run
const int                            TimeoutOnNicConfigurationInSec = 15;        //<! How long to wait for NIFM

// Socket Library
nn::socket::ConfigDefaultWithMemory  g_SocketConfigWithMemory;

// Nifm values
nn::socket::InAddr                       g_NifmIpv4Addr = { 0 };
nn::socket::InAddr                       g_NifmIpv4Mask = { 0 };
nn::socket::InAddr                       g_NifmIpv4Gateway = { 0 };
nn::socket::InAddr                       g_NifmIpv4Dns1 = { 0 };
nn::socket::InAddr                       g_NifmIpv4Dns2 = { 0 };
bool                                 g_IsNetworkInitialized = false;

// Multicast Receive
MulticastReceive                     g_MulticastReceive;
bool                                 g_isMulticastReceiveInitilized = false;

// Multicast Send
MulticastSend                        g_MulticastSend;
bool                                 g_isMulticastSendInitilized = false;

// ------------------------------------------------------------------------------------------------
// Setup network interface
// ------------------------------------------------------------------------------------------------
int SetupNetwork() NN_NOEXCEPT
{
    int            ret = 0;
    nn::Result     result = nn::ResultSuccess();
    uint32_t       timeoutSec = 0;
    bool           isSocketLibInitialized = false;
    bool           isNifmLibInitialized = false;

    do
    {
        // If already initialized
        if (g_IsNetworkInitialized == true)
        {
            NN_LOG("Network is already initialized!\n");
            break;
        }

        // Initialize NIFM
        result = nn::nifm::Initialize();
        if(result.IsFailure())
        {
            NN_LOG("Failed calling: 'nn::nifm::Initialize()'\n");
            ret = -1;
            break;
        }

        // Submit the NIFM Network Request
        nn::nifm::SubmitNetworkRequest();

        // Wait for NIFM to go online
        while(nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            if(timeoutSec++ > TimeoutOnNicConfigurationInSec)
            {
                NN_LOG("Failed to setup NIFM network interface.\n");
                ret = -1;
                break;
            }
        }

        // Did NIFM request timeout?
        if (ret < 0)
        {
            break;  // Yes
        }

        // Is the NIFM Network Available?
        if(nn::nifm::IsNetworkAvailable() == 0)
        {
            NN_LOG("Network is not available.\n");
            ret = -1;
            break;
        }
        isNifmLibInitialized = true;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &g_NifmIpv4Addr, &g_NifmIpv4Mask, &g_NifmIpv4Gateway, &g_NifmIpv4Dns1, &g_NifmIpv4Dns2 );

        NN_LOG( "Network Interface is UP!\n" );
        NN_LOG( "NIFM Address Ipv4      : %s\n", nn::socket::InetNtoa(g_NifmIpv4Addr));
        NN_LOG( "NIFM Network Mask Ipv4 : %s\n", nn::socket::InetNtoa(g_NifmIpv4Mask));
        NN_LOG( "NIFM Gateway Ipv4      : %s\n", nn::socket::InetNtoa(g_NifmIpv4Gateway));
        NN_LOG( "NIFM DNS1 Ipv4         : %s\n", nn::socket::InetNtoa(g_NifmIpv4Dns1));
        NN_LOG( "NIFM DNS2 Ipv4         : %s\n", nn::socket::InetNtoa(g_NifmIpv4Dns2));

        // Initialize the socket library
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize socket library.\n");
            ret = -1;
            break;
        }
        isSocketLibInitialized = true;

        // Successfully initialized the network
        g_IsNetworkInitialized = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(true));

    // If failed - disable network
    if (ret != 0)
    {
        if (isSocketLibInitialized == true)
        {
            nn::socket::Finalize();
            isSocketLibInitialized  = false;
            ret = -1;
        }

        if (isNifmLibInitialized == true)
        {
            nn::nifm::CancelNetworkRequest();
            isNifmLibInitialized = false;
            ret = -1;
        }
        g_IsNetworkInitialized = false;

    }

    return(ret);

}  // NOLINT(impl/function_size)

// ------------------------------------------------------------------------------------------------
// Finalize Network
// ------------------------------------------------------------------------------------------------
int FinalizeNetwork() NN_NOEXCEPT
{
    nn::Result     result = nn::ResultSuccess();
    int            ret = 0;

    if (g_IsNetworkInitialized == true)
    {
        result = nn::socket::Finalize();
        if (result.IsFailure())
        {
            NN_LOG("Failed calling 'nn::socket::Finalize()'!\n");
        }

        nn::nifm::CancelNetworkRequest();
        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Parse Arguments
// ------------------------------------------------------------------------------------------------
bool ParseArgs(bool* pOutIsReceiver, char** pOutInterface) NN_NOEXCEPT
{
    size_t   ifaceSize = 0;
    int      options = 0;
    int      argc = nn::os::GetHostArgc();
    char**   argv = nn::os::GetHostArgv();
    bool     ret = true;

    // Initialize passed variables
    *pOutIsReceiver = false;
    *pOutInterface  = nullptr;

    for(options = 0; options < argc; options++)
    {
        if(strcmp(argv[options], "-receive" ) == 0)   // Receiver
        {
            *pOutIsReceiver = true;
            continue;
        }
        else if(strcmp(argv[options], "-send" ) == 0)  // Sender
        {
            *pOutIsReceiver = false;
            continue;
        }
        else if(strcmp(argv[options], "-interface" ) == 0)  // Optional: Network Interface
        {
            options++;
            if(options >= argc)
            {
                NN_LOG( "Arg: -interface requires an additional option, which is an IPv4 address assigned to this host\n");
                ret = false;
                break;
            }

            // Figure out size of the network Interface
            ifaceSize = strlen(argv[options]) + 1;

            // Allocate Memory to hold network Interface
            *pOutInterface = new char[ifaceSize];
            if (*pOutInterface == nullptr)
            {
                NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", argv[options], ifaceSize);
                ret = false;
                break;
            }

            // Copy the network Interface
            memset(*pOutInterface, 0, ifaceSize);
            memcpy(*pOutInterface, argv[options], strlen(argv[options]));
            continue;
        }
    }

    if (ret == false)
    {
        if (*pOutInterface != nullptr)
        {
            delete [] *pOutInterface;
            *pOutInterface = nullptr;
        }
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Main Loop
// ------------------------------------------------------------------------------------------------
void mainLoop() NN_NOEXCEPT
{
    int      ret = 0;
    bool     isReceiver = true;
    char*    pInterface = nullptr;

    NN_LOG("-- SocketMulticastBasic --\n");

    do
    {
       // Determine if sample is going to send or recceive
       if (ParseArgs(&isReceiver, &pInterface) == false)
        {
            NN_LOG("\nError: Invalid arguments.\n");
            NN_LOG("\nUsage:\n");
            NN_LOG("  -receive                  - Receive IP Multicast UDP datagrams\n");
            NN_LOG("  -send                     - Send IP Multicast UDP daragrams\n");
            NN_LOG("  -interface <ipv4 address> - Network interface to send or receive multicast UDP datagrams on\n");
            break;
        }

        // Setup network interface
        ret = SetupNetwork();
        if (ret != 0)
        {
            NN_LOG("Failed to enable network interface\n");
            break;
        }

        // If sample is going to receive
        if (isReceiver == true)
        {
            NN_LOG( "Sample will receive Multicast Group messages using UDP from [%s:%d] on network interface [%s]\n",
                                          IpMulticastUdpAddress, IpMulticastUdpPort,
                                          (pInterface != nullptr ? pInterface : "default" ));

            // Initialize Multicast Receive
            ret = g_MulticastReceive.Initialize(IpMulticastUdpAddress, IpMulticastUdpPort, pInterface);
            if (ret != 0)
            {
                NN_LOG("Failing calling 'MulticastReceive::Initialize()!\n");
                break;
            }
            g_isMulticastReceiveInitilized = true;

            // Begin Receiving Multicast Messages
            ret = g_MulticastReceive.ReceiveMessages(MaximumNumberOfSecondsToRun);
            if (ret != 0)
            {
                NN_LOG("Failing calling 'MulticastReceive::ReceiveMessages()!\n");
                break;
            }

            // Finished
            NN_LOG("Total bytes received [%lld]\n", g_MulticastReceive.GetTotalBytesRecieved());
        }
        else     // if sample is going to send
        {
            NN_LOG( "Sample will send Multicast Group messages using UDP to [%s:%d] on network interface [%s]\n",
                                          IpMulticastUdpAddress, IpMulticastUdpPort,
                                          (pInterface != nullptr ? pInterface : "default" ));

            // Initialize Multicast Receive
            ret = g_MulticastSend.Initialize(IpMulticastUdpAddress, IpMulticastUdpPort, pInterface);
            if (ret != 0)
            {
                NN_LOG("Failing calling 'MulticastSend::Initialize()!\n");
                break;
            }
            g_isMulticastSendInitilized = true;

            // Begin Receiving Multicast Messages
            ret = g_MulticastSend.SendMessages(MaximumNumberOfSecondsToRun);
            if (ret != 0)
            {
                NN_LOG("Failing calling 'MulticastSend::SendMessages()!\n");
                break;
            }

            // Finished
            NN_LOG("Total bytes sent [%lu]\n", g_MulticastSend.GetTotalBytesSent());
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    // If using Multicast Receiver
    if (g_isMulticastReceiveInitilized == true)
    {
        ret = g_MulticastReceive.Finalize();
        if (ret != 0)
        {
            NN_LOG("Failing calling 'MulticastReceive::Finalize()!\n");
        }
        g_isMulticastReceiveInitilized = false;
    }

    // If using Multicast Sender
    if (g_isMulticastSendInitilized == true)
    {
        ret = g_MulticastSend.Finalize();
        if (ret != 0)
        {
            NN_LOG("Failing calling 'MulticastSend::Finalize()!\n");
        }
        g_isMulticastSendInitilized = false;
    }

    // Finalize NIFM and Socket Library
    if (g_IsNetworkInitialized == true)
    {
        ret = FinalizeNetwork();
        if (ret != 0)
        {
            NN_LOG("Failing calling 'FinalizeNetwork()'\n");
        }
    }

    // Free Parsed Options - Network Interface
    if (pInterface != nullptr)
    {
        delete [] pInterface;
        pInterface = nullptr;
    }
}

}}   // Namespace nns / socket

// ------------------------------------------------------------------------------------------------
// Startup function
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup() NN_NOEXCEPT
{
    // Setup the size of total memory heap
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT(result.IsSuccess());

    // Allcate memory block from the heap for malloc
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);
    NN_ASSERT( result.IsSuccess() );

    // Set memory chunk for malloc
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// nnMain
// ------------------------------------------------------------------------------------------------
extern "C" void nnMain() NN_NOEXCEPT
{
    nns::socket::mainLoop();
}
