﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include <nn/socket.h>

#include <algorithm>
#include <string>

/**
 * @examplesource{SocketBasic.cpp,PageSampleSocketBasic}
 *
 * @brief
 * Socket API によるソケット通信
 */

/**
 * @page PageSampleSocketBasic Socket API によるソケット通信
 * @tableofcontents
 *
 * @brief
 * Socket API によるソケット通信のサンプルプログラムの解説です。
 *
 * @section PageSampleSocketBasic_SectionBrief 概要
 * サーバとクライアントの間で TCP ソケットを用いてメッセージの送受信を行います。
 *
 * @section PageSampleSocketBasic_SectionFileStoructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/SocketBasic
 * Samples/Sources/Applications/SocketBasic @endlink 以下にあります。
 *
 * @section PageSampleSocketBasic_SectionNecessaryEnvironment 必要な環境
 * 事前に SettingsManager を使用してネットワーク接続設定をインポートする必要があります。
 * インポート方法については @confluencelink{104465190,NIFM ライブラリ} を参考にしてください。
 *
 * @section PageSampleSocketBasic_SectionHowToOperate 操作方法
 * 先にサーバを実行し、続けてクライアントを実行してください。
 * 下記のオプションを指定することでサーバあるいはクライアントを実行します。
 * - -server
 *    - サーバとして起動します。
 * - -client server=&lt;ip_addr&gt;:&lt;port&gt; [message]
 *    - クライアントとして起動します。
 *    - &lt;ip_addr&gt;:&lt;port&gt; には、サーバの IP アドレスとポート番号を指定してください。サーバの IP アドレスとポート番号は、サーバの実行ログで確認できます。
 *    - [message] には、サーバーへ送信するメッセージを指定してください。省略時は、"Test message" を送信します。
 *
 * @section PageSampleSocketBasic_SectionPrecaution 注意事項
 * ネットワーク接続に失敗した場合のハンドリング方法は未定です
 *
 * @section PageSampleSocketBasic_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleSocketBasic_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリを初期化します。
 * - オプションに応じて、サーバあるいはクライアントを立ち上げます。
 * - クライアントはサーバに接続し、サーバに対してメッセージを送信します。
 * - サーバはクライアントから受信したメッセージを、そのままクライアントへ送り返します。
 * - クライアントはサーバーからメッセージを受信すると、サーバーから切断して処理を終了します。
 * - サーバーも処理を終了します。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 * ただし、 Release ビルドではログメッセージが出力されません。
 *
 * @verbinclude SocketBasic_ServerOutput.txt
 *
 * @verbinclude SocketBasic_ClientOutput.txt
 */

namespace
{

// Socket configuration and memory pool
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

const int MessageBufferSize     = 256;
const int IpAddressBufferSize   = 16;   /* includes null terminator ("xxx.xxx.xxx.xxx") */
const int PortBufferSize        = 6;    /* includes null terminator */

void RunServerProcess()
{
    nn::socket::SockAddrIn  saServer = { 0 };
    char                    clientMessage[MessageBufferSize] = { 0 };
    int                     socketDescriptor = -1;
    int                     clientSocketDescriptor = -1;
    ssize_t                 readByteCount = 0;
    nn::socket::SockAddrIn  sa = { 0 };
    nn::socket::SockLenT    saLen = sizeof(sa);
    nn::socket::InAddr      outIpAddress = { 0 };
    nn::Result              result = nn::Result();
    const char *            ExitMessage = "exit";

    NN_LOG("-- Server startup --\n");
    NN_LOG("-- In order to exit, send the string, \"exit\", from a client. --\n");

    if( (socketDescriptor = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)) < 0 )
    {
        NN_LOG("server:  Socket failed (error %d)\n", nn::socket::GetLastError());
        return;
    }

    saServer.sin_addr.S_addr = nn::socket::InetHtonl(nn::socket::InAddr_Any);
    saServer.sin_port        = nn::socket::InetHtons(0);
    saServer.sin_family      = nn::socket::Family::Af_Inet;

    if (nn::socket::Bind(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
    {
        NN_LOG("server:  Bind failed (error %d)\n", nn::socket::GetLastError());
        goto EXIT;
    }

    if( nn::socket::GetSockName(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&sa), &saLen) < 0 )
    {
        NN_LOG("server:  GetSockName failed (error %d)\n", nn::socket::GetLastError());
        goto EXIT;
    }

    result = nn::nifm::GetCurrentPrimaryIpAddress(&outIpAddress);
    if( result.IsSuccess() )
    {
        NN_LOG("server:  listening for incoming messages at: %s:%d\n",
            nn::socket::InetNtoa(outIpAddress),
            static_cast<int>(nn::socket::InetNtohs(sa.sin_port)));
    }
    else
    {
        NN_LOG("server:  GetCurrentIpAddress failed (error %d)\n", result.GetDescription());
        goto EXIT;
    }

    nn::socket::Listen(socketDescriptor, 4);

    while (NN_STATIC_CONDITION(true))
    {
        nn::socket::SockAddrIn saClientAddress = { 0 };
        nn::socket::SockLenT clientAddressSize = sizeof(saClientAddress);
        if( (clientSocketDescriptor = nn::socket::Accept(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saClientAddress), &clientAddressSize)) < 0 )
        {
            NN_LOG("server:  Accept failed (error %d)\n", nn::socket::GetLastError());
            goto EXIT;
        }

        NN_LOG("server:  connection accepted from %s\n", nn::socket::InetNtoa(saClientAddress.sin_addr));

        if ((readByteCount = nn::socket::Recv(clientSocketDescriptor, clientMessage, sizeof(clientMessage) - 1, nn::socket::MsgFlag::Msg_None)) > 0)
        {
            clientMessage[readByteCount] = '\0';

            /* look for the exit message */
            if( strnlen(clientMessage, MessageBufferSize - 1) == strlen(ExitMessage) )
            {
                std::string strClientMessage = clientMessage;
                /* but case-insensitive */
                std::transform(strClientMessage.begin(), strClientMessage.end(), strClientMessage.begin(), [](char c){return static_cast<char>(::tolower(c));});

                if( strClientMessage == ExitMessage )
                {
                    NN_LOG("server:  received exit message from client\n");
                    nn::socket::Close(clientSocketDescriptor);
                    goto EXIT;
                }
            }

            NN_LOG("server:  received from client - '%s'\n", clientMessage);

            if (nn::socket::Send(clientSocketDescriptor, clientMessage, strnlen(clientMessage, MessageBufferSize - 1), nn::socket::MsgFlag::Msg_None) < 0)
            {
                NN_LOG("server: Send failed (error %d)\n", nn::socket::GetLastError());
                nn::socket::Close(clientSocketDescriptor);
                goto EXIT;
            }
        }
        else if (readByteCount == 0)
        {
            NN_LOG("server:  client disconnect\n");
        }
        else
        {
            NN_LOG("server:  Recv failed (error %d)\n", nn::socket::GetLastError());
        }

        nn::socket::Close(clientSocketDescriptor);
    }

EXIT:
    nn::socket::Close(socketDescriptor);
    return;
}

void RunClientProcess(char *serverIp, char *serverPort, char *message)
{
    nn::socket::SockAddrIn saServer = { 0 };
    char        serverMessage[MessageBufferSize] = { 0 };
    int         socketDescriptor = -1;
    ssize_t     readByteCount = 0;
    ssize_t     sent = 0;

    /* if message is empty, send "Test message" */
    if( strnlen(message, MessageBufferSize - 1) == 0 )
    {
        strncpy(message, "Test message", MessageBufferSize - 1);
    }

    NN_LOG("-- Client startup --\n");

    if ((socketDescriptor = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp)) < 0)
    {
        NN_LOG("client:  Socket failed (error %d)\n", nn::socket::GetLastError());
        return;
    }

    NN_LOG("client:  connecting to %s:%s\n", serverIp, serverPort);

    nn::socket::InAddr outAddr = { 0 };
    nn::socket::InetAton(serverIp, &outAddr);

    saServer.sin_addr        = outAddr;
    saServer.sin_port        = nn::socket::InetHtons(static_cast<uint16_t>(atoi(serverPort)));
    saServer.sin_family      = nn::socket::Family::Af_Inet;

    if (nn::socket::Connect(socketDescriptor, reinterpret_cast<nn::socket::SockAddr *>(&saServer), sizeof(saServer)) < 0)
    {
        NN_LOG("client: Connect failed (error %d)\n", nn::socket::GetLastError());
        goto EXIT;
    }

    NN_LOG("client:  established connection to server\n");
    NN_LOG("client:  sending to server - '%s'\n", message);

    if( (sent = nn::socket::Send(socketDescriptor, message, strnlen(message, MessageBufferSize - 1), nn::socket::MsgFlag::Msg_None)) < 0 )
    {
        NN_LOG("client:  Send failed (error %d)\n", nn::socket::GetLastError());
        goto EXIT;
    }
    else
    {
        NN_LOG("client:  sent %d bytes\n", sent);
    }

    if( (readByteCount = nn::socket::Recv(socketDescriptor, serverMessage, sizeof(serverMessage) - 1, nn::socket::MsgFlag::Msg_None)) < 0 )
    {
        NN_LOG("client:  Recv failed (error %d)\n", nn::socket::GetLastError());
        goto EXIT;
    }

    serverMessage[readByteCount] = '\0';
    NN_LOG("client:  received from server - '%s'\n", serverMessage);
    NN_LOG("client:  disconnecting...\n");

EXIT:
    nn::socket::Close(socketDescriptor);
    return;
}

bool ParseArgs(bool *pServer, char *serverIp, char *serverPort, char *message)
{
    if( nn::os::GetHostArgc() > 1 )
    {
        const char *ModeServer = "-server";
        const char *ModeClient = "-client";
        char *mode = nn::os::GetHostArgv()[1];

        if( strncmp(mode, ModeServer, strlen(ModeServer)) == 0 )
        {
            *pServer = true;
            return true;
        }
        else if( strncmp(mode, ModeClient, strlen(ModeClient)) == 0 )
        {
            if( nn::os::GetHostArgc() > 2 )
            {
                const char *ServerToken = "server=";
                std::string strIpArg = nn::os::GetHostArgv()[2];
                if( strIpArg.compare(0, strlen(ServerToken), ServerToken) == 0 )
                {
                    strIpArg.erase(0, strlen(ServerToken));
                    size_t ipLength = strIpArg.find(':', 0);
                    if( ipLength != std::string::npos )
                    {
                        strncpy(serverIp, strIpArg.substr(0, ipLength).c_str(), IpAddressBufferSize);
                        strncpy(serverPort, strIpArg.substr(ipLength + 1).c_str(), PortBufferSize);

                        if( nn::os::GetHostArgc() > 3 )
                        {
                            strncpy(message, nn::os::GetHostArgv()[3], MessageBufferSize - 1);
                        }

                        *pServer = false;
                        return true;
                    }
                }
            }
        }
    }

    return false;
}

} // namespace

extern "C" void nnMain()
{
    nn::Result result = nn::Result();
    bool bServer = true;
    char serverPort[PortBufferSize] = { 0 };
    char serverIp[IpAddressBufferSize] = { 0 };
    char message[MessageBufferSize] = { 0 };

    NN_LOG("-- SocketBasic --\n");

    if( !ParseArgs(&bServer, serverIp, serverPort, message) )
    {
        NN_LOG("\nError: Invalid arguments.\n");
        NN_LOG("\nUsage:\n");
        NN_LOG("  -server                                    run as a server and wait for\n");
        NN_LOG("                                             client messages\n");
        NN_LOG("  -client server=<ip_addr>:<port> [message]  run as a client and send message\n");
        NN_LOG("                                             to specified server\n");
        return;
    }

    /* if socket library is used to communicate with external nodes, we must initialize
       network interface manager (NIFM) first */
    result = nn::nifm::Initialize();
    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::nifm::Initialize() failed. (error %d)\n", result.GetDescription());
        return;
    }

    /* this application is now requesting to use network interface */
    nn::nifm::SubmitNetworkRequest();

    /* wait for network interface availability, while providing status */
    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Waiting for network interface availability...\n");
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    if( !nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("\nError: network is not available.\n");
        goto CLEANUP_NIFM;
    }

    /* Initialize socket library, while supplying configuration and memory pool */
    result = nn::socket::Initialize(g_SocketConfigWithMemory);

    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::socket::Initialize() failed. (error %d)\n\n", result.GetDescription());
        goto CLEANUP_NIFM;
    }

    if( bServer )
    {
        RunServerProcess();
    }
    else
    {
        RunClientProcess(serverIp, serverPort, message);
    }

    nn::socket::Finalize();

CLEANUP_NIFM:
    /* this application no longer requires use of network interface */
    nn::nifm::CancelNetworkRequest();
}
