﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{RoStaticApplication.cpp,PageSampleRoStaticApplication}
 *
 * @brief
 *  動的モジュールを使用するサンプルプログラム
 */

/**
 * @page PageSampleRoStaticApplication 動的モジュールの使用
 * @tableofcontents
 *
 * @brief
 *  動的モジュールを使用するサンプルプログラムの解説です。
 *
 * @section PageSampleRoStaticApplication_SectionBrief 概要
 *  ここでは、動的モジュールを使用するサンプルプログラムの説明をします。
 *
 *  ro ライブラリの使用方法については、
 *  @ref nn::ro "ROライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleRoStaticApplication_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/RoSimple/RoStaticApplication Samples/Sources/Applications/RoSimple/RoStaticApplication @endlink 以下にあります。
 *
 * @section PageSampleRoStaticApplication_SectionNecessaryEnvironment 必要な環境
 *  PC環境では動作しません。
 *
 * @section PageSampleRoStaticApplication_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleRoStaticApplication_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleRoStaticApplication_SectionHowToExecute 実行手順
 *  事前に @link ../../../Samples/Sources/Libraries/ro/RoDynamicModule Samples/Sources/Libraries/ro/RoDynamicModule @endlink をビルドしてから、本サンプルプログラムをビルドし、実行してください。
 *
 */

#include <cstdlib>
#include <new>
#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/ro.h>

void PrintHelloWorld();

size_t ReadAll(void* pOut, size_t bufferSize, const char* path)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, path, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS(fileSize, static_cast<int64_t>(bufferSize));

    size_t readSize;
    result = nn::fs::ReadFile(&readSize, file, 0, pOut, bufferSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_EQUAL(static_cast<int64_t>(readSize), fileSize);

    nn::fs::CloseFile(file);

    return readSize;
}

extern "C" void nnMain()
{
    nn::Result result;
    static const size_t MaxFileSize  = 0x400000;
    void* nro;
    void* bss;
    void* nrr;
    size_t imageSize;
    size_t bufferSize;
    size_t nrrSize;
    nn::ro::Module module;
    nn::ro::RegistrationInfo info;

    nn::ro::Initialize();

    // ファイルシステムのメタデータキャッシュに必要なバッファサイズを修得
    size_t cacheSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT(result.IsSuccess());

    // キャッシュバッファを確保
    char* cacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(cacheBuffer);

    // ファイルシステムをマウント
    result = nn::fs::MountRom("rom", cacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // nrr を読み込み
    nrr = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    nrrSize = ReadAll(nrr, MaxFileSize, "rom:/.nrr/RoStaticApplication.nrr");

    // nrr を登録
    result = nn::ro::RegisterModuleInfo(&info, nrr);
    NN_ASSERT(result.IsSuccess());

    nro = aligned_alloc(nn::os::MemoryPageSize, MaxFileSize);
    imageSize = ReadAll(nro, MaxFileSize, "rom:/nro/RoDynamicModule.nro");

    result = nn::ro::GetBufferSize(&bufferSize, nro);
    NN_ASSERT(result.IsSuccess());
    if (bufferSize != 0)
    {
        bss = aligned_alloc(nn::os::MemoryPageSize, bufferSize);
    }
    else
    {
        bss = 0;
    }

    // nro をロード(シンボルは遅延解決)
    result = nn::ro::LoadModule(&module, nro, bss, bufferSize, nn::ro::BindFlag_Lazy);
    NN_ASSERT(result.IsSuccess());

    // 動的モジュール内のシンボルを呼び出す
    PrintHelloWorld();

    nn::ro::UnloadModule(&module);
    nn::ro::UnregisterModuleInfo(&info);
    nn::ro::Finalize();

    // アンマウントする
    nn::fs::Unmount("rom");
    delete[] cacheBuffer;
    cacheBuffer = NULL;

    return;
}
