﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{PctlSimple.cpp,PageSamplePctlSimple}
 *
 * @brief
 * ペアレンタルコントロール設定に基づいた制限を確認するサンプルプログラム
 */

/**
 * @page PageSamplePctlSimple PctlSimple
 * @tableofcontents
 *
 * @brief
 * ペアレンタルコントロール設定に基づいた制限を確認するサンプルプログラムです。
 *
 * @section PageSamplePctlSimple_SectionBrief 概要
 * nn::pctl::CheckFreeCommunicationPermission を使用して「他の人との自由なコミュニケーション」の
 * 制限確認を行うシンプルなサンプルです。
 *
 * @section PageSamplePctlSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/PctlSimple Samples/Sources/Applications/PctlSimple @endlink 以下にあります。
 *
 * @section PageSamplePctlSimple_SectionNecessaryEnvironment 必要な環境
 * 画面表示が利用可能である必要があります。
 * また、本プログラム自体は Windows 環境でも動作しますが、
 * ペアレンタルコントロール設定による制限を確認するには
 * 実機環境での動作が必要です。
 *
 * @section PageSamplePctlSimple_SectionHowToOperate 操作方法
 * 最初の画面(メインシーン)でAを入力するとサブシーンに遷移します。
 * この際、ペアレンタルコントロール設定により「他の人との自由なコミュニケーション」が
 * 制限されている場合は、エラーが表示され遷移が行われません。
 *
 * サブシーンでBを入力するとメインシーンに戻り、メインシーンで
 * Bを入力するとアプリケーションを終了します。
 *
 * @section PageSamplePctlSimple_SectionPrecaution 注意事項
 * 実際に制限されているかどうかを確認する場合は、アプリケーション起動前に
 * HOMEメニュー等でペアレンタルコントロール設定を行う必要があります。@n
 * また、本サンプルプログラムでは対応していますが、制限を発生させるには
 * pctl の関数を呼び出すだけでなく、プログラムの Meta ファイルに
 * ペアレンタルコントロール設定の記述を追加する必要があります。
 * 本サンプルプログラムと同じディレクトリにある PctlSimple.*.nmeta ファイルに
 * その設定が含まれていますので、必要に応じて参考にしてください。
 *
 * @section PageSamplePctlSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 * ただし、Meta ファイルの設定が読み込まれないと正しく制限確認が行えないため、
 * nsp ファイルを作成してインストールするなど、管理データが読み込まれる
 * 環境で実行する必要があります。
 *
 * @section PageSamplePctlSimple_SectionDetail 解説
 * 本サンプルプログラムは nn::pctl::CheckFreeCommunicationPermission を利用し、
 * 「他の人との自由なコミュニケーション」の機能が制限されているかの確認を行うデモです。
 *
 * このサンプルプログラムではメインシーンとサブシーンの2つのシーンを持ちます。
 * このうちサブシーンは「他の人との自由なコミュニケーション」機能の利用を
 * 想定したシーンであり、シーンへの遷移前に制限確認を実施しています。
 * 事前に設定されたペアレンタルコントロール設定に基づき、
 * 制限が発生する場合はサブシーンへの遷移を行いません。
 *
 * 具体的には各関数で以下の処理を行っています。
 *
 * - nnMain: 初期化・終了処理とメインループ開始処理
 * - MainLoop: メインループ処理
 *   - ユーザー入力によってシーン切り替えも行います。
 * - ChangeSceneToMain: メインシーンへの遷移処理
 * - ChangeSceneToSub: サブシーンへの遷移処理
 *   - サブシーンへ遷移するための制限確認を行います。
 * - PrintMain: メインシーンの描画処理
 * - PrintSub: サブシーンの描画処理
 */

// NintendoSDK のヘッダファイルをインクルードする前に、NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET マクロを
// 定義することで、DebugFontWriter::Print() の入力文字コードを Windows のロケールのデフォルト
// (日本語の場合、CP932)に変更できます。
#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>

// ユーザー入力取得用
#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/util/util_BitFlagSet.h>

// 画面描画用
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

// ペアレンタルコントロールサービス利用
#include <nn/pctl.h>

namespace
{
    // シーン定数
    enum SceneId
    {
        // メインシーン(起動時のシーン)
        SceneId_Main = 0,
        // サブシーン(制限をかけるべきシーン)
        SceneId_Sub
    };

    // 状態によって出力関数を切り替えるための変数
    SceneId s_CurrentScene = SceneId_Main;

    // 制限に引っかかった旨を表示する残りフレーム数
    int s_ShowFramesForRestrictionMessage = 0;
} // namespace

//------------------------------------------------------------------------------
//  出力関数 - メインシーン
//------------------------------------------------------------------------------
void PrintMain(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };
    nn::util::Color4u8Type color1 = { { 0, 255, 0, 255 } };
    nn::util::Color4u8Type color2 = { { 255, 0, 0, 255 } };

    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(10, 10);
    writer.Print("PctlSimple - Main scene");

    if ((frame % 60) < 30)
    {
        writer.Print("■");
    }

    writer.SetTextColor(color1);
    writer.SetCursor(100, 100);
    writer.Print("Press A to enter sub scene for free communication.\n");
    writer.Print("Press B to exit application.\n");

    if (s_ShowFramesForRestrictionMessage > 0)
    {
        writer.SetTextColor(color2);
        writer.SetCursor(150, 200);
        writer.Print("Free communication is restricted.\n");
    }
}

//------------------------------------------------------------------------------
//  出力関数 - サブシーン
//------------------------------------------------------------------------------
void PrintSub(nn::gfx::util::DebugFontTextWriter& writer, int frame)
{
    nn::util::Color4u8Type color0 = { { 255, 128, 128, 255 } };
    nn::util::Color4u8Type color1 = { { 0, 255, 0, 255 } };

    writer.SetScale(1.0f, 1.0f);

    writer.SetTextColor(color0);
    writer.SetCursor(10, 10);
    writer.Print("PctlSimple - Sub scene for free communication");

    if ((frame % 60) < 30)
    {
        writer.Print("■");
    }

    writer.SetTextColor(color1);
    writer.SetCursor(100, 100);
    writer.Print("Press B to back to main scene.\n");
}

//------------------------------------------------------------------------------
//  メインシーンへの切り替え
//------------------------------------------------------------------------------
void ChangeSceneToMain()
{
    s_CurrentScene = SceneId_Main;
}

//------------------------------------------------------------------------------
//  サブシーンへの切り替え
//------------------------------------------------------------------------------
void ChangeSceneToSub()
{
    // 制限を行うシーンに入る前に必ずシーンに入れるかどうかを確認する
    // ※ アプリケーションがバックグラウンドにいる間に状態が変化することもあるので
    //    値をキャッシュせずに毎回呼び出しを行う
    if (nn::pctl::CheckFreeCommunicationPermission(true))
    {
        s_CurrentScene = SceneId_Sub;
    }
    else
    {
        // 制限されている旨をメインシーン画面にも表示する
        // ※ アプリケーション側での制限された旨の表示は任意です。
        s_ShowFramesForRestrictionMessage = 120;
    }
}

//------------------------------------------------------------------------------
//  計算処理コールバック
//------------------------------------------------------------------------------
struct CalculateCallbackUserData
{
    nn::hid::DebugPadButtonSet* pButtonMaskEnter;
    nn::hid::DebugPadButtonSet* pButtonMaskBack;
    nn::hid::DebugPadButtonSet* pButtonsPrev;
    int* pFrame;
    bool* pIsExited;
    nn::gfx::util::DebugFontTextWriter* pWriter;
};

void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    CalculateCallbackUserData* pCallbackUserData = reinterpret_cast<CalculateCallbackUserData*>(pUserData);

    // フォント表示
    switch (s_CurrentScene)
    {
    case SceneId_Main:
        PrintMain(*pCallbackUserData->pWriter, *pCallbackUserData->pFrame);
        break;
    case SceneId_Sub:
        PrintSub(*pCallbackUserData->pWriter, *pCallbackUserData->pFrame);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // 入力状態の更新
    nn::hid::DebugPadState pad;
    nn::hid::GetDebugPadState(&pad);

    nn::hid::DebugPadButtonSet buttonsTrigger = pad.buttons & ~(*pCallbackUserData->pButtonsPrev);
    (*pCallbackUserData->pButtonsPrev) = pad.buttons;

    // シーン別の状態確認処理
    switch (s_CurrentScene)
    {
    case SceneId_Main:
        if ((buttonsTrigger & (*pCallbackUserData->pButtonMaskBack)).IsAnyOn())
        {
            (*pCallbackUserData->pIsExited) = true;
        }
        else if ((buttonsTrigger & (*pCallbackUserData->pButtonMaskEnter)).IsAnyOn())
        {
            ChangeSceneToSub();
        }
        break;
    case SceneId_Sub:
        if ((buttonsTrigger & (*pCallbackUserData->pButtonMaskBack)).IsAnyOn())
        {
            ChangeSceneToMain();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    if (s_ShowFramesForRestrictionMessage > 0)
        --s_ShowFramesForRestrictionMessage;

    ++(*pCallbackUserData->pFrame);
    if (*pCallbackUserData->pFrame == 120)
        (*pCallbackUserData->pFrame) = 0;
}

//------------------------------------------------------------------------------
//  コマンド生成コールバック
//------------------------------------------------------------------------------
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    nn::gfx::util::DebugFontTextWriter* pWriter = reinterpret_cast<nn::gfx::util::DebugFontTextWriter*>(pUserData);

    // コマンド生成
    pGraphicsFramework->BeginFrame(bufferIndex);
    {
        nn::gfx::CommandBuffer* rootCommandBuffer = pGraphicsFramework->GetRootCommandBuffer(bufferIndex);

        // レンダーターゲット、ビューポートシザー設定
        nn::gfx::ColorTargetView* target = pGraphicsFramework->GetColorTargetView();
        rootCommandBuffer->ClearColor(target, 0.1f, 0.1f, 0.1f, 1.0f, nullptr);
        rootCommandBuffer->SetRenderTargets(1, &target, nullptr);
        rootCommandBuffer->SetViewportScissorState(pGraphicsFramework->GetViewportScissorState());

        // デバッグフォント用のコマンド生成
        pWriter->Draw(rootCommandBuffer);
    }
    pGraphicsFramework->EndFrame(bufferIndex);
}

//------------------------------------------------------------------------------
//  メインループ処理
//------------------------------------------------------------------------------
void MainLoop(nns::gfx::GraphicsFramework& gfw, nn::gfx::util::DebugFontTextWriter& writer)
{
    nn::hid::DebugPadButtonSet buttonMaskEnter = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();
    buttonMaskEnter.Set<nn::hid::DebugPadButton::A>(true);

    nn::hid::DebugPadButtonSet buttonMaskBack = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();
    buttonMaskBack.Set<nn::hid::DebugPadButton::B>(true);

    nn::hid::DebugPadButtonSet buttonsPrev = {};

    int frame = 0;
    bool isExited = false;

    // フレームワーク設定
    CalculateCallbackUserData calculateUserData;
    calculateUserData.pButtonMaskEnter = &buttonMaskEnter;
    calculateUserData.pButtonMaskBack = &buttonMaskBack;
    calculateUserData.pButtonsPrev = &buttonsPrev;
    calculateUserData.pFrame = &frame;
    calculateUserData.pIsExited = &isExited;
    calculateUserData.pWriter = &writer;

    gfw.SetCalculateCallback(CalculateCallback, &calculateUserData);
    gfw.SetMakeCommandCallback(MakeCommandCallback, &writer);

    // 毎フレームのレンダリング
    while (!isExited)
    {
        gfw.ProcessFrame();
    }
    gfw.QueueFinish();
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
    // FS を使用する場合はフレームワークよりも前に初期化します

    // デバッグ用入力の割り当てを保持するための変数
    nn::settings::DebugPadKeyboardMap map;

    // キーボード入力をデバッグ用コントローラー入力として扱うための処理
    map.buttonA = nn::hid::KeyboardKey::A::Index;
    map.buttonB = nn::hid::KeyboardKey::B::Index;
    nn::settings::SetDebugPadKeyboardMap(map);

    nn::hid::InitializeDebugPad();

    // フレームワーク初期化
    const int BufferCount = 2;
    const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    nns::gfx::GraphicsFramework gfw;
    gfw.Initialize(fwInfo);

    // デバッグフォント初期化
    nns::gfx::GraphicsFramework::DebugFontTextWriter writer;
    gfw.InitializeDebugFontTextWriter(&writer, 1024, BufferCount);

    // 初期シーンとしてメインシーンに切り替え
    ChangeSceneToMain();

    // メインループ
    MainLoop(gfw, writer.object);

    // デバッグフォント終了
    gfw.FinalizeDebugFontTextWriter(&writer);

    // フレームワーク終了
    gfw.Finalize();
}
