﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{PcmBasic.cpp,PageSamplePcmBasic}
 *
 * @brief
 *  性能選択のサンプルプログラム
 */

/**
 * @page PageSamplePcmBasic 消費電力計測
 * @tableofcontents
 *
 * @brief
 *  電力計測ライブラリを用いた消費電力計測サンプルプログラムの解説です。
 *
 * @section PageSamplePcmBasic_SectionBrief 概要
 *  ここでは、電力計測ライブラリを用いて消費電力を計測する方法を解説します。
 *
 *  電力計測ライブラリの使い方については、
 *  @ref nn::pcm "電力計測ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSamplePcmBasic_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/PcmBasic
 *  Samples/Sources/Applications/PcmBasic @endlink 以下にあります。
 *
 * @section PageSamplePcmBasic_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSamplePcmBasic_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSamplePcmBasic_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSamplePcmBasic_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSamplePcmBasic_SectionDetail 解説
 *
 * @subsection PageSamplePcmBasic_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  PcmBasic.cpp
 *  @includelineno PcmBasic.cpp
 *
 * @subsection PageSamplePcmBasic_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムは、指定可能な計測ポイントすべてで消費電力を計測し、その結果を表示します。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 電力計測ライブラリの初期化
 *  - 計測ポイントごとの消費電力の測定と表示
 *      - 計測したい計測ポイントに対応する nn::pcm::MeasuringPoint_ から始まる列挙値を引数に指定してください。
 *  - 実行結果の確認
 *      - 指定した計測ポイントで計測された消費電力（ mW 単位）が戻り値に格納されます。
 *
 *  このサンプルプログラムを SDEV で実行したときの結果を以下に示します。
 *  計測ポイントを搭載していない EDEV で実行した場合は、すべての計測ポイントで "Not implemented on this board." となります。
 *
 *  @verbinclude  PcmBasic_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/gfx.h>
#include <nn/os.h>
#include <nn/pcm/pcm.h>
#include <nv/nv_MemoryManagement.h>

//-----------------------------------------------------------------------------

namespace {

// 計測ポイントの情報セット
struct MeasuringPointInfo
{
    nn::pcm::MeasuringPoint point;
    const char*             pointName;
};

// 計測ポイントのセット（NX スペック固有の定義）
const MeasuringPointInfo MeasuringPointList[] =
{
    { nn::pcm::MeasuringPoint_Gpu, "GPU" },
    { nn::pcm::MeasuringPoint_Cpu, "CPU" },
    { nn::pcm::MeasuringPoint_Ddr, "DDR" },
};

// 計測ポイントの個数
const int MeasuringPointCountMax = sizeof(MeasuringPointList) / sizeof(MeasuringPointList[0]);

//
// グラフィックデバイスの初期化に必要な変数・関数
//
const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;

nn::gfx::Device g_Device;

void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    g_Device.Initialize(info);
}

void FinalizeDevice()
{
    g_Device.Finalize();
}

void* Allocate(size_t size, size_t alignment, void*)
{
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void Free(void* addr, void*)
{
    free(addr);
}

void* Reallocate(void* addr, size_t newSize, void*)
{
    return realloc(addr, newSize);
}

}   // namespace

//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    NN_LOG("PCM demo start.\n\n");

    // グラフィックスシステムのためのメモリ周りの初期化
    nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
    nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);

    // GPU の電源を入れることを目的としてグラフィックスシステムを初期化
    nn::gfx::Initialize();
    InitializeDevice();

    // GPU の計測電力が安定するのを待機
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));

    // 電力計測ライブラリの初期化
    nn::pcm::Initialize();

    // 計測ポイントごとの消費電力の測定と表示
    NN_LOG("[Measurement Result]\n");
    for (int point = 0; point < MeasuringPointCountMax; ++point)
    {
        if (nn::pcm::IsSupported(MeasuringPointList[point].point))
        {
            int power = nn::pcm::ReadCurrentPower(MeasuringPointList[point].point);
            NN_LOG("%4d mW (@ %s)\n", power, MeasuringPointList[point].pointName);
        }
        else
        {
            NN_LOG("---- mW (@ %s) / Not implemented on this board\n", MeasuringPointList[point].pointName);
        }
    }

    // 電力計測ライブラリの終了
    nn::pcm::Finalize();

    // グラフィックスシステムの終了
    FinalizeDevice();
    nn::gfx::Finalize();

    NN_LOG("\nPCM demo end.\n\n");
}
