﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/ovln/ovln_SenderForOverlay.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>
#include <nn/os.h>
#include <cstring>

namespace {

nn::ovln::Message MakeStringMessage(uint32_t tag, const char* s)
{
    nn::ovln::Message ret = {};
    ret.tag = tag;
    ret.dataSize = static_cast<uint32_t>(std::strlen(s) + 1);
    std::memcpy(&ret.data, s, ret.dataSize);
    return ret;
}

void WaitForAllMessageReceived(nn::ovln::SenderForOverlayType* pSender)
{
    while (!(nn::ovln::GetUnreceivedMessageCount(pSender) == 0))
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
}

}

void SampleNotifierSubMain(nn::os::Event* pTerminateRequestEvent)
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderLibraryForOverlay());
    NN_UTIL_SCOPE_EXIT
    {
        nn::ovln::FinalizeSenderLibraryForOverlay();
    };

    // valueSender を、値向け sender として初期化
    nn::ovln::SenderForOverlayType valueSender;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderForValue(&valueSender));
    NN_UTIL_SCOPE_EXIT
    {
        WaitForAllMessageReceived(&valueSender);
        nn::ovln::FinalizeSender(&valueSender);
    };

    // 値向け Sender に一定の間隔をあけて送信
    // → Receiver が受け取ってくれる
    // → 通常全てのメッセージが届く
    // ※ 値向け Sender への Send は必ず true を返すため、チェック不要
    {
        Send(&valueSender, MakeStringMessage(100001, "a"));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        Send(&valueSender, MakeStringMessage(100001, "b"));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        Send(&valueSender, MakeStringMessage(100001, "c"));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        Send(&valueSender, MakeStringMessage(100001, "d"));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        Send(&valueSender, MakeStringMessage(100001, "e"));
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }

    // 値向け Sender に間断なく送信
    // → Receiver の受け取りが追い付かないため、上書きされる
    // → 最後のメッセージだけが届く
    {
        Send(&valueSender, MakeStringMessage(100002, "a"));
        Send(&valueSender, MakeStringMessage(100002, "b"));
        Send(&valueSender, MakeStringMessage(100002, "c"));
        Send(&valueSender, MakeStringMessage(100002, "d"));
        Send(&valueSender, MakeStringMessage(100002, "e"));
    }

    //// queueSender を、長さ 5 のキューを持つ sender として初期化
    nn::ovln::SenderForOverlayType queueSender;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ovln::InitializeSenderWithQueue(&queueSender, 5));
    NN_UTIL_SCOPE_EXIT
    {
        WaitForAllMessageReceived(&queueSender);
        nn::ovln::FinalizeSender(&queueSender);
    };

    // キューの長さが 5 の sender に 5 個連続で送信
    // → Receiver が受け取らなくても、キューにすべて詰め込める
    // → 全てのメッセージの send に成功
    {
        auto sendCount = 0;
        sendCount += Send(&queueSender, MakeStringMessage(100003, "a")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100003, "b")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100003, "c")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100003, "d")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100003, "e")) ? 1 : 0;
        NN_LOG("sendCount[100003] = %d / 5\n", sendCount);
    }

    // 掃けるのを待機
    WaitForAllMessageReceived(&queueSender);

    // キューの長さが 5 の sender に 10 個連続で送信
    // → 多くの場合、キューが詰まる
    // → 全てのメッセージの send に成功しない可能性が高い
    {
        auto sendCount = 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "a")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "b")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "c")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "d")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "e")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "f")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "g")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "h")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "i")) ? 1 : 0;
        sendCount += Send(&queueSender, MakeStringMessage(100004, "j")) ? 1 : 0;
        NN_LOG("sendCount[100004] = %d / 10\n", sendCount);
    }

    // 掃けるのを待機
    WaitForAllMessageReceived(&queueSender);

    // キューの長さが 5 の sender に適度な間隔で 10 個連続で送信
    // → 多くの場合、キューが詰まらずすべてのメッセージを送れる可能性が高い
    {
        auto sendCount = 0;
        sendCount += Send(&queueSender, MakeStringMessage(100005, "a")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "b")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "c")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "d")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "e")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "f")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "g")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "h")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "i")) ? 1 : 0;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        sendCount += Send(&queueSender, MakeStringMessage(100005, "j")) ? 1 : 0;
        NN_LOG("sendCount[100005] = %d / 10\n", sendCount);
    }

    // 掃けるのを待機
    WaitForAllMessageReceived(&queueSender);

    // 終了リクエストは待たずに終了する
    NN_UNUSED(pTerminateRequestEvent);
}
