﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsTimerEvent.cpp,PageSampleOsTimerEvent}
 *
 * @brief
 *  タイマーイベント機能のサンプルプログラム
 */

/**
 * @page PageSampleOsTimerEvent TimerEvent
 * @tableofcontents
 *
 * @brief
 *  タイマーイベント機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsTimerEvent_SectionBrief 概要
 *  ここでは、タイマーイベント機能を使ったサンプルプログラムの説明をします。
 *
 *  タイマーイベント機能の使い方については、タイマーイベント機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsTimerEvent_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsTimerEvent Samples/Sources/Applications/OsTimerEvent @endlink 以下にあります。
 *
 * @section PageSampleOsTimerEvent_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsTimerEvent_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsTimerEvent_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsTimerEvent_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsTimerEvent_SectionDetail 解説
 *
 * @subsection PageSampleOsTimerEvent_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsTimerEvent.cpp
 *  @includelineno OsTimerEvent.cpp
 *
 * @subsection PageSampleOsTimerEvent_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのタイマーイベントと 2 つのスレッドを生成
 *  - 2 つのスレッドは、 ThreadFunction1() と ThreadFunction2()
 *  - 1 つ目のタイマーイベントを周期タイマーイベントとして起動
 *  - 2 つ目のタイマーイベントをワンショットタイマーイベントとして起動
 *  - ThreadFunction1() では WaitTimerEvent() で周期タイマーイベントを待機
 *  - ThreadFunction2() では WaitTimerEvent() でワンショットタイマーイベントを待機
 *  - ThreadFunction1() の最後で、周期タイマーイベントの動作を停止
 *  - 2 つのスレッドが終了するまで nnMain() で待機
 *  - nnMain() にて 2 つのスレッドを破棄、2 つのタイマーイベントも破棄
 *
 *  ThreadFunction1() と ThreadFunction2() は、それぞれ個別に起動された
 *  タイマーイベントを nn::os::WaitTimerEvent() 機能を使って待機しています。
 *
 *  1 つ目のタイマーイベントは周期起動ですが、 WaitTimerEvent() でスレッドが
 *  起床した後も自動的に次の周期カウントが行なわれて、 StartPeriodicTimerEvent()
 *  で指定された周期ごとに自動的に g_TimerEvent1 がシグナル化されている様子が
 *  分かると思います。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsTimerEvent_OutputExample.txt
 *
 *  このサンプルからも分かるように、タイマーイベントは通常のイベントに、
 *  StartPeriodicTimerEvent() や StartOneShotTimerEvent() で指定された時間に
 *  自動的にシグナル化される機能を持ったものであるということも出来ます。
 *
 */

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

nn::os::Tick            g_StartTick;                // 開始時点の Tick 値

nn::os::TimerEventType  g_TimerEvent1;              // タイマーイベント 1
nn::os::TimerEventType  g_TimerEvent2;              // タイマーイベント 2

const size_t            StackSize = 8192;           // スタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack1[StackSize];  // タイマーイベント 1 待機スレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char   g_ThreadStack2[StackSize];  // タイマーイベント 2 待機スレッドのスタック

}   // namespace

//-----------------------------------------------------------------------------

inline int64_t GetMilliSecondsFromStart()
{
    nn::os::Tick currentTick = nn::os::GetSystemTick();
    return nn::os::ConvertToTimeSpan(currentTick - g_StartTick).GetMilliSeconds();
}

//-----------------------------------------------------------------------------
//  1 つ目のスレッド（周期タイマーイベントを待機する）
//
void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    for (int i=0; i<10; ++i)
    {
        nn::os::WaitTimerEvent( &g_TimerEvent1 );
        NN_LOG("Thread1: PeriodicTimerEvent: count=%d  time=%5lld (msec)\n", i, GetMilliSecondsFromStart());
    }

    nn::os::StopTimerEvent( &g_TimerEvent1 );
}


//-----------------------------------------------------------------------------
//  2 つ目のスレッド（ワンショットタイマーイベントを待機する）
//
void ThreadFunction2(void *arg)
{
    NN_UNUSED(arg);

    nn::os::WaitTimerEvent( &g_TimerEvent2 );
    NN_LOG("Thread2: OneShotTimerEvent: time=%5lld (msec)\n", GetMilliSecondsFromStart());
}


//-----------------------------------------------------------------------------
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::Result          result;

    // タイマーイベントを初期化する
    nn::os::InitializeTimerEvent( &g_TimerEvent1, nn::os::EventClearMode_AutoClear );
    nn::os::InitializeTimerEvent( &g_TimerEvent2, nn::os::EventClearMode_AutoClear );

    // スレッドを生成する
    result = nn::os::CreateThread( &thread1, ThreadFunction1, NULL, g_ThreadStack1, StackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread1." );

    result = nn::os::CreateThread( &thread2, ThreadFunction2, NULL, g_ThreadStack2, StackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create thread2." );

    // 開始時点の時間
    g_StartTick = nn::os::GetSystemTick();

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );


    // タイマーイベント 1 は周期タイマーイベントとして開始
    const nn::TimeSpan interval = nn::TimeSpan::FromMilliSeconds(100);
    nn::os::StartPeriodicTimerEvent( &g_TimerEvent1, interval, interval );

    // タイマーイベント 2 はワンショットタイマーイベントとして開始
    const nn::TimeSpan oneShotTime = nn::TimeSpan::FromMilliSeconds(1500);
    nn::os::StartOneShotTimerEvent( &g_TimerEvent2, oneShotTime );


    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );

    // タイマーイベントをファイナライズする
    nn::os::FinalizeTimerEvent( &g_TimerEvent1 );
    nn::os::FinalizeTimerEvent( &g_TimerEvent2 );
}

