﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsThread.cpp,PageSampleOsThread}
 *
 * @brief
 *  スレッド機能のサンプルプログラム
 */

/**
 * @page PageSampleOsThread Thread
 * @tableofcontents
 *
 * @brief
 *  スレッド機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsThread_SectionBrief 概要
 *  ここでは、スレッド機能を使ったサンプルプログラムの説明をします。
 *
 *  スレッド機能の使い方については、スレッド機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsThread_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsThread Samples/Sources/Applications/OsThread @endlink 以下にあります。
 *
 * @section PageSampleOsThread_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsThread_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsThread_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsThread_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsThread_SectionDetail 解説
 *
 * @subsection PageSampleOsThread_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsThread.cpp
 *  @includelineno OsThread.cpp
 *
 * @subsection PageSampleOsThread_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成
 *  - 2 つのスレッドは、 ThreadFunction1() と ThreadFunction2()
 *  - ThreadFunction1() では自スレッドのスレッド情報を出力して終了
 *  - ThreadFunction2() では ThreadFunction1 が終了するのを待つ
 *  - ThreadFunction2() にて自スレッドのスレッド情報を出力
 *  - ThreadFunction2() にて自スレッドのスレッド名を変更
 *  - ThreadFunction2() にて自スレッドの優先度を変更
 *  - ThreadFunction2() にて再度、自スレッドのスレッド情報を出力して終了
 *  - 2 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  本サンプルプログラムではスレッド名の変更と取得、優先度の変更と取得を行なっています。
 *  主なスレッドの生成から終了までは、 nnMain() で制御しています。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsThread_OutputExample.txt
 *
 */

#include <stdint.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t           ThreadStackSize = 8192;              // スレッド操作スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack1[ ThreadStackSize ];   // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_ThreadStack2[ ThreadStackSize ];   // 2つ目のスレッドのスタック

nn::os::ThreadType  g_Thread1;
nn::os::ThreadType  g_Thread2;

}   // namespace

//-----------------------------------------------------------------------------

//
//  スレッド情報を表示
//
void    PrintThreadInfo(const char* name, nn::os::ThreadType* thread)
{
    NN_LOG("[%s]\n", name);

    const char* threadName = nn::os::GetThreadNamePointer( thread );
    NN_LOG("name     : %s\n", threadName);
    NN_LOG("priority : %d\n", nn::os::GetThreadPriority( thread ));
    NN_LOG("core No. : %d\n", nn::os::GetCurrentCoreNumber());
    NN_LOG("\n");
}


//
//  1 つ目のスレッド
//
void ThreadFunction1(void *arg)
{
    NN_UNUSED(arg);

    PrintThreadInfo( "thread1", &g_Thread1 );
}


//
//  2 つ目のスレッド
//
void ThreadFunction2(void *arg)
{
    NN_UNUSED(arg);

    // スレッド 1 が終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );

    PrintThreadInfo( "thread2", &g_Thread2 );

    // スレッド 2 の名前を変更
    NN_LOG("Change name of thread2\n");
    nn::os::SetThreadName( &g_Thread2, "Thread2_NewName" );

    // スレッド 2 の優先度を変更
    NN_LOG("Change priority of thread2 to 5\n");
    nn::os::ChangeThreadPriority( &g_Thread2, 5 );

    NN_LOG("\n");
    PrintThreadInfo( "thread2", &g_Thread2 );
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::Result      result;

    // スレッドを生成する
    result = nn::os::CreateThread( &g_Thread1, ThreadFunction1, NULL, g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread1." );

    result = nn::os::CreateThread( &g_Thread2, ThreadFunction2, NULL, g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create g_Thread2." );

    // スレッドの実行を開始する
    nn::os::StartThread( &g_Thread1 );
    nn::os::StartThread( &g_Thread2 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &g_Thread1 );
    nn::os::WaitThread( &g_Thread2 );

    // スレッドを破棄する
    nn::os::DestroyThread( &g_Thread1 );
    nn::os::DestroyThread( &g_Thread2 );
}

