﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsSemaphore.cpp,PageSampleOsSemaphore}
 *
 * @brief
 *  セマフォ機能のサンプルプログラム
 */

/**
 * @page PageSampleOsSemaphore Semaphore
 * @tableofcontents
 *
 * @brief
 *  セマフォ機能のサンプルプログラムの解説です。
 *
 * @section PageSampleOsSemaphore_SectionBrief 概要
 *  ここでは、セマフォ機能を使ったサンプルプログラムの説明をします。
 *
 *  セマフォ機能の使い方については、セマフォ機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsSemaphore_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsSemaphore Samples/Sources/Applications/OsSemaphore @endlink 以下にあります。
 *
 * @section PageSampleOsSemaphore_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsSemaphore_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsSemaphore_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *  また、本サンプルは利用可能な CPU コアが 3 つ以上の動作環境でのみ動作します。
 *
 * @section PageSampleOsSemaphore_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsSemaphore_SectionDetail 解説
 *
 * @subsection PageSampleOsSemaphore_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsSemaphore.cpp
 *  @includelineno OsSemaphore.cpp
 *
 * @subsection PageSampleOsSemaphore_SectionSampleDetail サンプルプログラムの解説
 *
 *  セマフォ機能は複数のスレッド間で共有するリソースを安全に排他しながら
 *  同期制御を行なうための機構です。ミューテックス機能と異なるのは、
 *
 *  - 複数のスレッドが同時にセマフォを獲得することができる
 *  - セマフォを獲得したスレッド以外のスレッドでもセマフォを返却することができる
 *
 *  本サンプルでは、上記のうち、前者の複数のスレッドがセマフォを獲得できる
 *  という点を示すために以下のようなロジックで構成されています。
 *
 *  - nnMain() にて、3 つのスレッドを生成（生成時に優先コアを 0～2 に分散）
 *  - 3 つのスレッドは、全て ThreadSample() をスレッド関数として実行
 *  - 各スレッドは以下の処理を一定回数繰り返す
 *    - g_Semaphore を獲得する
 *    - 自スレッド名をログ出力して一定時間待機する
 *    - 改行をログ出力して再び一定時間待機する
 *    - g_Semaphore を返却する
 *  - nnMain() は 3 つ全てのスレッドが終了するのを待機してから終了する
 *
 *  各スレッドは、セマフォカウント 2 で初期化した g_Semaphore を使って
 *  排他するため、各スレッドのスレッド名が横に最大 2 つ並んでログ出力されます。
 *
 *  どのスレッドがセマフォ獲得に成功するかは、各スレッドの優先度や処理速度、
 *  nn::os::AcquireSemaphore() への到着順などによって変化するため、
 *  ログの出力結果は動作環境および実行する度に異なります。そのため、
 *  本サンプルでは 3 つのスレッドのうちループを早く終えるスレッドと
 *  他スレッドよりも遅く終えるスレッドが存在します。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsSemaphore_OutputExample.txt
 *
 *  この例では、Thread-1 が他の 2 つのスレッドよりもセマフォを獲得する機会が
 *  遅かったことを示しています。
 *
 */

#include <cstdint>
#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t ThreadStackSize = 0x4000;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack1[ ThreadStackSize ];  // 1つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack2[ ThreadStackSize ];  // 2つ目のスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack3[ ThreadStackSize ];  // 3つ目のスレッドのスタック

nn::os::SemaphoreType   g_Semaphore;        // 操作対象のセマフォ

}

//-----------------------------------------------------------------------------

//
//  全スレッド共通のスレッド関数
//
void ThreadSample(void *arg)
{
    auto str = reinterpret_cast<char*>(arg);

    for (int i=0; i<10; ++i)
    {
        // セマフォを獲得
        nn::os::AcquireSemaphore( &g_Semaphore );

        // セマフォ獲得に成功したスレッドのスレッド名を出力
        NN_LOG("<%s> ", str);
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(700) );

        // 改行を出力
        NN_LOG("\n");
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(300) );

        // セマフォを返却
        nn::os::ReleaseSemaphore( &g_Semaphore );
    }
}


//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  thread1;
    nn::os::ThreadType  thread2;
    nn::os::ThreadType  thread3;

    // セマフォを初期化する
    nn::os::InitializeSemaphore( &g_Semaphore, 2, 2 );

    // スレッドを生成する
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &thread1, ThreadSample, const_cast<char*>("Thread-1"), g_ThreadStack1, ThreadStackSize, nn::os::DefaultThreadPriority, 0 ) );

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &thread2, ThreadSample, const_cast<char*>("Thread-2"), g_ThreadStack2, ThreadStackSize, nn::os::DefaultThreadPriority, 1 ) );

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &thread3, ThreadSample, const_cast<char*>("Thread-3"), g_ThreadStack3, ThreadStackSize, nn::os::DefaultThreadPriority, 2 ) );

    // スレッドの実行を開始する
    nn::os::StartThread( &thread1 );
    nn::os::StartThread( &thread2 );
    nn::os::StartThread( &thread3 );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &thread1 );
    nn::os::WaitThread( &thread2 );
    nn::os::WaitThread( &thread3 );

    // スレッドを破棄する
    nn::os::DestroyThread( &thread1 );
    nn::os::DestroyThread( &thread2 );
    nn::os::DestroyThread( &thread3 );

    // セマフォをファイナライズする
    nn::os::FinalizeSemaphore( &g_Semaphore );
}

