﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OsMultiWaitUtility.cpp,PageSampleOsMultiWaitUtility}
 *
 * @brief
 *  多重待ち機能のサンプルプログラム
 */

/**
 * @page PageSampleOsMultiWaitUtility MultiWaitUtility
 * @tableofcontents
 *
 * @brief
 *  多重待ち機能ユーティリティのサンプルプログラムの解説です。
 *
 * @section PageSampleOsMultiWaitUtility_SectionBrief 概要
 *  ここでは、多重待ち機能ユーティリティを使ったサンプルプログラムの説明をします。
 *
 *  多重待ち機能の使い方については、多重待ち機能マニュアル および
 *  @ref nn::os "OS の関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOsMultiWaitUtility_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OsMultiWaitUtility Samples/Sources/Applications/OsMultiWaitUtility @endlink 以下にあります。
 *
 * @section PageSampleOsMultiWaitUtility_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleOsMultiWaitUtility_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOsMultiWaitUtility_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOsMultiWaitUtility_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOsMultiWaitUtility_SectionDetail 解説
 *
 * @subsection PageSampleOsMultiWaitUtility_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OsMultiWaitUtility.cpp
 *  @includelineno OsMultiWaitUtility.cpp
 *
 * @subsection PageSampleOsMultiWaitUtility_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nnMain() にて、2 つのスレッドを生成、セマフォ、イベント、メッセージキューをそれぞれ 1 つずつ初期化
 *  - MultiWaitThreadFunction() はセマフォの解放、イベントのシグナル、メッセージキューへのメッセージの送信、そして SignalThread の終了を多重待ちする
 *    - 待機したオブジェクトに対応したログを出力し、待機していたオブジェクトのシグナルをクリアする
 *    - SignalThread が終了したとき、自らも終了する
 *  - SignalThreadFunction() はセマフォの解放、イベントのシグナル、メッセージキューへのメッセージの送信を順に実行し、最後にスレッドを終了する
 *  - 2 つのスレッドが終了するまで nnMain() は待機
 *  - nnMain() にて 2 つのスレッドを破棄
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OsMultiWaitUtility_OutputExample.txt
 *
 *  SignalThreadFunction() で各種シグナルを発行し、それを多重待ちしている MultiWaitThreadFunction() にて受け、対応した関数を呼び出しログが出力されています。
 *
 *  このサンプルは @link PageSampleOsMultiWait OsMultiWait @endlink のコードを多重待ち機能ユーティリティで書き直したものです。
 *
 */

#include <stdint.h>
#include <cctype>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

//-----------------------------------------------------------------------------

namespace {

const size_t ThreadStackSize = 8192;              // スレッドのスタックサイズ
NN_OS_ALIGNAS_THREAD_STACK char  g_MultiWaitThreadStack[ ThreadStackSize ]; // 多重待ちスレッドのスタック
NN_OS_ALIGNAS_THREAD_STACK char  g_SignalThreadStack[ ThreadStackSize ];    // シグナルスレッドのスタック

// セマフォ、イベント、メッセージキュー
nn::os::SemaphoreType       g_Semaphore;
nn::os::EventType           g_Event;
nn::os::MessageQueueType    g_MessageQueue;

// メッセージキュー用
const size_t    BufferSize = 16;                // メッセージキューサイズ
uintptr_t       g_messageBuffer[BufferSize];    // メッセージキューバッファ

typedef void (*Callback)();

}   // namespace

//-----------------------------------------------------------------------------

void    DealWithThreadFinished()
{
    NN_LOG( "SignalThread was finished.\n" );
}

void    DealWithSemaphoreSignaled()
{
    NN_LOG( "Semaphore is releasing.\n" );

    // セマフォを獲得
    nn::os::AcquireSemaphore( &g_Semaphore );
}

void    DealWithEventSignaled()
{
    NN_LOG( "Event is signaling.\n" );

    // イベントをクリア
    nn::os::ClearEvent( &g_Event );
}

void    DealWithMessageQueue()
{
    NN_LOG( "Message Queue is not empty.\n" );

    // メッセーキューからメッセージを受信
    uintptr_t data;
    nn::os::ReceiveMessageQueue( &data, &g_MessageQueue);
    NN_UNUSED(data);
}

// 関数ポインタ配列
const Callback callbacks[] =
{
    DealWithSemaphoreSignaled,
    DealWithEventSignaled,
    DealWithMessageQueue,
    DealWithThreadFinished
};

//
//  多重待ちをするスレッド
//
void MultiWaitThreadFunction(void *arg)
{
    nn::os::ThreadType* pThread = reinterpret_cast<nn::os::ThreadType*>(arg);

    int count = 0;
    for (;;)
    {
        int signal = nn::os::WaitAny( &g_Semaphore,
                                      &g_Event,
                                      &g_MessageQueue & nn::os::MessageQueueWaitType_WaitForNotEmpty,
                                      pThread );

        // シグナルされた番号を添字にしてコールバック関数を呼び出す
        callbacks[signal]();
        ++count;

        // このサンプルでは、４つ目のシグナルを検知したら終了する
        if ( count >= 4 )
        {
            break;
        }
    }
}

//
//  各種同期オブジェクトをシグナルするスレッド
//
void SignalThreadFunction(void *arg)
{
    NN_UNUSED( arg );

    // セマフォを解放
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    nn::os::ReleaseSemaphore( &g_Semaphore );

    // イベントをシグナル
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    nn::os::SignalEvent( &g_Event );

    // メッセージキューにメッセージを送信
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    uintptr_t data = 0x12345678;
    nn::os::SendMessageQueue( &g_MessageQueue, data );

    // スレッド終了
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
}

//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::os::ThreadType  multiWaitThread;
    nn::os::ThreadType  signalThread;
    nn::Result          result;

    // 各種同期オブジェクトの初期化
    nn::os::InitializeSemaphore( &g_Semaphore, 0, 1 );
    nn::os::InitializeEvent( &g_Event, false, nn::os::EventClearMode_ManualClear );
    nn::os::InitializeMessageQueue( &g_MessageQueue, g_messageBuffer, BufferSize);

    // シグナルスレッドを生成する
    result = nn::os::CreateThread( &signalThread, SignalThreadFunction, NULL, g_SignalThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create signalThread." );

    // 多重待ちスレッドスレッドを生成する
    // 引数としてシグナルスレッドのスレッド構造体を渡す
    result = nn::os::CreateThread( &multiWaitThread, MultiWaitThreadFunction, &signalThread, g_MultiWaitThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority );
    NN_ASSERT( result.IsSuccess(), "Cannot create multiWaitThread." );

    // 開始
    NN_LOG( "BEGIN\n" );

    // スレッドの実行を開始する
    nn::os::StartThread( &multiWaitThread );
    nn::os::StartThread( &signalThread );

    // スレッドが終了するのを待つ
    nn::os::WaitThread( &multiWaitThread );
    nn::os::WaitThread( &signalThread );

    // 終了
    NN_LOG( "END\n" );

    // スレッドを破棄する
    nn::os::DestroyThread( &multiWaitThread );
    nn::os::DestroyThread( &signalThread );

    // 各種同期オブジェクトのファイナライズ
    nn::os::FinalizeSemaphore( &g_Semaphore );
    nn::os::FinalizeEvent( &g_Event );
    nn::os::FinalizeMessageQueue( &g_MessageQueue );
}
