﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os/os_Config.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/os/os_InterruptEvent.h>

#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>

#include <winbase.h>

#include "OsInterruptEvent-os.win32.h"

namespace nns { namespace os { namespace interruptEvent {

//---------------------------------------------------------------------------

bool   m_TimerInterruptEnabled; // タイマー割込みの許可状態
HANDLE m_Win32Event;            // 割込み通知用イベントのハンドル／AutoReset
HANDLE m_Win32TimerQueue;       // タイマーキューのハンドル
HANDLE m_Win32TimerQueueTimer;  // タイマーキュータイマーのハンドル

//---------------------------------------------------------------------------
//  割込みイベント用の Win32 プロセスの準備
//  100msec 毎にタイマーキューを起床し、必要に応じて各イベントをセットする
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//  タイマー割込みの開始
//
void StartTimerInterrupt() NN_NOEXCEPT
{
    m_TimerInterruptEnabled = true;
}

//---------------------------------------------------------------------------
//  タイマー割込みの停止
//
void StopTimerInterrupt() NN_NOEXCEPT
{
    m_TimerInterruptEnabled = false;
}



//---------------------------------------------------------------------------
//  Win32 周期タイマーのハンドラ
//
VOID WINAPI TimerInterruptHandler(PVOID param, BOOLEAN fired)
{
    NN_UNUSED(param);
    NN_UNUSED(fired);

    if (m_TimerInterruptEnabled)
    {
        ::SetEvent( m_Win32Event );
    }
}

//---------------------------------------------------------------------------
//  Win32 疑似タイマー割込みのセットアップ
//
//
class Win32PseudoTimerInterrupt
{
public:
    Win32PseudoTimerInterrupt() NN_NOEXCEPT;
    ~Win32PseudoTimerInterrupt() NN_NOEXCEPT;
};

Win32PseudoTimerInterrupt   win32PseudoTimerInterruptInstance;

//---------------------------------------------------------------------------
//  - Win32 割込み通知用イベントの作成
//  - Win32 タイマーキューの作成
//  - Win32 タイマーキュータイマーの作成（周期タイマーが SetEvent する）
Win32PseudoTimerInterrupt::Win32PseudoTimerInterrupt() NN_NOEXCEPT
{
    // Win32 割込み通知用イベントの作成
    //  （AutoReset 版でも Win32 Event は Manual にしておく必要がある。）
    m_Win32Event = ::CreateEvent( NULL,
                              TRUE  /* ManualReset */,
                              FALSE /* Initially Not Signaled */,
                              reinterpret_cast<LPCTSTR>(TimerInterruptName) );
    NN_ASSERT(m_Win32Event != NULL, "*** Cannot create Win32 Event.");

    // Win32 タイマーキューの作成
    m_Win32TimerQueue = CreateTimerQueue();
    NN_ASSERT(m_Win32TimerQueue != NULL, "*** Cannot create Win32 TimerQueue.");

    // Win32 タイマーキュータイマーの作成
    BOOL ret = CreateTimerQueueTimer( &m_Win32TimerQueueTimer,
                                      m_Win32TimerQueue,
                                      &TimerInterruptHandler,
                                      NULL,
                                      100 /* msec */,
                                      100 /* msec */,
                                      0 );
    NN_ASSERT( ret != NULL, "*** Cannot create Win32 TimerQueueTimer.");
}

//---------------------------------------------------------------------------
//  - Win32 タイマーキュータイマーの削除
//  - Win32 タイマーキューの削除
//  - Win32 割込み通知用イベントの削除
Win32PseudoTimerInterrupt::~Win32PseudoTimerInterrupt() NN_NOEXCEPT
{
    // Win32 タイマーキュータイマーの削除
    ::DeleteTimerQueueTimer( m_Win32TimerQueue, m_Win32TimerQueueTimer, NULL );

    // Win32 タイマーキューの削除
    ::DeleteTimerQueueEx( m_Win32TimerQueue, NULL );

    // Win32 割込み通知用イベントの削除
    ::CloseHandle( m_Win32Event );
}

}}} // namespace nns::os::interruptEvent

