#!/bin/bash

#------------------------------------------------------------------------------
# os/dd/init サンプルの自動実行スクリプト
#
# 本スクリプトは、VCXPROJ_LIST で定義されたディレクトリリストを対象に
# PLATFORM_LIST および CONFIGURATION_LIST の条件の全組合わせで
# ビルド＆実行を全自動で行なうものです。
#
# 最初に SigloRootMark を探してから Samples/Sources/Applications/ 以下の
# 作業を行なうため、sdk/ 以下のどのディレクトリで実行しても構いません。
#------------------------------------------------------------------------------

function usage() {
cat << END
-----------------------------------------------------------------------
スクリプト内の VCXPROJ_LIST で定義された Siglo のサンプルプログラムを
MSBuild.exe を使って全自動ビルド ＆ 実行を行なうスクリプトです。

本スクリプトは、最初に SigloRootMark を探してから、Samples/ 以下の
作業を行なうため、sdk/ 以下のどこのディレクトリで実行しても構いません。

以下にコマンドリファレンスを示します。
リリースパッケージ環境上で利用する場合は -r オプションを付加して下さい。
-----------------------------------------------------------------------

１）SDK 開発ツリー上で使用する場合

cygwin> ./BuildAndExecuteAllOsSamples.sh build
cygwin> ./BuildAndExecuteAllOsSamples.sh exe


２）リリースパッケージ上で使用する場合

cygwin> ./BuildAndExecuteAllOsSamples.sh -r build
cygwin> ./BuildAndExecuteAllOsSamples.sh -r exe

END
}

#------------------------------------------------------------------------------
# 検査対象のものを y、除外するものを n にする
#
BUILD_VS2012=y
BUILD_VS2013=y
BUILD_VS2015=n


#PATH=/cygdrive/c/Program\ Files\ \(x86\)/MSBuild/12.0/Bin:$PATH
PATH=/cygdrive/c/Program\ Files\ \(x86\)/MSBuild/14.0/Bin:$PATH
MSBUILD=MSBuild.exe
FLAGS="/maxcpucount /verbosity:minimal /nologo /target:Rebuild"

#------------------------------------------------------------------------------
# SigloRootMark もしくは NintendoSdkRootMark 位置を探す。
# 前者は SDK ツリー、後者は公開パッケージ時に配置されるダミーファイルです。
#------------------------------------------------------------------------------

function SearchSigloRootMark() {
    pushd . > /dev/null
    while [ true ]
    do
        if [ "$PWD" == "/" ];      then break; fi
        if [ "$PWD" == "$HOME" ];  then break; fi
        if [ -e "SigloRootMark" -o -e "NintendoSdkRootMark" ]; then
            SIGLO_ROOT=$PWD
            popd > /dev/null
            return
        fi
        cd ../
    done
    echo "*** SigloRootMark is not found."
    popd > /dev/null
    exit 1
}

#------------------------------------------------------------------------------
# MSBUILD を呼び出してビルドを行なうシェル関数
#------------------------------------------------------------------------------

# vcxproj 内で定義されている <Platform> 種別を並べる
PLATFORM_LIST='
Win32
x64
'

# vcxproj 内で定義されている <Configuration> 種別を並べる
CONFIGURATION_LIST=''
if [ z"$BUILD_VS2012" == z"y" ]; then
CONFIGURATION_LIST=$CONFIGURATION_LIST'
Debug
Develop
Release
'
fi
if [ z"$BUILD_VS2013" == z"y" ]; then
CONFIGURATION_LIST=$CONFIGURATION_LIST'
VS2013_Debug
VS2013_Develop
VS2013_Release
'
fi
if [ z"$BUILD_VS2015" == z"y" ]; then
CONFIGURATION_LIST=$CONFIGURATION_LIST'
VS2015_Debug
VS2015_Develop
VS2015_Release
'
fi

function DoBuildSamples() {
    CURRENT_DIR=$PWD
    LOGFILE=$CURRENT_DIR/.logfile.$$

    for PRJ in $VCXPROJ_LIST; do
      VCXPROJ=$PRJ/$PRJ-spec.Generic.vcxproj
      if [ ! -e $VCXPROJ ]; then
        echo $VCXPROJ": Not found"
        continue;
      fi
      for PF in $PLATFORM_LIST; do
        for CFG in $CONFIGURATION_LIST; do
          echo $VCXPROJ": "$PF" "$CFG
          $MSBUILD $VCXPROJ $FLAGS /p:Platform=$PF /p:Configuration=$CFG > $LOGFILE
          ERRORCODE=$?
          if [ $ERRORCODE != 0 ]; then
            cat $LOGFILE
            rm -f $LOGFILE
            exit $ERRORCODE
          fi
          rm -f $LOGFILE
        done
      done
    done
    exit 0
}

#------------------------------------------------------------------------------
# ビルド済みのサンプル実行ファイルを順次実行するシェル関数
# （リリースパッケージ用）
#------------------------------------------------------------------------------

function DoExecuteSamplesForReleasePackage() {
    for PRJ in $VCXPROJ_LIST; do
      if [ ! -e $PRJ ]; then
        echo $PRJ": Not found"
        continue;
      fi
      for PF in $PLATFORM_LIST; do
        for CFG in $CONFIGURATION_LIST; do
          echo "-------------------------------------------------------------------------------"
          echo "<< "$PRJ": "$PF" "$CFG" >>"
          echo "-------------------------------------------------------------------------------"
          EXEPATH=$SIGLO_ROOT/Samples/Sources/Applications/$PRJ/Binaries/$PF/$CFG/$PRJ.exe
          $EXEPATH
          ERRORCODE=$?
          if [ $ERRORCODE != 0 ]; then
            exit $ERRORCODE
          fi
        done
      done
    done
}

#------------------------------------------------------------------------------
# ビルド済みのサンプル実行ファイルを順次実行するシェル関数
# （SDK 開発ツリー用）
#------------------------------------------------------------------------------

# platforms 別に生成ディレクトリを並べる
PLATFORM_LIST_SDK=''
if [ z"$BUILD_VS2012" == z"y" ]; then
PLATFORM_LIST_SDK=$PLATFORM_LIST_SDK'
Win32
x64-v110
'
fi
if [ z"$BUILD_VS2013" == z"y" ]; then
PLATFORM_LIST_SDK=$PLATFORM_LIST_SDK'
Win32-v120
x64-v120
'
fi
if [ z"$BUILD_VS2015" == z"y" ]; then
PLATFORM_LIST_SDK=$PLATFORM_LIST_SDK'
Win32-v140
x64-v140
'
fi

# BUILD 別の生成ディレクトリを並べる
BUILD_LIST_SDK='
Debug
Develop
Release
'

function DoExecuteSamples() {
    for PRJ in $VCXPROJ_LIST; do
      for PFS in $PLATFORM_LIST_SDK; do
        for BLD in $BUILD_LIST_SDK; do
          echo "-------------------------------------------------------------------------------"
          echo "<< "$PRJ": "$PFS" "$BLD" >>"
          echo "-------------------------------------------------------------------------------"
          EXEPATH=$SIGLO_ROOT/Samples/Outputs/$PFS/Applications/$PRJ/$BLD/$PRJ.exe
          if [ ! -e $EXEPATH ]; then
            echo $PRJ": Not found"
            continue;
          fi
          $EXEPATH
          ERRORCODE=$?
          if [ $ERRORCODE != 0 ]; then
            exit $ERRORCODE
          fi
        done
      done
    done
}

#------------------------------------------------------------------------------
# メイン処理
#------------------------------------------------------------------------------

SearchSigloRootMark
pushd $SIGLO_ROOT/Samples/Sources/Applications/ > /dev/null
VCXPROJ_LIST=`find .  -maxdepth 1 -type d -name "Dd*" -or -type d -name "Os*" -or -type d -name "Init*" | grep -v OsAll | sed -e 's/^\.\///'`
popd > /dev/null

RELEASE_PACKAGE=false
while [ true ]
do
    case $1 in

        build)
            pushd $SIGLO_ROOT/Samples/Sources/Applications/ > /dev/null
            DoBuildSamples
            popd > /dev/null
            exit 0
            ;;

        execute | exe)
            pushd $SIGLO_ROOT/Samples/Sources/Applications/ > /dev/null
            if [ z"$RELEASE_PACKAGE" == z"true" ]; then
                DoExecuteSamplesForReleasePackage
            else
                DoExecuteSamples
            fi
            popd > /dev/null
            exit 0
            ;;

        -r )
            RELEASE_PACKAGE=true
            shift
            continue
            ;;

        * )
            usage
            exit 0
            ;;
    esac
done

