﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeMode.cpp,PageSampleOeMode}
 *
 * @brief
 *  モード変更通知のサンプルプログラム
 */

/**
 * @page PageSampleOeMode モード変更通知
 * @tableofcontents
 *
 * @brief
 *  アプリケーション動作環境操作ライブラリを用いたモード変更通知の解説です。
 *
 * @section PageSampleOeMode_SectionBrief 概要
 *  ここでは、アプリケーション動作環境操作ライブラリを用いてモード変更通知を受け取る方法を解説します。
 *
 *  アプリケーション動作環境操作ライブラリの使い方については、
 *  @ref nn::oe "アプリケーション動作環境操作ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOeMode_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeMode
 *  Samples/Sources/Applications/OeMode @endlink 以下にあります。
 *
 * @section PageSampleOeMode_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeMode_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleOeMode_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOeMode_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOeMode_SectionDetail 解説
 *
 * @subsection PageSampleOeMode_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeMode.cpp
 *  @includelineno OeMode.cpp
 *
 * @subsection PageSampleOeMode_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムは動作モードおよび性能モードの変更通知を受け取りを有効化し、変更通知メッセージを待ちます。
 *  SDEV クレードル や Switchドック の抜き差しによってモードが変更されることが確認できます。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - アプリケーション動作環境操作ライブラリを初期化。
 *  - 動作モードの変更通知を有効にする。
 *  - 性能モードの変更通知を有効にする。
 *  - 現在の動作モード、性能モードの表示。
 *  - 通知メッセージを受信し続ける。
 *      - 動作モードまたは性能モードが変更したとき、その旨と現在のモードを表示する。
 *
 *  このサンプルプログラムの実行結果を以下に示します。
 *
 *  @verbinclude  OeMode_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/oe.h>

//-----------------------------------------------------------------------------

//
//  メイン関数
//
extern "C" void nnMain()
{
    // アプリケーション動作環境操作ライブラリを初期化。
    nn::oe::Initialize();

    // 動作モードの変更通知を有効にする。
    nn::oe::SetOperationModeChangedNotificationEnabled(true);

    // 性能モードの変更通知を有効にする。
    nn::oe::SetPerformanceModeChangedNotificationEnabled(true);

    // 現在の動作モード、性能モードの表示。
    NN_LOG("Current Operation Mode is %s Mode.\n",
        nn::oe::GetOperationMode() == nn::oe::OperationMode_Handheld ? "Handheld" : "Console");
    NN_LOG("Current Performance Mode is %s Mode.\n",
        nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal ? "Normal" : "Boost");

    // 通知メッセージを受信し続ける。
    for (;;)
    {
        auto message = nn::oe::PopNotificationMessage();

        switch (message)
        {
        // 動作モードが変更された。
        case nn::oe::MessageOperationModeChanged :
            NN_LOG("Operation Mode has been changed. Current Operation Mode is %s Mode.\n",
                nn::oe::GetOperationMode() == nn::oe::OperationMode_Handheld ? "Handheld" : "Console");
            break;

        // 性能モードが変更された。
        case nn::oe::MessagePerformanceModeChanged :
            NN_LOG("Performance Mode has been changed. Current Performance Mode is %s Mode.\n",
                nn::oe::GetPerformanceMode() == nn::oe::PerformanceMode_Normal ? "Normal" : "Boost");
            break;

        default:
            // 何もしない。
            break;
        }
    }
}
