﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{OeGetLanguage.cpp,PageSampleOeGetLanguage}
 *
 * @brief
 *  言語設定取得のサンプルプログラム
 */

/**
 * @page PageSampleOeGetLanguage 言語設定取得
 * @tableofcontents
 *
 * @brief
 *  アプリケーション動作環境操作ライブラリを用いた言語設定取得のサンプルプログラムの解説です。
 *
 * @section PageSampleOeGetLanguage_SectionBrief 概要
 *  ここでは、アプリケーション動作環境操作ライブラリを用いて、言語設定を取得する方法を解説します。
 *
 *  言語設定については、「nmeta ファイルの書き方」をご参照ください。
 *
 *  アプリケーション動作環境操作ライブラリの使い方については、
 *  @ref nn::oe "アプリケーション動作環境操作ライブラリの関数リファレンス" も併せて参照して下さい。
 *
 * @section PageSampleOeGetLanguage_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/OeGetLanguage Samples/Sources/Applications/OeGetLanguage @endlink 以下にあります。
 *
 * @section PageSampleOeGetLanguage_SectionNecessaryEnvironment 必要な環境
 *  本サンプルプログラムは NX プラットフォームでのみビルドと実行が可能です。
 *
 * @section PageSampleOeGetLanguage_SectionHowToOperate 操作方法
 *  特になし。
 *
 * @section PageSampleOeGetLanguage_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleOeGetLanguage_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleOeGetLanguage_SectionDetail 解説
 *
 * @subsection PageSampleOeGetLanguage_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  OeGetLanguage.cpp
 *  @includelineno OeGetLanguage.cpp
 *
 * @subsection PageSampleOeGetLanguage_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムは言語設定を取得し、それをログに出力します。
 *  言語設定取得については nn::oe::GetDesiredLanguage() の詳細を参照して下さい。
 *
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - 設定言語と表示する文字をマップに格納。
 *  - 言語設定を取得。
 *  - 取得した言語設定を出力。
 *
 *  このサンプルプログラムの実行結果を以下に示します。(本体で Japanese を設定している場合)
 *
 *  @verbinclude  OeGetLanguage_OutputExample.txt
 *
 */

//-----------------------------------------------------------------------------
#include <map>
#include <string>
#include <nn/nn_Abort.h>
#include <nn/oe.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>
//-----------------------------------------------------------------------------

namespace
{
// 設定言語と表示する文字をマップに格納
const std::map<std::string, std::string> g_LanguageMap = {
    { "en-US", "American English" },
    { "en-GB", "British English" },
    { "ja", "Japanese" },
    { "fr", "French" },
    { "de", "German" },
    { "es-419", "LatinAmerican Spanish" },
    { "es", "Spanish" },
    { "it", "Italian" },
    { "nl", "Dutch" },
    { "fr-CA", "Canadian French" },
    { "pt", "Portuguese" },
    { "ru", "Russian" },
    { "zh-Hans", "Simplified Chinese" },
    { "zh-Hant", "Traditional Chinese" },
    { "ko", "Korean" },
};
} // namespace

extern "C" void nnMain()
{
    NN_LOG("\n\nOeGetLanguage\n");

    // 言語を取得
    const auto code = nn::oe::GetDesiredLanguage();
    const auto it = g_LanguageMap.find(code.string);
    if (it != g_LanguageMap.end())
    {
        // 取得した言語が対応している場合、言語をログ出力
        NN_LOG("%s\n\n\n", it->second.c_str());
    }
    else
    {
        // 取得した言語が対応していない場合、エラー出力
        NN_LOG("It is a language not supported.\n\n\n");
    }
}
