﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NvnTutorial10.cpp,PageSampleNvnTutorial10}
 *
 * @brief
 *  This tutorial demonstrates OpenGL/NVN interoperability through
 *  sharing texture resources.
 */

/**
 * @page PageSampleNvnTutorial10 NVN Tutorial 10: NVN and OpenGL Interoperation
 * @tableofcontents
 *
 * @brief
 *  This tutorial demonstrates OpenGL/NVN interoperability through
 *  sharing texture resources.
 *
 * @section PageSampleNvnTutorial10_SectionBrief Overview
 *  This tutorial demonstrates OpenGL/NVN interoperability through
 *  sharing texture resources. In addition to the NVN initialization
 *  that is done in previous tutorials, an EGL context is setup
 *  with a specific configuration such that texture resources may
 *  be shared between APIs. For this example, a frame buffer object
 *  is setup on the GL side and a cube is drawn into it. An NVN
 *  texture is setup using the GL texture name so that it shares
 *  the same format and storage as the GL texture. Once the GL rendering
 *  is done, the NVN side of the application copys from the interop
 *  texture into the current render target. This target is displayed
 *  on screen to show the geometry initially rendered by GL.
 *
 * @subsection PageSampleNvnTutorial10_SectionExpectedOutput Expected Output
 * @image html NvnTutorial10.png
 *
 * @section PageSampleNvnTutorial10_SectionFileStructure File Structure
 *  The main tutorial file and other files that make up the tutorial as well as the Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/NvnTutorial10OpenGLInterop Samples/Sources/Applications/NvnTutorial10OpenGLInterop @endlink
 *
 *  The tutorial library which contains common code shared by the tutorials can be found at
 *  Samples/Sources/Libraries/NvnTutorial
 *
 * @section PageSampleNvnTutorial10_SectionNecessaryEnvironment System Requirements
 *  This sample only runs on NX.
 *
 * @section PageSampleNvnTutorial10_SectionHowToOperate Operation Procedure
 *  This sample runs on its own without any additional input. Touching the
 *  touch screen on NX will exit the program.
 *
 * @section PageSampleNvnTutorial10_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleNvnTutorial10_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution and run it.
 *
 * @section PageSampleNvnTutorial10_SectionDetail Description
 *
 * @subsection PageSampleNvnTutorial10_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  NvnTutorial10.cpp
 *  @includelineno NvnTutorial10.cpp
 *
 * @subsection PageSampleNvnTutorial10_SectionSampleDetail Sample Program Description
 *  This application renders a spinning cube to a texture through GL and presents
 *  it to the screen via NVN.
 */

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <EGL/egl.h>
#include <GLES3/gl32.h>
#include <nn/util/util_Matrix.h>
#include <nvntutorial/TutorialBaseClass.h>
#include "nvntutorial/MemoryPool.h"
#include "nvntutorial/ManagedCommandBuffer.h"
#include "nvntutorial/FrameBufferManager.h"

static const int    g_NumChunks = 2;
static const size_t g_CommandMemoryChunkSize = 2048;
static const size_t g_ControlMemoryChunkSize = 2048;

static const int    g_NumNvnColorBuffers = 2;

    /* Simple vertex shader. */
static const char*  g_pVertexShaderSource   = "#version 320 es\n"
                                              "precision highp float;\n"
                                              "\n"
                                              "layout( location = 0 ) in vec3 a_position;\n"
                                              "layout( location = 1 ) in vec2 a_texCoord;\n"
                                              "\n"
                                              "uniform mat4 u_worldMtx;\n"
                                              "uniform mat4 u_camMtx;\n"
                                              "uniform mat4 u_projMtx;\n"
                                              "\n"
                                              "out vec2 v_texCoord;\n"
                                              "\n"
                                              "void main()\n"
                                              "{\n"
                                                  "gl_Position = u_projMtx * u_camMtx * u_worldMtx * vec4(a_position, 1.0);\n"
                                                  "v_texCoord = a_texCoord;\n"
                                              "}\n";

    /* Simple fragment shader that outputs the texture coordinate as color. */
static const char*  g_pFragmentShaderSource = "#version 320 es\n"
                                              "precision highp float;\n"
                                              "\n"
                                              "layout( location = 0 ) out vec4 o_Color;\n"
                                              "\n"
                                              "in vec2 v_texCoord;\n"
                                              "\n"
                                              "void main()\n"
                                              "{\n"
                                                  "o_Color = vec4(v_texCoord.xy, 0.0, 1.0);\n"
                                              "}\n";


class OpenGLNvnInterop : public TutorialBaseClass
{
    NN_DISALLOW_COPY(OpenGLNvnInterop);

public:
    OpenGLNvnInterop();
    virtual ~OpenGLNvnInterop();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual void Resize(int width, int height);

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* pUser
    );

    int                          m_ScreenWidth;
    int                          m_ScreenHeight;

        /* NVN Functions and Data */
    void InitNvn(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    void CleanUpNvn();
    void DrawNvn();

    NVNdevice                    m_Device;
    NVNqueue                     m_Queue;
    void*                        m_pQueueMemory;

    NVNtextureBuilder            m_RenderTargetBuilder;
    NVNtexture*                  m_RenderTargets[g_NumNvnColorBuffers];

    NVNtexture*                  m_pInteropTexture;

    MemoryPool*                  m_pRenderTargetMemoryPool;

    FrameBufferedSyncManager*    m_pFrameBufferedSyncManager;

    ManagedCommandBuffer*        m_pRenderTargetManagedCommandBuffer;
    NVNcommandHandle             m_RenderTargetCommandHandle;

    NVNwindow*                   m_pWindow;
    NVNwindowBuilder             m_WindowBuilder;
    int                          m_CurrentWindowIndex;
    NVNsync                      m_WindowSync;

    size_t                       m_NvnColorTargetSize;
        /* NVN Functions and Data */

        /* OpenGL Functions and Data */
    void InitEgl();
    void SetupGLShaders();
    void SetupGLVertexBuffer();
    void CleanUpEgl();

    void SetGLRenderState();
    void UpdateScene();
    void DrawGL();

    EGLDisplay                   m_EglDisplay;
    EGLConfig                    m_EglConfig;
    EGLContext                   m_EglContext;

    GLuint                       m_FBOTexture;
    GLuint                       m_FBODepth;
    GLuint                       m_FBOName;
    GLuint                       m_VertexShaderId;
    GLuint                       m_FragmentShaderId;
    GLuint                       m_ShaderProgramId;
    GLuint                       m_PositionBufferId;
    GLuint                       m_TexCoordBufferId;
    GLuint                       m_IndexBufferId;

    GLuint                       m_PositionLocation;
    GLuint                       m_TexCoordLocation;
    GLuint                       m_WorldMtxLocation;
    GLuint                       m_CamMtxLocation;
    GLuint                       m_ProjMtxLocation;

    float                        m_RotY;
    GLsizei                      m_NumIndices;

    nn::util::Matrix4x4fType     m_ProjMtx;
    nn::util::Matrix4x4fType     m_CamMtx;
    nn::util::Matrix4x4fType     m_WorldMtx;
        /* OpenGL Functions and Data */
};

/*
 * OpenGLNvnInterop Constructor
 * ----------------------------
 * Sets up default values for the members of the class.
 */
OpenGLNvnInterop::OpenGLNvnInterop() :
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_pQueueMemory(NULL),
    m_pInteropTexture(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pFrameBufferedSyncManager(NULL),
    m_pRenderTargetManagedCommandBuffer(NULL),
    m_RenderTargetCommandHandle(0),
    m_pWindow(NULL),
    m_CurrentWindowIndex(-1),
    m_NvnColorTargetSize(0),
    m_EglDisplay(NULL),
    m_EglConfig(NULL),
    m_EglContext(NULL),
    m_FBOTexture(0),
    m_FBODepth(0),
    m_FBOName(0),
    m_VertexShaderId(0),
    m_FragmentShaderId(0),
    m_ShaderProgramId(0),
    m_PositionBufferId(0),
    m_TexCoordBufferId(0),
    m_IndexBufferId(0),
    m_PositionLocation(-1),
    m_TexCoordLocation(-1),
    m_WorldMtxLocation(-1),
    m_CamMtxLocation(-1),
    m_ProjMtxLocation(-1),
    m_RotY(45.0f),
    m_NumIndices(0)
{
    for (int i = 0; i < g_NumNvnColorBuffers; ++i)
    {
        m_RenderTargets[i] = NULL;
    }

    nn::util::MatrixIdentity(&m_ProjMtx);
    nn::util::MatrixIdentity(&m_CamMtx);
    nn::util::MatrixIdentity(&m_WorldMtx);
}

/*
 * OpenGLNvnInterop Destructor
 * ---------------------------
 * Empty destructor.
 */
OpenGLNvnInterop::~OpenGLNvnInterop()
{
}

/*
 * OpenGLNvnInterop::Init
 * ----------------------
 * Initializes the Egl context, sets up the shaders and
 * vertex buffers needed for GL rendering, and initializes
 * NVN.
 */
void OpenGLNvnInterop::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
        /*
         * Initialize EGL and setup up the shaders and
         * vertex data for drawing. Currently, EGL must
         * be set up with a context made current before
         * any NVN calls are made if they are to be used
         * on the same process.
         */
    InitEgl();
    SetupGLShaders();
    SetupGLVertexBuffer();

        /* Initialize NVN. */
    InitNvn(pLoader, nativeWindow);
}

/*
 * OpenGLNvnInterop::InitNvn
 * -------------------------
 * Initializes NVN and sets up all NVN objects needed for
 * the application to run.
 */
void OpenGLNvnInterop::InitNvn(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(pLoader != NULL, "Bootstrap loader function pointer is NULL\n");

        /* Load the function pointer to get other nvn function pointers. */
    pfnc_nvnDeviceGetProcAddress = (PFNNVNDEVICEGETPROCADDRESSPROC)(*pLoader)("nvnDeviceGetProcAddress");

    if (pfnc_nvnDeviceGetProcAddress == NULL)
    {
            /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceGetProcAddress");
    }

        /*
         * Load function pointers for creating nvn device
         */
    nvnLoadCProcs(NULL, pfnc_nvnDeviceGetProcAddress);

        /* Check the API version. */
    int majorVersion, minorVersion;
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &majorVersion);
    nvnDeviceGetInteger(NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &minorVersion);

    if (majorVersion != NVN_API_MAJOR_VERSION || minorVersion < NVN_API_MINOR_VERSION)
    {
        NN_ASSERT(0, "NVN SDK not supported by current driver.");
    }

        /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
            /*
             * This can fail for a few reasons; the most likely on Horizon is
             * insufficent device memory.
             */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

        /*
         * Ensure function pointers are all loaded properly
         */
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);

        /* Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
        );
    }

        /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

        /* Sets up the memory for the queue. */
    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

        /* Create a managed Command Buffer */
    m_pRenderTargetManagedCommandBuffer = new ManagedCommandBuffer(&m_Device, g_CommandMemoryChunkSize, g_ControlMemoryChunkSize, g_NumChunks);

        /* Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_NvnColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);


        /* Allocate the render target memory. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_NvnColorTargetSize * g_NumNvnColorBuffers,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

        /* Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }

        /* Setup the window builder. */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

    m_pFrameBufferedSyncManager = new FrameBufferedSyncManager(&m_Device, &m_Queue);
    m_pFrameBufferedSyncManager->RegisterMemoryManager(m_pRenderTargetManagedCommandBuffer);
}

/*
 * OpenGLNvnInterop::CleanUpNvn
 * ----------------------------
 * Cleans up all NVN related objects.
 */
void OpenGLNvnInterop::CleanUpNvn()
{
    nvnQueueFinish(&m_Queue);

    nvnSyncFinalize(&m_WindowSync);

    if (m_pRenderTargetManagedCommandBuffer)
    {
        delete m_pRenderTargetManagedCommandBuffer;
        m_pRenderTargetManagedCommandBuffer = NULL;
    }

    if (m_pWindow != NULL)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for (int i = 0; i < g_NumNvnColorBuffers; ++i)
    {
        if (m_RenderTargets[i] != NULL)
        {
            nvnTextureFinalize(m_RenderTargets[i]);
            delete m_RenderTargets[i];
            m_RenderTargets[i] = NULL;
        }
    }

    if (m_pInteropTexture)
    {
        nvnTextureFinalize(m_pInteropTexture);
        delete m_pInteropTexture;
        m_pInteropTexture = NULL;
    }

    if (m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if (m_pFrameBufferedSyncManager != NULL)
    {
        delete m_pFrameBufferedSyncManager;
        m_pFrameBufferedSyncManager = NULL;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*
 * OpenGLNvnInterop::InitEgl
 * -------------------------
 * Initializes EGL in
 */
void OpenGLNvnInterop::InitEgl()
{
        /* Get the default EGL display. */
    m_EglDisplay = eglGetDisplay(EGL_DEFAULT_DISPLAY);

    if (m_EglDisplay == NULL)
    {
        NN_ASSERT(0, "eglGetDisplay failed.");
    }

        /* Initialize the EGL display. */
    if (eglInitialize(m_EglDisplay, 0, 0) == EGL_FALSE)
    {
        NN_ASSERT(0, "eglInitialize failed.");
    }

    EGLint configAttribs[] = {
        EGL_RENDERABLE_TYPE, EGL_OPENGL_ES3_BIT,
        EGL_RED_SIZE, 8,
        EGL_GREEN_SIZE, 8,
        EGL_BLUE_SIZE, 8,
        EGL_ALPHA_SIZE, 8,
        EGL_NONE
    };

    EGLint numConfigs = 0;

        /* Create an EGLConfig with the required attributes. */
    if ((eglChooseConfig(m_EglDisplay, configAttribs, &m_EglConfig, 1, &numConfigs) == EGL_FALSE) || (numConfigs != 1))
    {
        NN_ASSERT(0, "eglChooseConfig failed.");
        CleanUpEgl();
    }

    EGLint contextAttribs[] = {
        EGL_CONTEXT_CLIENT_VERSION, 3,
        EGL_NONE
    };

        /* Create the EGLContext based on the EGLConfig. */
    m_EglContext = eglCreateContext(m_EglDisplay, m_EglConfig, EGL_NO_CONTEXT, contextAttribs);

    if (m_EglContext == EGL_NO_CONTEXT)
    {
        NN_ASSERT(0, "eglCreateContext failed.");
        CleanUpEgl();
    }

        /* Bind the EGLContext. */
    if (eglMakeCurrent(m_EglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, m_EglContext) == EGL_FALSE)
    {
        NN_ASSERT(0, "eglMakeCurrent failed.");
        CleanUpEgl();
    }

        /* Set up the GL render state. */
    SetGLRenderState();
}

/*
 * OpenGLNvnInterop::SetupGLShaders
 * --------------------------------
 * Compiles and links a simple shader program to draw
 * the GL scene. Also queries the vertex attribute
 * and uniform locations for the shader.
 */
void OpenGLNvnInterop::SetupGLShaders()
{
    GLint result;
    GLchar shaderLog[1024];
    GLsizei shaderLogSize;

        /* Create the GL vertex shader. */
    m_VertexShaderId = glCreateShader(GL_VERTEX_SHADER);
    NN_ASSERT(m_VertexShaderId != 0, "Failed to create vertex shader");

        /* Set the shader source and compile. */
    glShaderSource(m_VertexShaderId, 1, &g_pVertexShaderSource, 0);
    glCompileShader(m_VertexShaderId);

        /* Check to see if there were any compilation errors. */
    glGetShaderiv(m_VertexShaderId, GL_COMPILE_STATUS, &result);
    if(!result)
    {
        glGetShaderInfoLog(m_VertexShaderId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile vertex shader: %s", shaderLog);
    }

        /* Create the GL fragment shader. */
    m_FragmentShaderId = glCreateShader(GL_FRAGMENT_SHADER);
    NN_ASSERT(m_FragmentShaderId != 0, "Failed to create Fragment shader");

        /* Set the shader source and compile. */
    glShaderSource(m_FragmentShaderId, 1, &g_pFragmentShaderSource, 0);
    glCompileShader(m_FragmentShaderId);

        /* Check to see if there were any compilation errors. */
    glGetShaderiv(m_FragmentShaderId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(m_FragmentShaderId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile fragment shader: %s", shaderLog);
    }

        /* Create the GL shader program. */
    m_ShaderProgramId = glCreateProgram();
    NN_ASSERT(m_ShaderProgramId != 0, "Failed to create shader program");

        /* Attach the compiled shaders and link the program. */
    glAttachShader(m_ShaderProgramId, m_VertexShaderId);
    glAttachShader(m_ShaderProgramId, m_FragmentShaderId);
    glLinkProgram(m_ShaderProgramId);

        /* Grab the attribute locations. */
    m_PositionLocation = glGetAttribLocation(m_ShaderProgramId, "a_position");
    m_TexCoordLocation = glGetAttribLocation(m_ShaderProgramId, "a_texCoord");

        /* Grab the uniform locations. */
    m_WorldMtxLocation = glGetUniformLocation(m_ShaderProgramId, "u_worldMtx");
    m_CamMtxLocation = glGetUniformLocation(m_ShaderProgramId, "u_camMtx");
    m_ProjMtxLocation = glGetUniformLocation(m_ShaderProgramId, "u_projMtx");
}

/*
 * OpenGLNvnInterop::SetupGLVertexBuffer
 * -------------------------------------
 * Setups the vertex buffers and index buffer for the cube.
 */
void OpenGLNvnInterop::SetupGLVertexBuffer()
{
        /* Position data for the cube. */
    static const GLfloat positionBufferData[] = { -0.5f,  0.5f,  0.5f,
                                                   0.5f,  0.5f,  0.5f,
                                                   0.5f,  0.5f, -0.5f,
                                                  -0.5f,  0.5f, -0.5f,
                                                   0.5f, -0.5f,  0.5f,
                                                  -0.5f, -0.5f,  0.5f,
                                                  -0.5f, -0.5f, -0.5f,
                                                   0.5f, -0.5f, -0.5f,
                                                   0.5f, -0.5f,  0.5f,
                                                   0.5f, -0.5f, -0.5f,
                                                   0.5f,  0.5f, -0.5f,
                                                   0.5f,  0.5f,  0.5f,
                                                  -0.5f, -0.5f, -0.5f,
                                                  -0.5f, -0.5f,  0.5f,
                                                  -0.5f,  0.5f,  0.5f,
                                                  -0.5f,  0.5f, -0.5f,
                                                  -0.5f, -0.5f,  0.5f,
                                                   0.5f, -0.5f,  0.5f,
                                                   0.5f,  0.5f,  0.5f,
                                                  -0.5f,  0.5f,  0.5f,
                                                   0.5f, -0.5f, -0.5f,
                                                  -0.5f, -0.5f, -0.5f,
                                                  -0.5f,  0.5f, -0.5f,
                                                   0.5f,  0.5f, -0.5f };

        /* Texture coordinates for the cube. Used as color. */
    static const GLfloat texCoordBufferData[] = { 0.0f, 1.0f,
                                                  1.0f, 1.0f,
                                                  1.0f, 0.0f,
                                                  0.0f, 0.0f,
                                                  0.0f, 1.0f,
                                                  1.0f, 1.0f,
                                                  1.0f, 0.0f,
                                                  0.0f, 0.0f,
                                                  0.0f, 1.0f,
                                                  1.0f, 1.0f,
                                                  1.0f, 0.0f,
                                                  0.0f, 0.0f,
                                                  0.0f, 1.0f,
                                                  1.0f, 1.0f,
                                                  1.0f, 0.0f,
                                                  0.0f, 0.0f,
                                                  0.0f, 1.0f,
                                                  1.0f, 1.0f,
                                                  1.0f, 0.0f,
                                                  0.0f, 0.0f,
                                                  0.0f, 1.0f,
                                                  1.0f, 1.0f,
                                                  1.0f, 0.0f,
                                                  0.0f, 0.0f };

        /* Indices for the cube. */
    static const GLuint indexBufferData[] = { 0,  1,  2,
                                              0,  2,  3,
                                              4,  5,  6,
                                              4,  6,  7,
                                              8,  9, 10,
                                              8, 10, 11,
                                             12, 13, 14,
                                             12, 14, 15,
                                             16, 17, 18,
                                             16, 18, 19,
                                             20, 21, 22,
                                             20, 22, 23 };

        /* Create the position vertex buffer. */
    glGenBuffers(1, &m_PositionBufferId);
    glBindBuffer(GL_ARRAY_BUFFER, m_PositionBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(positionBufferData), positionBufferData, GL_STATIC_DRAW);

        /* Create the texture coordinate vertex buffer. */
    glGenBuffers(1, &m_TexCoordBufferId);
    glBindBuffer(GL_ARRAY_BUFFER, m_TexCoordBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(texCoordBufferData), texCoordBufferData, GL_STATIC_DRAW);

        /* Create the index buffer. */
    glGenBuffers(1, &m_IndexBufferId);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_IndexBufferId);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER, sizeof(indexBufferData), indexBufferData, GL_STATIC_DRAW);

    m_NumIndices = sizeof(indexBufferData) / sizeof(indexBufferData[0]);
}

/*
 * OpenGLNvnInterop::CleanUpEgl
 * ----------------------------
 * Cleans up all GL related objects.
 */
void OpenGLNvnInterop::CleanUpEgl()
{
    glFinish();

    if (m_ShaderProgramId != 0 && m_VertexShaderId != 0)
    {
        glDetachShader(m_ShaderProgramId, m_VertexShaderId);
    }

    if (m_ShaderProgramId != 0 && m_FragmentShaderId != 0)
    {
        glDetachShader(m_ShaderProgramId, m_FragmentShaderId);
    }

    if (m_VertexShaderId != 0)
    {
        glDeleteShader(m_VertexShaderId);
        m_VertexShaderId = 0;
    }

    if (m_FragmentShaderId != 0)
    {
        glDeleteShader(m_FragmentShaderId);
        m_FragmentShaderId = 0;
    }

    if (m_ShaderProgramId != 0)
    {
        glDeleteProgram(m_ShaderProgramId);
        m_ShaderProgramId = 0;
    }

    if (m_PositionBufferId != 0)
    {
        glDeleteBuffers(1, &m_PositionBufferId);
        m_PositionBufferId = 0;
    }

    if (m_TexCoordBufferId != 0)
    {
        glDeleteBuffers(1, &m_TexCoordBufferId);
        m_TexCoordBufferId = 0;
    }

    if (m_IndexBufferId != 0)
    {
        glDeleteBuffers(1, &m_IndexBufferId);
        m_IndexBufferId = 0;
    }

    if (m_FBOTexture != 0)
    {
        glDeleteTextures(1, &m_FBOTexture);
        m_FBOTexture = 0;
    }

    if (m_FBODepth != 0)
    {
        glDeleteRenderbuffers(1, &m_FBODepth);
        m_FBODepth = 0;
    }

    if (m_FBOName != 0)
    {
        glDeleteFramebuffers(1, &m_FBOName);
        m_FBOName = 0;
    }

    EGLBoolean eglResult = ::eglMakeCurrent(m_EglDisplay, EGL_NO_SURFACE, EGL_NO_SURFACE, EGL_NO_CONTEXT);
    NN_ASSERT(eglResult, "eglMakeCurrent failed.");
    eglResult = ::eglTerminate(m_EglDisplay);
    NN_ASSERT(eglResult, "eglTerminate failed.");
    eglResult = ::eglReleaseThread();
    NN_ASSERT(eglResult, "eglReleaseThread failed.");
}

/*
 * OpenGLNvnInterop::Shutdown
 * --------------------------
 * Shutdown NVN and EGL.
 */
void OpenGLNvnInterop::Shutdown()
{
    CleanUpNvn();
    CleanUpEgl();
}

/*
 * OpenGLNvnInterop::SetGLRenderState
 * ----------------------------------
 * Sets up the GL render state.
 */
void OpenGLNvnInterop::SetGLRenderState()
{
        /* Enable back face culling. */
    glCullFace(GL_BACK);
    glEnable(GL_CULL_FACE);

        /* Enable the depth test. */
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);
}

/*
 * OpenGLNvnInterop::UpdateScene
 * -----------------------------
 * Updates the GL scene. Slowly rotates a cube
 * around the y-axis.
 */
void OpenGLNvnInterop::UpdateScene()
{
    nn::util::Matrix4x3fType simpleTranslateMat;
    nn::util::MatrixIdentity(&simpleTranslateMat);
    nn::util::Vector3fType simpleTranslateVector;
    nn::util::VectorSet(&simpleTranslateVector, 0.0f, 0.0f, -3.0f);
    nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

    nn::util::Matrix4x3fType simpleRotateXMat;
    nn::util::MatrixIdentity(&simpleRotateXMat);
    nn::util::Vector3fType simpleRotateXVector;
    nn::util::VectorSet(&simpleRotateXVector, 30.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

    nn::util::Matrix4x3fType simpleRotateYMat;
    nn::util::MatrixIdentity(&simpleRotateYMat);
    nn::util::Vector3fType simpleRotateYVector;
    nn::util::VectorSet(&simpleRotateYVector, 0.0f, m_RotY * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

    nn::util::Matrix4x3fType simpleModelMat;
    nn::util::Matrix4x3fType tempMat1;
    nn::util::MatrixMultiply(&tempMat1, simpleRotateYMat, simpleRotateXMat);
    nn::util::MatrixMultiply(&simpleModelMat, tempMat1, simpleTranslateMat);
    nn::util::MatrixConvert(&m_WorldMtx, simpleModelMat);

    m_RotY += 0.15f;
}

/*
 * OpenGLNvnInterop::DrawGL
 * ------------------------
 * Draws the GL scene to the frame buffer object
 * that contains the texture being used for the
 * NVN interop. The glFinish call at the end of
 * the function is required for the interop to
 * work properly.
 */
void OpenGLNvnInterop::DrawGL()
{
        /* Bind the FBO with the interop texture. */
    glBindFramebuffer(GL_FRAMEBUFFER, m_FBOName);

        /* Set the viewport. */
    glViewport(0, 0, m_ScreenWidth, m_ScreenHeight);

        /* Clear the interop texture and the depth buffer. */
    glClearColor(0.4f, 0.55f, 0.6f, 1.0f);
    glClearDepthf(1.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

        /* Use the compiled shader program. */
    glUseProgram(m_ShaderProgramId);

        /* Set the shader uniforms. */
    nn::util::Float4x4 worldMtx;
    nn::util::MatrixStore(&worldMtx, m_WorldMtx);

    nn::util::Float4x4 camMtx;
    nn::util::MatrixStore(&camMtx, m_CamMtx);

    nn::util::Float4x4 projMtx;
    nn::util::MatrixStore(&projMtx, m_ProjMtx);

    glUniformMatrix4fv(m_WorldMtxLocation, 1, GL_FALSE, reinterpret_cast<GLfloat*>(&worldMtx));
    glUniformMatrix4fv(m_CamMtxLocation, 1, GL_FALSE, reinterpret_cast<GLfloat*>(&camMtx));
    glUniformMatrix4fv(m_ProjMtxLocation, 1, GL_FALSE, reinterpret_cast<GLfloat*>(&projMtx));

        /* Bind the position vertex buffer. */
    glEnableVertexAttribArray(m_PositionLocation);
    glBindBuffer(GL_ARRAY_BUFFER, m_PositionBufferId);
    glVertexAttribPointer(m_PositionLocation, 3, GL_FLOAT, GL_FALSE, 0, 0);

        /* Bind the texture coordinate vertex buffer. */
    glEnableVertexAttribArray(m_TexCoordLocation);
    glBindBuffer(GL_ARRAY_BUFFER, m_TexCoordBufferId);
    glVertexAttribPointer(m_TexCoordLocation, 2, GL_FLOAT, GL_FALSE, 0, 0);

        /* Bind the index buffer. */
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_IndexBufferId);

        /* Draw the cube. */
    glDrawElements(GL_TRIANGLES, m_NumIndices, GL_UNSIGNED_INT, nullptr);

        /* Unbind bound resources. */
    glDisableVertexAttribArray(m_PositionLocation);
    glDisableVertexAttribArray(m_TexCoordLocation);
    glUseProgram(0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
    glBindFramebuffer(GL_FRAMEBUFFER, 0);

        /* Check for any OpenGL errors. */
    GLenum error = glGetError();
    if (error != GL_NO_ERROR)
    {
        NN_ASSERT(0, "GL Error Occurred");
    }

        /*
         * glFinish needs to be called before using the
         * interop texture on the NVN side.
         */
    glFinish();
}

/*
 * OpenGLNvnInterop::DrawNvn
 * -------------------------
 * Copies the interop texture into the current on screen
 * render target and presents it to the screen.
 */
void OpenGLNvnInterop::DrawNvn()
{
        /* Get next render target to be used */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        /* Record the command buffer to set and clear the target. */
    m_pRenderTargetManagedCommandBuffer->BeginRecording();
    {
        NVNcommandBuffer* pCommandBuffer = m_pRenderTargetManagedCommandBuffer->GetCommandBuffer();

        nvnCommandBufferSetRenderTargets(pCommandBuffer, 1, &m_RenderTargets[m_CurrentWindowIndex], NULL, NULL, NULL);

        nvnCommandBufferSetScissor(pCommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
        nvnCommandBufferSetViewport(pCommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

        float clear_color[4] = { 1.0f, 0.0f, 0.0f, 1.0f };
        nvnCommandBufferClearColor(pCommandBuffer, 0, clear_color, NVN_CLEAR_COLOR_MASK_RGBA);

        NVNcopyRegion copyRegion;
        memset(&copyRegion, 0, sizeof(copyRegion));
        copyRegion.width = m_ScreenWidth;
        copyRegion.height = m_ScreenHeight;
        copyRegion.depth = 1;

            /* Copy the interop texture to the current render target. */
        nvnCommandBufferCopyTextureToTexture(pCommandBuffer, m_pInteropTexture, NULL, &copyRegion, m_RenderTargets[m_CurrentWindowIndex], NULL, &copyRegion, 0);
    }
    m_RenderTargetCommandHandle = m_pRenderTargetManagedCommandBuffer->EndRecording();

        /* Insert a fence in the sync manager. */
    m_pFrameBufferedSyncManager->InsertFence();

        /*
         * Wait on the window sync before using the render target.
         */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Submit command buffer for switching render target. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

    nvnQueuePresentTexture(&m_Queue, m_pWindow, m_CurrentWindowIndex);

    m_pFrameBufferedSyncManager->SwapPools();
}

/*
 * OpenGLNvnInterop::Draw
 * ----------------------
 * Updates the scene and calls the GL and NVN
 * draw functions.
 */
void OpenGLNvnInterop::Draw(uint64_t millisec)
{
    UpdateScene();
    DrawGL();
    DrawNvn();
}

/*
 * OpenGLNvnInterop::Resize
 * ------------------------
 * This method is called when the screen is resized. It
 * frees any old GL/NVN objects that depends on the screen
 * size and reallocates them with the newly given size.
 */
void OpenGLNvnInterop::Resize(int width, int height)
{
    m_ScreenWidth = width;
    m_ScreenHeight = height;

        /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

        /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
        /*
         * Otherwise finalize (free) the NVNwindow used for the previous window size.
         * The NVNWindow must be finalized before a render target it owns is finalized.
         */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

        /* Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    for (int i = 0; i < g_NumNvnColorBuffers; ++i)
    {
            /* If it's the first time Resize is called, allocate the render target. */
        if (m_RenderTargets[i] == NULL)
        {
            m_RenderTargets[i] = new NVNtexture;
        }
            /* Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_RenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_NvnColorTargetSize * i);

            /* Create the texture using the current state of the texture builder. */
        nvnTextureInitialize(m_RenderTargets[i], &m_RenderTargetBuilder);
    }

        /* Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumNvnColorBuffers, m_RenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

        /* Delete old frame buffer information */
    if (m_FBOName != 0)
    {
        glDeleteFramebuffers(1, &m_FBOName);
    }

    if (m_FBOTexture != 0)
    {
            /* Make sure nvn is done using the texture before deleting. */
        nvnQueueFinish(&m_Queue);
        glDeleteTextures(1, &m_FBOTexture);
    }

    if (m_FBODepth != 0)
    {
        glDeleteRenderbuffers(1, &m_FBODepth);
    }

        /*
         * Generate texture for the GL frame buffer object.
         * This is the texture that will be used for the
         * OpenGL/NVN interop.
         */
    glGenTextures(1, &m_FBOTexture);

        /* Bind the texture. */
    glBindTexture(GL_TEXTURE_2D, m_FBOTexture);

        /*
         * Setup the attributes and storage for the texture.
         * All of the texture attributes must have NVN
         * equivalents and the texture must be immutable.
         */
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_BORDER);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_BORDER);
    glTexStorage2D(GL_TEXTURE_2D, 1, GL_RGBA8, m_ScreenWidth, m_ScreenHeight);

        /*
         * Check to make sure that the FBO texture is an immutable
         * format texture.
         */
    GLint immutable = 0;
    glGetTexParameteriv(GL_TEXTURE_2D, GL_TEXTURE_IMMUTABLE_FORMAT, &immutable);

    if (immutable != 1)
    {
        NN_ASSERT(0, "GL side of texture interop must be immutable");
    }

        /* Unbind the texture. */
    glBindTexture(GL_TEXTURE_2D, 0);

        /* Create the GL frame buffer and attach the texture. */
    glGenFramebuffers(1, &m_FBOName);
    glBindFramebuffer(GL_FRAMEBUFFER, m_FBOName);
    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, m_FBOTexture, 0);

        /* Create and attach a depth buffer to the frame buffer. */
    glGenRenderbuffers(1, &m_FBODepth);
    glBindRenderbuffer(GL_RENDERBUFFER, m_FBODepth);
    glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH_COMPONENT24, m_ScreenWidth, m_ScreenHeight);
    glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, m_FBODepth);

        /* Check whether the frame buffer was setup properly. */
    GLenum frameBufferStatus = glCheckFramebufferStatus(GL_FRAMEBUFFER);

    if (frameBufferStatus != GL_FRAMEBUFFER_COMPLETE)
    {
        NN_ASSERT(0, "Failed to create GL frame buffer");
    }

    glBindFramebuffer(GL_FRAMEBUFFER, 0);

        /* Create the NVNtexture for the interop. */
    if (m_pInteropTexture == NULL)
    {
        m_pInteropTexture = new NVNtexture;
    }
    else
    {
        nvnTextureFinalize(m_pInteropTexture);
    }

        /* Setup the texture builder for the interop texture. */
    NVNtextureBuilder textureBuilder;
    nvnTextureBuilderSetDefaults(&textureBuilder);
    nvnTextureBuilderSetDevice(&textureBuilder, &m_Device);

        /*
         * NVN Texture Builder: SetGLTextureName
         * -------------------------------------
         * This function allows an NVN texture to be initialized
         * using the state and texel storage of the given GL
         * texture. If this function is called with a non-zero
         * texture name, any other state set in the texture
         * builder will be ignored with the exception of the
         * NVN device. The given GL texture name will be used to
         * look up a GL texture using the currently bound GL
         * context. If no texture is found, the result of
         * nvnTextureInitialize is undefined. The GL texture
         * should not be destroyed while the NVN texture is
         * still in use. There is no cross API synchronization
         * so it is the responsibility of the application to
         * keep texture accesses syncrhonized.
         *
         * Restrictions on the GL Texture:
         * -------------------------------
         * - The GL texture must be an immutable-format texture.
         *
         * - All GL texture attributes (e.g., target, internal
         *   format, etc.) must have NVN equivalents.
         *
         * - Buffer textures are not supported.
         *
         * - Sparse textures are not supported.
         */
    nvnTextureBuilderSetGLTextureName(&textureBuilder, m_FBOTexture);

        /* Initialize the interop texture with the texture builder. */
    if (nvnTextureInitialize(m_pInteropTexture, &textureBuilder) == NVN_FALSE)
    {
        NN_ASSERT(0, "Failed to create texture");
    }

    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&m_ProjMtx, 60.0f * 3.14159f / 180.0f, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);
}


// static
void OpenGLNvnInterop::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
)
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);

    NN_ASSERT(0, "NVN Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static OpenGLNvnInterop tut;
    return (&tut);
}

extern "C" void nnMain()
{
    TutorialRun();
}
