﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NvnTutorial05.cpp,PageSampleNvnTutorial05}
 *
 * @brief
 *  This tutorial builds off of Tutorial03 and shows various
 *  combinations of render states that can be used while
 *  rendering geometry as well as a way to render debug
 *  text to the screen.
 */

/**
 * @page PageSampleNvnTutorial05 NVN Tutorial 05: Setting Up Render State And Rendering Debug Text
 * @tableofcontents
 *
 * @brief
 *  This tutorial builds off of Tutorial03 and shows various
 *  combinations of render states that can be used while
 *  rendering geometry as well as a way to render debug
 *  text to the screen.
 *
 * @section PageSampleNvnTutorial05_SectionBrief Overview
 *  This tutorial builds off of Tutorial03 and shows
 *  various combinations of render states that can be
 *  used while rendering geometry as well as a way to
 *  render debug text to the screen. The tutorial renders
 *  the model from Tutorial03 and the render state changes
 *  over time. The render states used are an opaque state,
 *  a wirefram state, an alpha state, and a front face
 *  culling state.
 *
 *  The debug text is rendered using a helper class
 *  defined in the DebugTextRenderer.h/.cpp files. The helper
 *  class provides functionality to set the text's size
 *  relative to the screen, write to a specific line,
 *  start from a specific column, and more. The text
 *  renderer accumulates text to draw over the frame and
 *  will draw it to a provided render target.
 *
 * @subsection PageSampleNvnTutorial05_SectionExpectedOutput Expected Output
 * @image html NvnTutorial05.png
 *
 * @section PageSampleNvnTutorial05_SectionFileStructure File Structure
 *  The main tutorial file and Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/NvnTutorial05SettingUpRenderState Samples/Sources/Applications/NvnTutorial05SettingUpRenderState @endlink
 *
 *  The tutorial library which contains common code shared by the tutorials can be found at
 *  Samples/Sources/Libraries/NvnTutorial
 *
 * @section PageSampleNvnTutorial05_SectionNecessaryEnvironment System Requirements
 *  No extra system requirements.
 *
 * @section PageSampleNvnTutorial05_SectionHowToOperate Operation Procedure
 *  This sample runs on its own without any additional input. Hitting escape on Windows
 *  or touching the touch screen on NX will exit the program.
 *
 * @section PageSampleNvnTutorial05_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleNvnTutorial05_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution in the desired configuration and run it.
 *
 * @section PageSampleNvnTutorial05_SectionDetail Description
 *
 * @subsection PageSampleNvnTutorial05_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  NvnTutorial05.cpp
 *  @includelineno NvnTutorial05.cpp
 *
 * @subsection PageSampleNvnTutorial05_SectionSampleDetail Sample Program Description
 *  This tutorial renders the same spinning textured cube as
 *  Tutorial03, but the render states used for the cube are
 *  changed every 180 frames. There are four sets of states used:
 *  opaque, wire frame, alpha, and cull front. The debug text shows
 *  the current state name, the current configuration of render states,
 *  and a frame counter.
 */

#include <cstdio>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>

#ifdef _WIN32
    #ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
    #endif
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include <nn/nn_Windows.h>
#endif

#include <nvntutorial/TutorialBaseClass.h>
#include <nvntutorial/AssetFileLoadingHelper.h>
#include <nvntutorial/TutorialUtil.h>
#include <nvntutorial/DebugTextRenderer.h>
#include <nvntutorial/ShaderHeaders/SimpleTexturedModelAlphaDataHelper.h>
#include <nvntutorial/TextureIDManager.h>
#include <nvntutorial/UniformBuffer.h>

static const size_t g_CommandMemorySize = 1024;
static const size_t g_ControlMemorySize = 1024;
static const int    g_NumColorBuffers = 2;

class SettingUpRenderState : public TutorialBaseClass
{
    NN_DISALLOW_COPY(SettingUpRenderState);

public:
    SettingUpRenderState();
    virtual ~SettingUpRenderState();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual void Resize(int width, int height);

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* user
        );

    void UpdateCommandBuffer();
    void SetupModelUniformBlock();
    void UpdateUniformBlock(const nn::util::Matrix4x4fType& modelMat, const nn::util::Matrix4x4fType& projMat, const NVNtextureHandle handle);
    void SetDefaultState();
    void SetStates(uint32_t frame);

    int UpdateRenderTargets();

    NVNdevice               m_Device;
    NVNqueue                m_Queue;
    void*                   m_pQueueMemory;

    MemoryPool*             m_pCommandMemoryPool;

    ptrdiff_t               m_CommandPoolOffset;
    void*                   m_pControlPool;
    NVNcommandBuffer        m_CommandBuffer;
    NVNcommandHandle        m_CommandHandle;

    NVNtextureBuilder       m_RenderTargetBuilder;
    NVNtexture*             m_pRenderTargets[g_NumColorBuffers];
    NVNtexture*             m_pDepthTarget;

    NVNsamplerBuilder       m_SamplerBuilder;
    NVNbufferBuilder        m_BufferBuilder;

    NVNsync                 m_RenderTargetPresentSync;
    NVNsync                 m_CommandBufferSync;
    AssetFileLoadingHelper* m_pAssetLoader;
    AssetFileDataHolder*    m_pDataHolder;

    NVNblendState           m_BlendState;
    NVNchannelMaskState     m_ChannelMaskState;
    NVNcolorState           m_ColorState;
    NVNdepthStencilState    m_DepthStencilState;
    NVNmultisampleState     m_MultisampleState;
    NVNpolygonState         m_PolygonState;

    MemoryPool*             m_pUniformBufferMemoryPool;
    UniformBuffer           m_UniformBlockVertex;
    UniformBuffer           m_UniformBlockFragment;
    NVNsync                 m_UniformBlockUpdateSync;

    TextureIDManager*       m_pTextureIDManager;

    int                     m_ScreenWidth;
    int                     m_ScreenHeight;

    DebugTextRenderer*      m_pDebugTextRenderer;

    bool                    m_State1;
    bool                    m_State2;
    bool                    m_State3;
    bool                    m_State4;
    bool                    m_ChangeState;

    MemoryPool*             m_pRenderTargetMemoryPool;

    MemoryPool*             m_pShaderScratchMemoryPool;
    size_t                  m_ShaderScratchSize;

    void*                   m_pRenderTargetControlPool;
    ptrdiff_t               m_RenderTargetCommandPoolOffset;
    NVNcommandBuffer        m_RenderTargetCommandBuffer;
    NVNcommandHandle        m_RenderTargetCommandHandle;

    NVNwindow*              m_pWindow;
    NVNwindowBuilder        m_WindowBuilder;
    int                     m_CurrentWindowIndex;
    NVNsync                 m_WindowSync;

    size_t                  m_ColorTargetSize;
    size_t                  m_DepthTargetSize;
};

/*
 * SettingUpRenderState Constructor
 * --------------------------------
 * Sets up default values for member data.
 */
SettingUpRenderState::SettingUpRenderState() :
    m_pQueueMemory(NULL),
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_CommandHandle(NULL),
    m_pDepthTarget(NULL),
    m_pAssetLoader(NULL),
    m_pDataHolder(NULL),
    m_pUniformBufferMemoryPool(NULL),
    m_pTextureIDManager(NULL),
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_pDebugTextRenderer(NULL),
    m_State1(false),
    m_State2(false),
    m_State3(false),
    m_State4(false),
    m_ChangeState(false),
    m_pRenderTargetMemoryPool(NULL),
    m_pShaderScratchMemoryPool(NULL),
    m_ShaderScratchSize(0),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pWindow(NULL),
    m_CurrentWindowIndex(NULL),
    m_ColorTargetSize(0),
    m_DepthTargetSize(0)
{
    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_pRenderTargets[i] = NULL;
    }
}

/*
 * SettingUpRenderState::Init
 * --------------------------
 * Initialize NVN, load asset files, and create objects needed for the
 * application to run.
 */
void SettingUpRenderState::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(pLoader);

        /* Load the function pointer to get other nvn function pointers. */
    pfnc_nvnDeviceGetProcAddress = (PFNNVNDEVICEGETPROCADDRESSPROC) (*pLoader)("nvnDeviceGetProcAddress");

    if (pfnc_nvnDeviceGetProcAddress == NULL)
    {
            /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceGetProcAddress");
    }

        /*
         * Load function pointers for creating nvn device
         */
    nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress );

        /* Check the API version. */
    int majorVersion, minorVersion;
    nvnDeviceGetInteger( NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &majorVersion );
    nvnDeviceGetInteger( NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &minorVersion );
    if( majorVersion != NVN_API_MAJOR_VERSION || minorVersion < NVN_API_MINOR_VERSION )
    {
        NN_ASSERT( 0, "NVN SDK not supported by current driver." );
    }

        /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
            /*
             * This can fail for a few reasons; the most likely on Horizon is
             * insufficent device memory.
             */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

        /*
         * Ensure function pointers are all loaded properly
         */
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress );

        /* Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

        /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

        /* Initialize the rendering command buffer. */
    if (nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device) == NULL)
    {
        NN_ASSERT(0, "nvnCommandBufferInitialize");
    }

        /*
         * Queries the device for the proper control and command
         * memory alignment for a command buffer.
         */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /* Setup the command memory for the command buffers. */
    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(NULL, Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, &m_Device);

        /* Grab the offset from the memory pool. */
    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

        /* Allocate control memory for the command buffer. */
    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /* Add the memory to the command buffer. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        /* Render Target Setting Command Buffer. */
    if (!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize render target command buffer");
    }

        /* Grab the offset from the memory pool. */
    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

        /* Allocate control memory for the command buffer. */
    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /* Add the memory to the command buffer. */
    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

        /* Initialize the render target sync. */
    if (!nvnSyncInitialize(&m_RenderTargetPresentSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize render target present sync");
    }

        /* Initialize the rendering command buffer sync. */
    if (!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize command buffer sync");
    }

        /* Initialize the uniform buffer update sync. */
    if (!nvnSyncInitialize(&m_UniformBlockUpdateSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize uniform block update sync");
    }

        /*! Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }
        /*
         * NVN Blend State
         * ---------------
         * This render state controls the color blending equations
         * used for a single color buffer.
         *
         * SetBlendTarget   - Specifies the index of the color buffer
         *                    that the blend state is associated with.
         *
         * SetBlendFunc     - Speficies the blend functions for the
         *                    source/destination color and alpha channels.
         *
         * SetBlendEquation - Specifies the blend equation for the color
         *                    and alpha channels.
         */
    nvnBlendStateSetDefaults(&m_BlendState);

        /* NVN Channel Mask State
         * -----------------------
         * Enables/disables color channel writes for a single color buffer.
         */
    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);

        /* NVN Color State
         * ----------------
         * Controls processing of color values.
         *
         * SetBlendEnable - Enables/disables blending for a single
         *                  color buffer.
         *
         * SetLogicOp     - Specifies the logical operation to perform
         *                  for all color targets.
         *
         * SetAlphaTest   - Specifies a comparison function to use for
         *                  alpha testing.
         */
    nvnColorStateSetDefaults(&m_ColorState);

        /* NVN Depth Stencil State
         * ------------------------
         * Controls the depth testing and stencil testing operations.
         *
         * SetDepthTestEnable   - Specifies whether depth testing is enabled.
         *
         * SetDepthWriteEnable  - Specifies whether the depth buffer will be
         *                        updated by the depth test.
         *
         * SetDepthFunc         - Sets the comparison function used by the
         *                        depth test. (NVNdepthFunc)
         *
         * SetStencilTestEnable - Specifies whether stencil testing is enabled.
         *
         * SetStencilFunc       - Sets the comparison function used by the
         *                        stencil test. (NVNstencilFunc)
         *
         * SetStencilOp         - Specifies operations performed on stored stencil
         *                        values by the stencil test.
         */
    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);

        /* NVN Multisample State
         * -----------------------
         * Controls multisample rasterization and sample processing.
         *
         * SetMultisampleEnable     - Enables/disables multisampling.
         *
         * SetSamples               - Specify number of sampler in the framebuffer.
         *
         * SetAlphaToCoverageEnable - Enable or disable alpha-to-coverage operation.
         */
    nvnMultisampleStateSetDefaults(&m_MultisampleState);

        /* NVN Polygon State
         * -----------------------
         * Controls the rasterization settings for rendering polygons.
         *
         * SetCullFace             - Specifies whether the front, back, or no face
         *                           should be culled.
         *
         * SetFrontFace            - Specifies whether clockwise or counter-clockwise
                                     polygons are considered front-facing.
         *
         * SetPolygonMode          - Specifies whether polygons are rendered as points,
         *                           lines, or triangles.
         *
         * SetPolygonOffsetEnables - Takes a bitfield that specifies whether polygon
         *                           offset is applied for point-mode, line-mode, and
         *                           filled polygons.
         */
    nvnPolygonStateSetDefaults(&m_PolygonState);

        /* Builders */
    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    nvnBufferBuilderSetDevice(&m_BufferBuilder, &m_Device);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    m_DepthTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

        /* Set up the render target memory pool. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers + m_DepthTargetSize,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

        /* Create the texture ID manager. */
    m_pTextureIDManager = new TextureIDManager(&m_Device);

        /* Set up the uniform block for the cube. */
    SetupModelUniformBlock();

    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    m_pAssetLoader = new AssetFileLoadingHelper(&m_Device, m_pTextureIDManager);

        /* Load in the asset file for the tutorial. */
    m_pDataHolder = m_pAssetLoader->LoadAssetFile("cubeAlpha.out");
    m_pDataHolder->SetupAttributeStatesNVN(SimpleTexturedModelAlpha::Attributes::GetAttributeLocation);
    m_pDataHolder->SetupTextureSamplerHandle(&m_Device, m_pTextureIDManager, &m_SamplerBuilder);

    m_ShaderScratchSize = m_pDataHolder->GetProgramData()[0]->m_ShaderScratchMemorySize;

    if (m_ShaderScratchSize != 0)
    {
        m_pShaderScratchMemoryPool = new MemoryPool();
        m_pShaderScratchMemoryPool->Init(nullptr,
                                         m_ShaderScratchSize,
                                         NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
                                         &m_Device);
    }

        /* Get an instance of the text renderer singleton and initialize it. */
    m_pDebugTextRenderer = DebugTextRenderer::GetInstance();
    m_pDebugTextRenderer->Init(&m_Device, &m_Queue, m_pTextureIDManager);
    m_pDebugTextRenderer->SetDrawEnable(true);
    m_pDebugTextRenderer->SetSpacing(false);

        /* Set the default (opaque, filled triangles) render state. */
    SetDefaultState();

        /* Initial NVNwindow setup. */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

    nn::fs::Unmount("rom");
    delete[] mountRomCacheBuffer;
}//NOLINT(impl/function_size)

/*
 * SettingUpRenderState::SetDefaultState
 * -------------------------------------
 * This function sets up the render states in the
 * most common configuration for rendering simple
 * geometry (opaque, filled primitives, depth
 * testing, etc).
 */
void SettingUpRenderState::SetDefaultState()
{
        /* Setting up the depth stencil state. */
    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);
        /* Enable the depth test. */
    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
        /* Enable writing to the depth buffer. */
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);

        /*
         * Set the depth function such that the
         * fragment passes the depth test if its
         * depth value is less than what is currently
         * in the depth buffer.
         */
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVN_DEPTH_FUNC_LESS);

        /* Setting up the polygon state. */
    nvnPolygonStateSetDefaults(&m_PolygonState);

        /*
         * Set the front face to be the one with the
         * counter-clockwise winding order.
         */
    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CCW);

        /* Set the face to be culled to the back face. */
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);

        /* Set the polygons to be filled. */
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_FILL);

        /* Set the default multisample state. */
    nvnMultisampleStateSetDefaults(&m_MultisampleState);

        /* Setting up the color state. */
    nvnColorStateSetDefaults(&m_ColorState);

        /* Turn off color blending. */
    nvnColorStateSetBlendEnable(&m_ColorState, 0, NVN_FALSE);

        /* Set the default channel mask state. */
    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);

        /* Setting up the blend state. */
    nvnBlendStateSetDefaults(&m_BlendState);

        /* Set some reasonable default alpha and color blend functions. */
    nvnBlendStateSetBlendFunc(&m_BlendState, NVNblendFunc::NVN_BLEND_FUNC_SRC_COLOR, NVNblendFunc::NVN_BLEND_FUNC_DST_COLOR, NVNblendFunc::NVN_BLEND_FUNC_SRC_ALPHA, NVNblendFunc::NVN_BLEND_FUNC_ONE_MINUS_SRC_ALPHA);
}

/*
 * SettingUpRenderState::SetStates
 * -------------------------------
 * This functions sets different render states
 * according to the range of values the current
 * frame is within.
 */
void SettingUpRenderState::SetStates(uint32_t frame)
{
        /* Set up the debug text grid size. */
    m_pDebugTextRenderer->SetGridSize(120.0f, 36.0f);

    uint32_t segment = frame % 720;

    std::string name;

        /* Segment from 0-179 set the default (opaque) state. */
    if(segment < 180)
    {
            /*
             * If we're not already in the state,
             * change the render states and state bools.
             */
        if (m_State1 == NULL)
        {
            m_ChangeState = m_State1 = true;
            m_State2 = m_State3 = m_State4 = false;

                /* Just set the default state. */
            SetDefaultState();
        }
        else
        {
            m_ChangeState = false;
        }

        name = "Opaque State";

            /* Print out the current relevant state values. */
        m_pDebugTextRenderer->Printf(0, 5, "Depth Test Enable: NVN_TRUE");
        m_pDebugTextRenderer->Printf(0, 6, "Polygon Mode:      NVN_POLGYON_MODE_FILL");
        m_pDebugTextRenderer->Printf(0, 7, "Blend Enable:      NVN_FALSE");
        m_pDebugTextRenderer->Printf(0, 8, "Cull Face:         NVN_FACE_BACK");
    }
        /* Segment from 180-359 set the wire frame state. */
    else if(segment < 360)
    {
            /*
             * If we're not already in the state,
             * change the render states and state bools.
             */
        if (m_State2 == NULL)
        {
            m_ChangeState = m_State2 = true;
            m_State1 = m_State3 = m_State4 = false;
            SetDefaultState();

                /* Don't cull and set the mode to draw lines. */
            nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_NONE);
            nvnPolygonStateSetPolygonMode(&m_PolygonState, NVN_POLYGON_MODE_LINE);
        }
        else
        {
            m_ChangeState = false;
        }

        name = "Wire Frame State";

            /* Print out the current relevant state values. */
        m_pDebugTextRenderer->Printf(0, 5, "Depth Test Enable: NVN_FALSE");
        m_pDebugTextRenderer->Printf(0, 6, "Polygon Mode:      NVN_POLGYON_MODE_LINE");
        m_pDebugTextRenderer->Printf(0, 7, "Blend Enable:      NVN_FALSE");
        m_pDebugTextRenderer->Printf(0, 8, "Cull Face:         NVN_FACE_NONE");
    }
        /* Segment from 360-539 set the alpha state. */
    else if(segment < 540)
    {
            /*
             * If we're not already in the state,
             * change the render states and state bools.
             */
        if (m_State3 == NULL)
        {
            m_ChangeState = m_State3 = true;
            m_State1 = m_State2 = m_State4 = false;
            SetDefaultState();

                /* Don't cull */
            nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_NONE);

                /* Disable depth test. */
            nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_FALSE);

                /* Enable color blending. */
            nvnColorStateSetBlendEnable(&m_ColorState, 0, NVN_TRUE);
        }
        else
        {
            m_ChangeState = false;
        }

        name = "Alpha State";

            /* Print out the current relevant state values. */
        m_pDebugTextRenderer->Printf(0, 5, "Depth Test Enable: NVN_FALSE");
        m_pDebugTextRenderer->Printf(0, 6, "Polygon Mode:      NVN_POLGYON_MODE_FILL");
        m_pDebugTextRenderer->Printf(0, 7, "Blend Enable:      NVN_TRUE");
        m_pDebugTextRenderer->Printf(0, 8, "Cull Face:         NVN_FACE_NONE");
    }
        /* Segment from 540-719 set the alpha state. */
    else
    {
            /*
             * If we're not already in the state,
             * change the render states and state bools.
             */
        if(m_State4 == NULL)
        {
            m_ChangeState = m_State4 = true;
            m_State1 = m_State2 = m_State3 = false;
            SetDefaultState();

                /* Cull the front facing polygons. */
            nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_FRONT);
        }
        else
        {
            m_ChangeState = false;
        }

        name = "Cull Front State";

            /* Print out the current relevant state values. */
        m_pDebugTextRenderer->Printf(0, 5, "Depth Test Enable: NVN_TRUE");
        m_pDebugTextRenderer->Printf(0, 6, "Polygon Mode:      NVN_POLGYON_MODE_FILL");
        m_pDebugTextRenderer->Printf(0, 7, "Blend Enable:      NVN_FALSE");
        m_pDebugTextRenderer->Printf(0, 8, "Cull Face:         NVN_FACE_FRONT");
    }

    std::string elipses;
    uint32_t second = ((segment % 180) / 60);

    for(uint32_t i = 0; i < second + 1; ++i)
    {
        name += " .";
    }

        /* Print out the state name. */
    m_pDebugTextRenderer->SetGridSize(90.0f, 24.0f);
    m_pDebugTextRenderer->Printf(36, 0, name.c_str(), frame);
}

/*
 * SettingUpRenderState::UpdateCommandBuffer
 * -----------------------------------------
 * Builds the command buffer to render the cube.
 */
void SettingUpRenderState::UpdateCommandBuffer()
{
        /* Sync to safely overwrite memory. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Reset the command buffer memory. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        /* Starts the recording of a new set of commands for the given command buffer. */
    nvnCommandBufferBeginRecording(&m_CommandBuffer);
    {
            /* Bind the texture and sampler descriptor pools. */
        m_pTextureIDManager->SetSamplerPool(&m_CommandBuffer);
        m_pTextureIDManager->SetTexturePool(&m_CommandBuffer);

            /* Sets the scissor rectangle and viewport to the full screen */
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
        nvnCommandBufferSetViewport(&m_CommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

            /* Clears the currently set render target at a given index. */
        float clear_color[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, clear_color, NVN_CLEAR_COLOR_MASK_RGBA);
        nvnCommandBufferClearDepthStencil(&m_CommandBuffer, 1.0, NVN_TRUE, 0, 0);

        std::vector<NVNModelData*>&     modelData   = m_pDataHolder->GetModelData();
        std::vector<NVNProgramData*>&   programData = m_pDataHolder->GetProgramData();

        if (m_pShaderScratchMemoryPool)
        {
            nvnCommandBufferSetShaderScratchMemory(&m_CommandBuffer, m_pShaderScratchMemoryPool->GetMemoryPool(), 0, m_ShaderScratchSize);
        }

            /* Bind the render state objects. */
        nvnCommandBufferBindBlendState(&m_CommandBuffer, &m_BlendState);
        nvnCommandBufferBindChannelMaskState(&m_CommandBuffer, &m_ChannelMaskState);
        nvnCommandBufferBindColorState(&m_CommandBuffer, &m_ColorState);
        nvnCommandBufferBindDepthStencilState(&m_CommandBuffer, &m_DepthStencilState);
        nvnCommandBufferBindMultisampleState(&m_CommandBuffer, &m_MultisampleState);
        nvnCommandBufferBindPolygonState(&m_CommandBuffer, &m_PolygonState);
        nvnCommandBufferSetSampleMask(&m_CommandBuffer, static_cast<uint32_t>(~0));

            /* Bind the vertex buffer(s). */
        Model* model = &modelData[0]->m_Model;
        NVNbufferAddress vboAddress = nvnBufferGetAddress(&modelData[0]->m_VertexBuffer);
        for(size_t i = 0; i < model->m_VertexAttributes.size(); ++i)
        {
            VertexAttribute& attribute = model->m_VertexAttributes[i];
            nvnCommandBufferBindVertexBuffer(&m_CommandBuffer, attribute.m_Location, vboAddress + modelData[0]->m_VertexAttributeBufferOffsets[i], attribute.m_DataSize);
        }

            /* Bind the vertex states. */
        nvnCommandBufferBindVertexAttribState(&m_CommandBuffer, static_cast<int>(modelData[0]->m_VertexAttributeStates.size()), &modelData[0]->m_VertexAttributeStates[0]);
        nvnCommandBufferBindVertexStreamState(&m_CommandBuffer, static_cast<int>(modelData[0]->m_VertexStreamStates.size()), &modelData[0]->m_VertexStreamStates[0]);

            /* Bind the shader program and uniforms. */
        int vertBindingLoc = SimpleTexturedModelAlpha::BlockVSUniformBlockData::GetBinding(NVN_SHADER_STAGE_VERTEX);
        nvnCommandBufferBindUniformBuffer(&m_CommandBuffer, NVN_SHADER_STAGE_VERTEX, vertBindingLoc, m_UniformBlockVertex.GetBufferAddress(), sizeof(SimpleTexturedModelAlpha::BlockVSUniformBlockData));

        int fragBindingLoc = SimpleTexturedModelAlpha::BlockFSUniformBlockData::GetBinding(NVN_SHADER_STAGE_FRAGMENT);
        nvnCommandBufferBindUniformBuffer(&m_CommandBuffer, NVN_SHADER_STAGE_FRAGMENT, fragBindingLoc, m_UniformBlockFragment.GetBufferAddress(), sizeof(SimpleTexturedModelAlpha::BlockFSUniformBlockData));

        programData[0]->BindShaderProgram(&m_CommandBuffer);

            /* Draw the primitives. */
        NVNindexType indexType = static_cast<NVNindexType>(modelData[0]->m_Model.m_IndexData.m_IndexType);
        uint32_t numIndices = modelData[0]->m_Model.m_IndexData.m_DataSize / modelData[0]->m_Model.m_IndexData.m_Stride;
        nvnCommandBufferDrawElements(&m_CommandBuffer,
                                     (NVNdrawPrimitive)modelData[0]->m_Model.m_NvnDrawPrimitiveType,
                                     indexType,
                                     numIndices,
                                     nvnBufferGetAddress(&modelData[0]->m_IndexBuffer));
    }

    m_CommandHandle = nvnCommandBufferEndRecording(&m_CommandBuffer);
}

/*
 * SettingUpRenderState::Draw
 * --------------------------
 * This method updates uniform data and submits the commands
 * recorded in the command buffer to the queue and presents
 * the render target to the screen.
 */
void SettingUpRenderState::Draw(uint64_t millisec)
{
    NN_UNUSED(millisec);
    static int s_Frame = 0;
    ++s_Frame;

        /* Get the current render target and setup/submit a command buffer to set it. */
    int index = UpdateRenderTargets();

    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

        /* Update the uniform data. */
    nn::util::Matrix4x3fType simpleTranslateMat;
    nn::util::MatrixIdentity(&simpleTranslateMat);
    nn::util::Vector3fType simpleTranslateVector;
    nn::util::VectorSet(&simpleTranslateVector, -0.0f, 0.0f, -3.0f);
    nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

    nn::util::Matrix4x3fType simpleRotateXMat;
    nn::util::MatrixIdentity(&simpleRotateXMat);
    nn::util::Vector3fType simpleRotateXVector;
    nn::util::VectorSet(&simpleRotateXVector, 30.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

    nn::util::Matrix4x3fType simpleRotateYMat;
    nn::util::MatrixIdentity(&simpleRotateYMat);
    nn::util::Vector3fType simpleRotateYVector;
    nn::util::VectorSet(&simpleRotateYVector, 0.0f, s_Frame * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

    nn::util::Matrix4x3fType simpleModelMat;
    nn::util::Matrix4x3fType tempMat1;
    nn::util::MatrixMultiply(&tempMat1, simpleRotateYMat, simpleRotateXMat);
    nn::util::MatrixMultiply(&simpleModelMat, tempMat1, simpleTranslateMat);
    nn::util::Matrix4x4fType simpleModelMat44;
    nn::util::MatrixConvert(&simpleModelMat44, simpleModelMat);

    nn::util::Matrix4x4fType projMat44;
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projMat44, 60.0f * 3.14159f / 180.0f, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);

    std::vector<NVNTextureData*>& textureData = m_pDataHolder->GetTextureData();
    UpdateUniformBlock(simpleModelMat44, projMat44, textureData[0]->m_TextureHandle);

        /* Set the render state for the current frame. */
    SetStates(s_Frame);

        /* If the state changed, re-record the command buffer. */
    if (m_ChangeState)
    {
        UpdateCommandBuffer();
    }

        /*!
         * Wait on sync that was received in UpdateRenderTargets now that we are
         * actually ready to use the render target
         */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);
    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);

        /* Print out the frame counter on the screen. */
    m_pDebugTextRenderer->Printf(0, 0, "Frames: %i", s_Frame);

        /* Draw the accumulated debug text for the frame. */
    m_pDebugTextRenderer->Draw(m_pRenderTargets[index], m_ScreenWidth, m_ScreenHeight);

        /* Present the texture. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);
}

/*
 * SettingUpRenderState::UpdateRenderTargets
 * -----------------------------------------
 * Gets the index of the current render target from the NVNwindow
 * and records a command buffer that sets it up to be rendered to.
 */
int SettingUpRenderState::UpdateRenderTargets()
{
        /* Get next render target to be used */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        /* Wait for the previous fram to finish before rewriting the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Reset the command buffer memory. */
    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

        /* Record the command buffer to set the target. */
    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_pRenderTargets[m_CurrentWindowIndex], NULL, m_pDepthTarget, NULL);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

        /* Return the index. */
    return m_CurrentWindowIndex;
}

/*
 * SettingUpRenderState::Resize
 * ----------------------------
 * This method is called everytime the window is resized and is passed
 * the new size of the window. It frees the old render target and creates a new
 * render target with the new screen size. A new set of commands for the command buffer
 * is recorded.
 */
void SettingUpRenderState::Resize(int width, int height)
{
        /* Check for the window being minimized or having no visible surface. */
    if (width == 0 || height == 0)
    {
        return;
    }

        /* Save the current height and width of the screen. */
    m_ScreenWidth = width;
    m_ScreenHeight = height;

        /* Sync to safely rewrite the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
        /*
         * Otherwise finalize (free) the NVNwindow used for the previous window size.
         * The NVNWindow must be finalized before a render target it owns is finalized.
         */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

        /* Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
            /* If it's the first time Resize is called, allocate the render target. */
        if (!m_pRenderTargets[i])
        {
            m_pRenderTargets[i] = new NVNtexture;
        }
            /* Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

            /* Create the texture using the current state of the texture builder. */
        if (!nvnTextureInitialize(m_pRenderTargets[i], &m_RenderTargetBuilder))
        {
            NN_ASSERT(0, "Failed to initialize render target");
        }
    }

    if (!m_pDepthTarget)
    {
        m_pDepthTarget = new NVNtexture;
    }
    else
    {
        nvnTextureFinalize(m_pDepthTarget);
    }

        /* Initialize depth buffer for render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);

    nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * g_NumColorBuffers);
    nvnTextureInitialize(m_pDepthTarget, &m_RenderTargetBuilder);

        /* Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_pRenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

        /* Update the cube rendering command buffer. */
    UpdateCommandBuffer();
}

/*
 * SettingUpRenderState Destructor
 * -------------------------------
 * Empty destructor.
 */
SettingUpRenderState::~SettingUpRenderState()
{
}

/*
 * SettingUpRenderState::Shutdown
 * ------------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void SettingUpRenderState::Shutdown()
{
    nvnQueueFinish(&m_Queue);

    nvnSyncFinalize(&m_CommandBufferSync);
    nvnSyncFinalize(&m_UniformBlockUpdateSync);
    nvnSyncFinalize(&m_RenderTargetPresentSync);
    nvnSyncFinalize(&m_WindowSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if(m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);

    if(m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if (m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if(m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if(m_pRenderTargets[i])
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
            delete m_pRenderTargets[i];
            m_pRenderTargets[i] = NULL;
        }
    }

    if(m_pDepthTarget)
    {
        nvnTextureFinalize(m_pDepthTarget);
        delete m_pDepthTarget;
        m_pDepthTarget = NULL;
    }

    m_UniformBlockVertex.Finalize();
    m_UniformBlockFragment.Finalize();

    if(m_pUniformBufferMemoryPool)
    {
        m_pUniformBufferMemoryPool->Shutdown();
        delete m_pUniformBufferMemoryPool;
        m_pUniformBufferMemoryPool = NULL;
    }

    if(m_pDataHolder)
    {
        delete m_pDataHolder;
        m_pDataHolder = NULL;
    }

    if(m_pAssetLoader)
    {
        delete m_pAssetLoader;
        m_pAssetLoader = NULL;
    }

    if(m_pTextureIDManager)
    {
        delete m_pTextureIDManager;
        m_pTextureIDManager = NULL;
    }

    if(m_pDebugTextRenderer)
    {
        m_pDebugTextRenderer->CleanUp();
        m_pDebugTextRenderer = NULL;
    }

    if(m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if (m_pShaderScratchMemoryPool)
    {
        m_pShaderScratchMemoryPool->Shutdown();
        delete m_pShaderScratchMemoryPool;
        m_pShaderScratchMemoryPool = NULL;
        m_ShaderScratchSize = 0;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*
 * SettingUpRenderState::SetupModelUniformBlock
 * --------------------------------------------
 * Sets up the uniform blocks for rendering.
 */
void SettingUpRenderState::SetupModelUniformBlock()
{
    m_pUniformBufferMemoryPool = new MemoryPool();

    int bufferAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    m_pUniformBufferMemoryPool->Init(NULL,
                                     Align(sizeof(SimpleTexturedModel::BlockVSUniformBlockData), bufferAlignment) +
                                     Align(sizeof(SimpleTexturedModel::BlockFSUniformBlockData), bufferAlignment),
                                     NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                     &m_Device);
        /*
        * Initialize the uniform buffers with a pointer to the
        * device and the appropriate size for the shader.
        */
    m_UniformBlockVertex.Init(&m_Device, sizeof(SimpleTexturedModel::BlockVSUniformBlockData), m_pUniformBufferMemoryPool);
    m_UniformBlockFragment.Init(&m_Device, sizeof(SimpleTexturedModel::BlockFSUniformBlockData), m_pUniformBufferMemoryPool);
}

/*
 * SettingUpRenderState::UpdateUniformBlock
 * ----------------------------------------
 * Updates the uniform block data with new data.
 */
void SettingUpRenderState::UpdateUniformBlock(const nn::util::Matrix4x4fType& modelMatrix, const nn::util::Matrix4x4fType& projectionMatrix, const NVNtextureHandle handle)
{
        /* Sync to safely overwrite memory. */
    nvnQueueFenceSync(&m_Queue, &m_UniformBlockUpdateSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_UniformBlockUpdateSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Creates an instance of the vertex shader's helper class. */
    SimpleTexturedModel::BlockVSUniformBlockData vertexShaderData;

        /*
         * Sets the data in the helper class. The helper class functions
         * automatically handle placing the data at the correct offset
         * in the uniform block.
         */

    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, modelMatrix);
    vertexShaderData.SetUniform_u_modelMtx(*reinterpret_cast<const float(*)[16]>(&temp));

    nn::util::Matrix4x4fType cameraMat;
    nn::util::MatrixIdentity(&cameraMat);
    nn::util::MatrixStore(&temp, cameraMat);
    vertexShaderData.SetUniform_u_viewMtx(*reinterpret_cast<const float(*)[16]>(&temp));

    nn::util::MatrixStore(&temp, projectionMatrix);
    vertexShaderData.SetUniform_u_projMtx(*reinterpret_cast<const float(*)[16]>(&temp));

        /* Copy the set up data into the uniform buffer. */
    m_UniformBlockVertex.SetData(&vertexShaderData, sizeof(vertexShaderData));

        /* Creates an instance of the fragment shader's helper class. */
    SimpleTexturedModel::BlockFSUniformBlockData fragmentShaderData;

        /* Sets the data in the helper class. */
    fragmentShaderData.SetUniform_u_bindlessTex(handle);

        /* Copy the set up data into the uniform buffer. */
    m_UniformBlockFragment.SetData(&fragmentShaderData, sizeof(fragmentShaderData));
}

void SettingUpRenderState::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* user
    )
{
    NN_ASSERT(user == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static SettingUpRenderState tut;
    return (&tut);
}

extern "C" void nnMain()
{
    TutorialRun();
}
