﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NvnTutorial04.cpp,PageSampleNvnTutorial04}
 *
 * @brief
 *  This tutorial builds off of Tutorial03 and introduces asset
 *  file loading in parallel on multiple threads and rendering
 *  multiple models with different asset data and shaders.
 */

/**
 * @page PageSampleNvnTutorial04 NVN Tutorial 04: Multithreaded Asset File Loading, Cube Maps, and Mipmaps
 * @tableofcontents
 *
 * @brief
 *  This tutorial builds off of Tutorial03 and introduces asset
 *  file loading in parallel on multiple threads and rendering
 *  multiple models with different asset data and shaders.
 *
 * @section PageSampleNvnTutorial04_SectionBrief Overview
 *  This tutorial builds off of Tutorial03 and introduces asset
 *  file loading in parallel on multiple threads and rendering
 *  multiple models with different asset data and shaders. Three
 *  cubes are rendered, one with a simple texture, one with a mipmapped
 *  texture, and one with a cubemap. The multi threaded loading
 *  is shown in the MultiThreadedLoadfunction of this tutorial class
 *  and in MultiThreadAssetFileLoadingHelper.cpp/.h.
 *
 * @subsection PageSampleNvnTutorial04_SectionExpectedOutput Expected Output
 * @image html NvnTutorial04.png
 *
 * @section PageSampleNvnTutorial04_SectionFileStructure File Structure
 *  The main tutorial file and Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/NvnTutorial04MultiThreadedAssetFileLoading Samples/Sources/Applications/NvnTutorial04MultiThreadedAssetFileLoading @endlink
 *
 *  The tutorial library which contains common code shared by the tutorials can be found at
 *  Samples/Sources/Libraries/NvnTutorial
 *
 * @section PageSampleNvnTutorial04_SectionNecessaryEnvironment System Requirements
 *  No extra system requirements.
 *
 * @section PageSampleNvnTutorial04_SectionHowToOperate Operation Procedure
 *  This sample runs on its own without any additional input. Hitting escape on Windows
 *  or touching the touch screen on NX will exit the program.
 *
 * @section PageSampleNvnTutorial04_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleNvnTutorial04_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution in the desired configuration and run it.
 *
 * @section PageSampleNvnTutorial04_SectionDetail Description
 *
 * @subsection PageSampleNvnTutorial04_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  NvnTutorial04.cpp
 *  @includelineno NvnTutorial04.cpp
 *
 * @subsection PageSampleNvnTutorial04_SectionSampleDetail Sample Program Description
 *  This tutorial renders three cubes: one that is the same as
 *  Tutorial03, and second cube that demostrates mipmapping, and
 *  a third that is textured with a cube map. The mipmapped cube
 *  has different color for each mip level so that it's easy to
 *  see at what points the levels switch. The mipmapped texture
 *  was built using a texture file per mip level by Tutorial02.
 *  The mipmapped cube and the basic cube are rendered with the
 *  same shader. The third cube is textured with a cube map built
 *  out of six textures and rendered using a second shader program.
 */

#include <cstdio>
#include <cmath>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nvn/nvn_FuncPtrImpl.h>

#include <nvntutorial/TutorialBaseClass.h>
#include <nvntutorial/MultiThreadedAssetFileLoadingHelper.h>
#include <nvntutorial/TextureIDManager.h>
#include <nvntutorial/TutorialUtil.h>
#include <nvntutorial/UniformBuffer.h>
#include <nvntutorial/MemoryPool.h>

#ifdef _WIN32
    #ifndef WIN32_LEAN_AND_MEAN
    #define WIN32_LEAN_AND_MEAN
    #endif
    #ifndef NOMINMAX
    #define NOMINMAX
    #endif
    #include <nn/nn_Windows.h>
#endif

static const int    g_CubesToRender = 3;
static const float  g_Pi = 3.14159f;
static const size_t g_CommandMemorySize = 2 * 1024;
static const size_t g_ControlMemorySize = 2 * 1024;
static const size_t g_NumColorBuffers = 2;

class MultiThreadedAssetFileLoading : public TutorialBaseClass
{
    NN_DISALLOW_COPY(MultiThreadedAssetFileLoading);

public:
    MultiThreadedAssetFileLoading();
    virtual ~MultiThreadedAssetFileLoading();
    virtual void Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow);
    virtual void Shutdown();

    virtual void Draw(uint64_t millisec);
    virtual void Resize(int width, int height);

private:
    static void NVNAPIENTRY DebugLayerCallback(
        NVNdebugCallbackSource source,
        NVNdebugCallbackType type,
        int id,
        NVNdebugCallbackSeverity severity,
        const char* message,
        void* pUser
        );

    void SetupUniformBlock();
    void UpdateSimpleTexturedModelUniformBlock(UniformBuffer* pVertUniformBuffer, UniformBuffer* pFragUniformBuffer, nn::util::Matrix4x4fType& modelMat, nn::util::Matrix4x4fType& projMat, NVNtextureHandle handle);
    void UpdateCubeMappedModelUniformBlock(UniformBuffer* pVertUniformBuffer, UniformBuffer* pFragUniformBuffer, nn::util::Matrix4x4fType& modelMat, nn::util::Matrix4x4fType& projMat, NVNtextureHandle handle);
    void MultiThreadedLoad();

    int UpdateRenderTargets();

    NVNdevice                         m_Device;
    NVNqueue                          m_Queue;
    void*                             m_pQueueMemory;

    MemoryPool*                       m_pCommandMemoryPool;

    ptrdiff_t                         m_CommandPoolOffset;
    void*                             m_pControlPool;
    NVNcommandBuffer                  m_CommandBuffer;
    NVNcommandHandle                  m_CommandHandle;

    NVNtextureBuilder                 m_RenderTargetBuilder;
    NVNtexture*                       m_pRenderTargets[g_NumColorBuffers];
    NVNtexture*                       m_pDepthBuffer;
    NVNsamplerBuilder                 m_SamplerBuilder;
    NVNbufferBuilder                  m_BufferBuilder;

    NVNsync                           m_RenderTargetPresentSync;
    NVNsync                           m_CommandBufferSync;
    NVNsync                           m_UniformBlockUpdateSync;
    std::vector<AssetFileDataHolder*> m_DataHolders;

    NVNblendState                     m_BlendState;
    NVNchannelMaskState               m_ChannelMaskState;
    NVNcolorState                     m_ColorState;
    NVNdepthStencilState              m_DepthStencilState;
    NVNmultisampleState               m_MultiSampleState;
    NVNpolygonState                   m_PolygonState;

    MemoryPool*                       m_pUniformBufferMemoryPool;

    UniformBuffer                     m_BasicCubeUniformVertexBuffer;
    UniformBuffer                     m_BasicCubeUniformFragmentBuffer;

    UniformBuffer                     m_MipmapCubeUniformVertexBuffer;
    UniformBuffer                     m_MipmapCubeUniformFragmentBuffer;

    UniformBuffer                     m_CubeMapUniformVertexBuffer;
    UniformBuffer                     m_CubeMapUniformFragmentBuffer;

    TextureIDManager*                 m_pTextureIDManager;

    MemoryPool*                       m_pRenderTargetMemoryPool;

    MemoryPool*                       m_pShaderScratchMemoryPool;
    size_t                            m_ShaderScratchSize;

    ptrdiff_t                         m_RenderTargetCommandPoolOffset;
    void*                             m_pRenderTargetControlPool;
    NVNcommandBuffer                  m_RenderTargetCommandBuffer;
    NVNcommandHandle                  m_RenderTargetCommandHandle;

    NVNwindow*                        m_pWindow;
    NVNwindowBuilder                  m_WindowBuilder;
    int                               m_CurrentWindowIndex;
    NVNsync                           m_WindowSync;

    int                               m_ScreenWidth;
    int                               m_ScreenHeight;

    float                             m_RotY;
    float                             m_Oscillate;

    int                               m_CommandMemoryUsed;
    int                               m_ControlMemoryUsed;

    size_t                            m_ColorTargetSize;
    size_t                            m_DepthTargetSize;
};

/*
 * MultiThreadedAssetFileLoading Constructor
 * -----------------------------------------
 * Setup default member values.
 */
MultiThreadedAssetFileLoading::MultiThreadedAssetFileLoading() :
    m_pQueueMemory(NULL),
    m_pCommandMemoryPool(NULL),
    m_pControlPool(NULL),
    m_CommandHandle(NULL),
    m_pDepthBuffer(NULL),
    m_pUniformBufferMemoryPool(NULL),
    m_pTextureIDManager(NULL),
    m_pRenderTargetMemoryPool(NULL),
    m_pShaderScratchMemoryPool(NULL),
    m_ShaderScratchSize(0),
    m_pRenderTargetControlPool(NULL),
    m_RenderTargetCommandHandle(NULL),
    m_pWindow(NULL),
    m_CurrentWindowIndex(-1),
    m_ScreenWidth(0),
    m_ScreenHeight(0),
    m_RotY(45.0f),
    m_Oscillate(g_Pi),
    m_CommandMemoryUsed(0),
    m_ControlMemoryUsed(0),
    m_ColorTargetSize(0),
    m_DepthTargetSize(0)
{
    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        m_pRenderTargets[i] = NULL;
    }
}

/*
 * MultiThreadedAssetFileLoading::SetupUniformBlock
 * ------------------------------------------------
 * Initialize the uniform blocks for each model to render.
 */
void MultiThreadedAssetFileLoading::SetupUniformBlock()
{
        /* Setup the memory pool for the uniform blocks. */
    m_pUniformBufferMemoryPool = new MemoryPool();

    int bufferAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_UNIFORM_BUFFER_ALIGNMENT, &bufferAlignment);
    m_pUniformBufferMemoryPool->Init(NULL,
                                     2 * Align(sizeof(SimpleTexturedModel::BlockVSUniformBlockData), bufferAlignment) +
                                     2 * Align(sizeof(SimpleTexturedModel::BlockFSUniformBlockData), bufferAlignment) +
                                     Align(sizeof(CubeMap::BlockVSUniformBlockData), bufferAlignment) +
                                     Align(sizeof(CubeMap::BlockFSUniformBlockData), bufferAlignment),
                                     NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                                     &m_Device);

    m_BasicCubeUniformVertexBuffer.Init(&m_Device, sizeof(SimpleTexturedModel::BlockVSUniformBlockData), m_pUniformBufferMemoryPool);
    m_BasicCubeUniformFragmentBuffer.Init(&m_Device, sizeof(SimpleTexturedModel::BlockFSUniformBlockData), m_pUniformBufferMemoryPool);

    m_MipmapCubeUniformVertexBuffer.Init(&m_Device, sizeof(SimpleTexturedModel::BlockVSUniformBlockData), m_pUniformBufferMemoryPool);
    m_MipmapCubeUniformFragmentBuffer.Init(&m_Device, sizeof(SimpleTexturedModel::BlockFSUniformBlockData), m_pUniformBufferMemoryPool);

    m_CubeMapUniformVertexBuffer.Init(&m_Device, sizeof(CubeMap::BlockVSUniformBlockData), m_pUniformBufferMemoryPool);
    m_CubeMapUniformFragmentBuffer.Init(&m_Device, sizeof(CubeMap::BlockFSUniformBlockData), m_pUniformBufferMemoryPool);
}

/*
 * MultiThreadedAssetFileLoading::UpdateSimpleTexturedModelUniformBlock
 * --------------------------------------------------------------------
 * Update the uniforms for the simple/mipmapped cubes.
 */
void MultiThreadedAssetFileLoading::UpdateSimpleTexturedModelUniformBlock(UniformBuffer* pVertUniformBuffer, UniformBuffer* pFragUniformBuffer, nn::util::Matrix4x4fType& modelMat, nn::util::Matrix4x4fType& projMat, NVNtextureHandle handle)
{
    SimpleTexturedModel::BlockVSUniformBlockData vsData;

    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, modelMat);
    vsData.SetUniform_u_modelMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::Matrix4x4fType cameraMat;
    nn::util::MatrixIdentity(&cameraMat);
    nn::util::MatrixStore(&temp, cameraMat);
    vsData.SetUniform_u_viewMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::MatrixStore(&temp, projMat);
    vsData.SetUniform_u_projMtx(*reinterpret_cast<float(*)[16]>(&temp));

    pVertUniformBuffer->SetData(&vsData, sizeof(vsData));

    SimpleTexturedModel::BlockFSUniformBlockData fsData;
    fsData.SetUniform_u_bindlessTex(handle);
    pFragUniformBuffer->SetData(&fsData, sizeof(fsData));
}

/*
 * MultiThreadedAssetFileLoading::UpdateCubeMappedModelUniformBlock
 * ----------------------------------------------------------------
 * Update the uniforms for the cube mapped cube.
 */
void MultiThreadedAssetFileLoading::UpdateCubeMappedModelUniformBlock(UniformBuffer* pVertUniformBuffer, UniformBuffer* pFragUniformBuffer, nn::util::Matrix4x4fType& modelMat, nn::util::Matrix4x4fType& projMat, NVNtextureHandle handle)
{
    CubeMap::BlockVSUniformBlockData vsData;

    nn::util::Float4x4 temp;
    nn::util::MatrixStore(&temp, modelMat);
    vsData.SetUniform_u_modelMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::Matrix4x4fType cameraMat;
    nn::util::MatrixIdentity(&cameraMat);
    nn::util::MatrixStore(&temp, cameraMat);
    vsData.SetUniform_u_viewMtx(*reinterpret_cast<float(*)[16]>(&temp));

    nn::util::MatrixStore(&temp, projMat);
    vsData.SetUniform_u_projMtx(*reinterpret_cast<float(*)[16]>(&temp));

    pVertUniformBuffer->SetData(&vsData, sizeof(vsData));

    CubeMap::BlockFSUniformBlockData fsData;
    fsData.SetUniform_u_bindlessTex(handle);
    pFragUniformBuffer->SetData(&fsData, sizeof(fsData));
}

/*
 * MultiThreadedAssetFileLoading Destructor
 * ----------------------------------------
 * Empty destructor.
 */
MultiThreadedAssetFileLoading::~MultiThreadedAssetFileLoading()
{
}

/*
 * MultiThreadedAssetFileLoading::Init
 * -----------------------------------
 * Initialize NVN, load asset files, and create objects needed for the
 * application to run.
 */
void MultiThreadedAssetFileLoading::Init(PFNNVNBOOTSTRAPLOADERPROC pLoader, NVNnativeWindow nativeWindow)
{
    NN_ASSERT(pLoader != NULL);

        /* Load the function pointer to get other nvn function pointers. */
    pfnc_nvnDeviceGetProcAddress = (PFNNVNDEVICEGETPROCADDRESSPROC) (*pLoader)("nvnDeviceGetProcAddress");

    if (pfnc_nvnDeviceGetProcAddress == NULL)
    {
            /* This can happen if an NVN driver is not installed on a Windows PC. */
        NN_ASSERT(0, "BootstrapLoader failed to find nvnDeviceGetProcAddress");
    }

        /*
         * Load function pointers for creating nvn device
         */
    nvnLoadCProcs( NULL, pfnc_nvnDeviceGetProcAddress );

        /* Check the API version. */
    int majorVersion, minorVersion;
    nvnDeviceGetInteger( NULL, NVN_DEVICE_INFO_API_MAJOR_VERSION, &majorVersion );
    nvnDeviceGetInteger( NULL, NVN_DEVICE_INFO_API_MINOR_VERSION, &minorVersion );
    if( majorVersion != NVN_API_MAJOR_VERSION || minorVersion < NVN_API_MINOR_VERSION )
    {
        NN_ASSERT( 0, "NVN SDK not supported by current driver." );
    }

        /* If debug or develop is enabled, turn on NVN's debug layer. */
    int deviceFlags = 0;
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    deviceFlags = NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT;
#endif

    NVNdeviceBuilder deviceBuilder;
    nvnDeviceBuilderSetDefaults(&deviceBuilder);
    nvnDeviceBuilderSetFlags(&deviceBuilder, deviceFlags);

    if (nvnDeviceInitialize(&m_Device, &deviceBuilder) == false)
    {
            /*
             * This can fail for a few reasons; the most likely on Horizon is
             * insufficent device memory.
             */
        NN_ASSERT(0, "nvnDeviceInitialize");
    }

        /*
         * Ensure function pointers are all loaded properly
         */
    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress );

        /* Setup the debug callback for the debug layer. */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

        /* Initialize the queue. */
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDevice(&queueBuilder, &m_Device);
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetComputeMemorySize(&queueBuilder, 0);

    int minQueueCommandMemorySize = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_QUEUE_COMMAND_MEMORY_MIN_SIZE, &minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandMemorySize(&queueBuilder, minQueueCommandMemorySize);
    nvnQueueBuilderSetCommandFlushThreshold(&queueBuilder, minQueueCommandMemorySize);

    size_t neededQueueMemorySize = nvnQueueBuilderGetQueueMemorySize(&queueBuilder);

    if ((neededQueueMemorySize % NVN_MEMORY_POOL_STORAGE_GRANULARITY) != 0)
    {
        NN_ASSERT(0, "Memory size reported for queue is not the proper granularity");
    }

#if defined( NN_BUILD_TARGET_PLATFORM_OS_WIN )
    m_pQueueMemory = NULL;
#else
    m_pQueueMemory = AlignedAllocate(neededQueueMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
#endif

    nvnQueueBuilderSetQueueMemory(&queueBuilder, m_pQueueMemory, neededQueueMemorySize);

    if (nvnQueueInitialize(&m_Queue, &queueBuilder) == false)
    {
        NN_ASSERT(0, "nvnQueueInitialize failed");
    }

        /* Initialize the rendering command buffer. */
    if (!nvnCommandBufferInitialize(&m_CommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize command buffer");
    }

        /*
         * Queries the device for the proper control and command
         * memory alignment for a command buffer.
         */
    int commandBufferCommandAlignment = 0;
    int commandBufferControlAlignment = 0;
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT, &commandBufferCommandAlignment);
    nvnDeviceGetInteger(&m_Device, NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT, &commandBufferControlAlignment);

        /* Setup the command buffer memory pool. */
    m_pCommandMemoryPool = new MemoryPool();
    m_pCommandMemoryPool->Init(NULL,
                               Align(g_CommandMemorySize, commandBufferCommandAlignment) * 2,
                               NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT,
                               &m_Device);

        /* Grab the offset from the memory pool. */
    m_CommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

        /* Setup the control memory with the proper alignment. */
    m_pControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

        /* Add the memory to the command buffer. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        /* Render Target Setting Command Buffer. */
    if (!nvnCommandBufferInitialize(&m_RenderTargetCommandBuffer, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize render target command buffer");
    }

        /* Grab the offset from the memory pool. */
    m_RenderTargetCommandPoolOffset = m_pCommandMemoryPool->GetNewMemoryChunkOffset(g_CommandMemorySize, commandBufferCommandAlignment);

        /* Setup the control memory with the proper alignment. */
    m_pRenderTargetControlPool = AlignedAllocate(g_ControlMemorySize, commandBufferControlAlignment);

    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

        /* Initialize the render target sync. */
    if (!nvnSyncInitialize(&m_RenderTargetPresentSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize render target present sync");
    }

        /* Initialize the rendering command buffer sync. */
    if (!nvnSyncInitialize(&m_CommandBufferSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize command buffer sync");
    }

        /* Initialize the uniform buffer update sync. */
    if (!nvnSyncInitialize(&m_UniformBlockUpdateSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize uniform block update sync");
    }

        /*! Initialize the window sync. */
    if (!nvnSyncInitialize(&m_WindowSync, &m_Device))
    {
        NN_ASSERT(0, "Failed to initialize window sync");
    }

        /* Blend State */
    nvnBlendStateSetDefaults(&m_BlendState);

        /* Channel Mask State */
    nvnChannelMaskStateSetDefaults(&m_ChannelMaskState);

        /* Color State */
    nvnColorStateSetDefaults(&m_ColorState);

        /* Depth Stencil State */
    nvnDepthStencilStateSetDefaults(&m_DepthStencilState);

    nvnDepthStencilStateSetDepthTestEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthWriteEnable(&m_DepthStencilState, NVN_TRUE);
    nvnDepthStencilStateSetDepthFunc(&m_DepthStencilState, NVNdepthFunc::NVN_DEPTH_FUNC_LESS);

        /* Multisample state */
    nvnMultisampleStateSetDefaults(&m_MultiSampleState);

        /* Polygon state */
    nvnPolygonStateSetDefaults(&m_PolygonState);
    nvnPolygonStateSetFrontFace(&m_PolygonState, NVNfrontFace::NVN_FRONT_FACE_CCW);
    nvnPolygonStateSetCullFace(&m_PolygonState, NVNface::NVN_FACE_BACK);
    nvnPolygonStateSetPolygonMode(&m_PolygonState, NVNpolygonMode::NVN_POLYGON_MODE_FILL);

        /* Builders */
    nvnSamplerBuilderSetDevice(&m_SamplerBuilder, &m_Device);
    nvnSamplerBuilderSetDefaults(&m_SamplerBuilder);

    nvnBufferBuilderSetDevice(&m_BufferBuilder, &m_Device);
    nvnBufferBuilderSetDefaults(&m_BufferBuilder);

        /* Set up the texture builder for the render target. */
    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    m_DepthTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

        /* Set up the render target memory pool. */
    m_pRenderTargetMemoryPool = new MemoryPool();
    m_pRenderTargetMemoryPool->Init(
        NULL,
        m_ColorTargetSize * g_NumColorBuffers + m_DepthTargetSize,
        NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
        &m_Device);

    m_pTextureIDManager = new TextureIDManager(&m_Device);

    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);

        /* Setup the uniform blocks for the models. */
    SetupUniformBlock();

        /* Load the asset files in parallel. */
    MultiThreadedLoad();
}//NOLINT(impl/function_size)

/*
 * LoadAssetFiles
 * --------------
 * Function called by the asset loading worker threads.
 */
void LoadAssetFiles(void* pParam )
{
    MultiThreadedAssetFileLoadingHelper* loader = (MultiThreadedAssetFileLoadingHelper*)pParam;
    loader->LoadAssetFile();
}

/*
 * MultiThreadedAssetFileLoading::MultiThreadedLoad
 * ------------------------------------------------
 * This function sets up multiple worker threads to
 * load asset files in parallel. Each thread handles
 * loading in the data from the binary file and setiing
 * up the NVN objects for each section of the file.
 */
void MultiThreadedAssetFileLoading::MultiThreadedLoad()
{
    AssetLoaderArg* pArgs[3];
    nn::os::MutexType poolMutex;
    nn::os::InitializeMutex(&poolMutex, false, 0);

    pArgs[0] = new AssetLoaderArg(&m_Device, "multiThreadedAssetFile0.out", &poolMutex, m_pTextureIDManager, false, false);
    pArgs[1] = new AssetLoaderArg(&m_Device, "multiThreadedAssetFile1.out", &poolMutex, m_pTextureIDManager, true, false);
    pArgs[2] = new AssetLoaderArg(&m_Device, "multiThreadedAssetFile2.out", &poolMutex, m_pTextureIDManager, false, true);

    MultiThreadedAssetFileLoadingHelper* pLoaders[3];
    pLoaders[0] = new MultiThreadedAssetFileLoadingHelper(pArgs[0]);
    pLoaders[1] = new MultiThreadedAssetFileLoadingHelper(pArgs[1]);
    pLoaders[2] = new MultiThreadedAssetFileLoadingHelper(pArgs[2]);

    nn::os::ThreadType hThreadArray[3];

    size_t cacheSize = 0;
    nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

    result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

#ifdef _WIN32
    size_t threadStackSize = 8 * 1024;
    char* threadStack = (char*)AlignedAllocate(g_CubesToRender * threadStackSize, 4096);
#else
    size_t threadStackSize = 1024 * 1024;
    char* threadStack = (char*)AlignedAllocate(g_CubesToRender * threadStackSize, 4096);
    NN_ASSERT(threadStack);
#endif

    {
        for(int i = 0; i < g_CubesToRender; ++i)
        {
            result = nn::os::CreateThread(
                &hThreadArray[i],                   // Address of ThreadType
                LoadAssetFiles,                     // Function pointer to call
                pLoaders[i],                        // Data to pass
                threadStack + i * threadStackSize,  // Stack memory for thread to use
                threadStackSize,                    // Size of stack
                nn::os::DefaultThreadPriority);     // Priority

            NN_ASSERT(result.IsSuccess());
            nn::os::StartThread(&hThreadArray[i]);
        }
    }

    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);

        /* Wait for the threads to be done. */
    for(int i = 0; i < g_CubesToRender; ++i)
    {
        nn::os::WaitThread(&hThreadArray[i]);
    }

        /* Grab the data holders for each file and do some setup. */
    for(int i = 0; i < g_CubesToRender; ++i)
    {
        AssetFileDataHolder* pDataHolder = pLoaders[i]->GetAssetFileDataHolder();
        m_DataHolders.push_back(pDataHolder);

        std::vector<NVNProgramData*>& programData = m_DataHolders[i]->GetProgramData();
        ShaderTypes::ShaderType shaderType = programData[0]->m_ShaderType;

        m_DataHolders[i]->SetupAttributeStatesNVN(ShaderTypes::Attributes_GetAttributeLocationFunction(shaderType));

        m_ShaderScratchSize = std::max(m_DataHolders[i]->GetProgramData()[0]->m_ShaderScratchMemorySize, m_ShaderScratchSize);
    }

    if (m_ShaderScratchSize != 0)
    {
        m_pShaderScratchMemoryPool = new MemoryPool();
        m_pShaderScratchMemoryPool->Init(nullptr,
                                         m_ShaderScratchSize,
                                         NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_COMPRESSIBLE_BIT,
                                         &m_Device);
    }

    nn::os::FinalizeMutex(&poolMutex);
    for(int i = 0; i < g_CubesToRender; ++i)
    {
        delete pArgs[i];
        pArgs[i] = NULL;
        delete pLoaders[i];
        pLoaders[i] = NULL;

        nn::os::DestroyThread(&hThreadArray[i]);
    }

    AlignedDeallocate(threadStack);
    nn::fs::Unmount("rom");
    delete[] mountRomCacheBuffer;
}

/*
 * MultiThreadedAssetFileLoading::Shutdown
 * ---------------------------------------
 * This method cleans up all nvn objects and dynamically allocated memory.
 */
void MultiThreadedAssetFileLoading::Shutdown()
{
    nvnQueueFinish(&m_Queue);
    nvnSyncFinalize(&m_RenderTargetPresentSync);
    nvnSyncFinalize(&m_CommandBufferSync);
    nvnSyncFinalize(&m_UniformBlockUpdateSync);
    nvnSyncFinalize(&m_WindowSync);

    nvnCommandBufferFinalize(&m_RenderTargetCommandBuffer);

    if(m_pRenderTargetControlPool)
    {
        AlignedDeallocate(m_pRenderTargetControlPool);
        m_pRenderTargetControlPool = NULL;
    }

    nvnCommandBufferFinalize(&m_CommandBuffer);

    if(m_pCommandMemoryPool)
    {
        m_pCommandMemoryPool->Shutdown();
        delete m_pCommandMemoryPool;
        m_pCommandMemoryPool = NULL;
    }

    if(m_pControlPool)
    {
        AlignedDeallocate(m_pControlPool);
        m_pControlPool = NULL;
    }

    if(m_pWindow)
    {
        nvnWindowFinalize(m_pWindow);
        delete m_pWindow;
        m_pWindow = NULL;
    }

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
        if(m_pRenderTargets[i])
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
            delete m_pRenderTargets[i];
            m_pRenderTargets[i] = NULL;
        }
    }

    if(m_pDepthBuffer)
    {
        nvnTextureFinalize(m_pDepthBuffer);
        delete m_pDepthBuffer;
        m_pDepthBuffer = NULL;
    }

    for (size_t i = 0; i < m_DataHolders.size(); ++i)
    {
        if(m_DataHolders[i])
        {
            delete m_DataHolders[i];
            m_DataHolders[i] = NULL;
        }
    }

    m_BasicCubeUniformVertexBuffer.Finalize();
    m_BasicCubeUniformFragmentBuffer.Finalize();

    m_MipmapCubeUniformVertexBuffer.Finalize();
    m_MipmapCubeUniformFragmentBuffer.Finalize();

    m_CubeMapUniformVertexBuffer.Finalize();
    m_CubeMapUniformFragmentBuffer.Finalize();

    if(m_pUniformBufferMemoryPool)
    {
        m_pUniformBufferMemoryPool->Shutdown();
        delete m_pUniformBufferMemoryPool;
        m_pUniformBufferMemoryPool = NULL;
    }

    if(m_pRenderTargetMemoryPool)
    {
        m_pRenderTargetMemoryPool->Shutdown();
        delete m_pRenderTargetMemoryPool;
        m_pRenderTargetMemoryPool = NULL;
    }

    if (m_pShaderScratchMemoryPool)
    {
        m_pShaderScratchMemoryPool->Shutdown();
        delete m_pShaderScratchMemoryPool;
        m_pShaderScratchMemoryPool = NULL;
        m_ShaderScratchSize = 0;
    }

    if(m_pTextureIDManager)
    {
        delete m_pTextureIDManager;
        m_pTextureIDManager = NULL;
    }

    nvnQueueFinalize(&m_Queue);
    if (m_pQueueMemory)
    {
        AlignedDeallocate(m_pQueueMemory);
        m_pQueueMemory = NULL;
    }

    nvnDeviceFinalize(&m_Device);
}

/*
 * MultiThreadedAssetFileLoading::Draw
 * -----------------------------------
 * This method updates uniform data and submits the commands
 * recorded in the command buffer to the queue and presents
 * the render target to the screen.
 */
void MultiThreadedAssetFileLoading::Draw(uint64_t /*millisec*/)
{
        /* Get the current render target and setup/submit a command buffer to set it. */
    int index = UpdateRenderTargets();

    nvnQueueSubmitCommands(&m_Queue, 1, &m_RenderTargetCommandHandle);

        /* Wait to update uniform data safely. */
    nvnQueueFenceSync(&m_Queue, &m_UniformBlockUpdateSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_UniformBlockUpdateSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Update the uniform values for the basic cube. */
    nn::util::Matrix4x3fType simpleTranslateMat;
    nn::util::MatrixIdentity(&simpleTranslateMat);
    nn::util::Vector3fType simpleTranslateVector;
    nn::util::VectorSet(&simpleTranslateVector, -1.5f, 0.0f, -3.0f);
    nn::util::MatrixSetTranslate(&simpleTranslateMat, simpleTranslateVector);

    nn::util::Matrix4x3fType simpleRotateXMat;
    nn::util::MatrixIdentity(&simpleRotateXMat);
    nn::util::Vector3fType simpleRotateXVector;
    nn::util::VectorSet(&simpleRotateXVector, 30.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateXMat, simpleRotateXVector);

    nn::util::Matrix4x3fType simpleRotateYMat;
    nn::util::MatrixIdentity(&simpleRotateYMat);
    nn::util::Vector3fType simpleRotateYVector;
    nn::util::VectorSet(&simpleRotateYVector, 0.0f, m_RotY * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&simpleRotateYMat, simpleRotateYVector);

    nn::util::Matrix4x3fType simpleScaleMat;
    nn::util::MatrixIdentity(&simpleScaleMat);
    nn::util::Vector3fType simpleScaleVector;
    nn::util::VectorSet(&simpleScaleVector, 0.75f, 0.75f, 0.75f);
    nn::util::MatrixSetScale(&simpleScaleMat, simpleScaleVector);

    nn::util::Matrix4x3fType simpleModelMat;
    nn::util::Matrix4x3fType tempMat1;
    nn::util::Matrix4x3fType tempMat2;
    nn::util::MatrixMultiply(&tempMat1, simpleScaleMat, simpleRotateYMat);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, simpleRotateXMat);
    nn::util::MatrixMultiply(&simpleModelMat, tempMat2, simpleTranslateMat);

    nn::util::Matrix4x4fType projMat44;
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projMat44, 60.0f * 3.14159f / 180.0f, static_cast<float>(m_ScreenWidth) / static_cast<float>(m_ScreenHeight), 1.0f, 1000.0f);

    std::vector<NVNTextureData*>& textureDataSimple = m_DataHolders[0]->GetTextureData();

    nn::util::Matrix4x4fType simpleModelMat44;
    nn::util::MatrixConvert(&simpleModelMat44, simpleModelMat);

    UpdateSimpleTexturedModelUniformBlock(&m_BasicCubeUniformVertexBuffer, &m_BasicCubeUniformFragmentBuffer, simpleModelMat44, projMat44, textureDataSimple[0]->m_TextureHandle);

        /* Update the uniform values for the mipmapped cube. */
    nn::util::Matrix4x3fType mipmapTranslateMat;
    nn::util::MatrixIdentity(&mipmapTranslateMat);
    nn::util::Vector3fType mipmapTranslateVector;
    nn::util::VectorSet(&mipmapTranslateVector, 0.0f, 0.0f, -52.0f - 50.0f * cos(m_Oscillate));
    nn::util::MatrixSetTranslate(&mipmapTranslateMat, mipmapTranslateVector);

    nn::util::Matrix4x3fType mipmapRotateXMat;
    nn::util::MatrixIdentity(&mipmapRotateXMat);
    nn::util::Vector3fType mipmapRotateXVector;
    nn::util::VectorSet(&mipmapRotateXVector, 35.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&mipmapRotateXMat, mipmapRotateXVector);

    nn::util::Matrix4x3fType mipmapRotateYMat;
    nn::util::MatrixIdentity(&mipmapRotateYMat);
    nn::util::Vector3fType mipmapRotateYVector;
    nn::util::VectorSet(&mipmapRotateYVector, 0.0f, 45.0f * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&mipmapRotateYMat, mipmapRotateYVector);

    nn::util::Matrix4x3fType mipmapScaleMat;
    nn::util::MatrixIdentity(&mipmapScaleMat);
    nn::util::Vector3fType mipmapScaleVector;
    nn::util::VectorSet(&mipmapScaleVector, 1.0f, 1.0f, 1.0f);
    nn::util::MatrixSetScale(&mipmapScaleMat, simpleScaleVector);

    nn::util::Matrix4x3fType mipmapModelMat;
    nn::util::MatrixMultiply(&tempMat1, mipmapScaleMat, mipmapRotateYMat);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, mipmapRotateXMat);
    nn::util::MatrixMultiply(&mipmapModelMat, tempMat2, mipmapTranslateMat);

    std::vector<NVNTextureData*>& textureDataMipmap = m_DataHolders[1]->GetTextureData();

    nn::util::Matrix4x4fType mipmapModelMat44;
    nn::util::MatrixConvert(&mipmapModelMat44, mipmapModelMat);

    UpdateSimpleTexturedModelUniformBlock(&m_MipmapCubeUniformVertexBuffer, &m_MipmapCubeUniformFragmentBuffer, mipmapModelMat44, projMat44, textureDataMipmap[0]->m_TextureHandle);

        /* Update the uniforms for the cube mapped cube. */
    nn::util::Matrix4x3fType cubeMapTranslateMat;
    nn::util::MatrixIdentity(&cubeMapTranslateMat);
    nn::util::Vector3fType cubeMapTranslateVector;
    nn::util::VectorSet(&cubeMapTranslateVector, 1.5f, 0.0f, -3.0f);
    nn::util::MatrixSetTranslate(&cubeMapTranslateMat, cubeMapTranslateVector);

    nn::util::Matrix4x3fType cubeMapRotateXMat;
    nn::util::MatrixIdentity(&cubeMapRotateXMat);
    nn::util::Vector3fType cubeMapRotateXVector;
    nn::util::VectorSet(&cubeMapRotateXVector, sin(m_Oscillate) * 30.0f * 3.1459f / 180.0f, 0.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&cubeMapRotateXMat, cubeMapRotateXVector);

    nn::util::Matrix4x3fType cubeMapRotateYMat;
    nn::util::MatrixIdentity(&cubeMapRotateYMat);
    nn::util::Vector3fType cubeMapRotateYVector;
    nn::util::VectorSet(&cubeMapRotateYVector, 0.0f, m_RotY * 3.14159f / 180.0f, 0.0f);
    nn::util::MatrixSetRotateXyz(&cubeMapRotateYMat, cubeMapRotateYVector);

    nn::util::Matrix4x3fType cubeMapScaleMat;
    nn::util::MatrixIdentity(&cubeMapScaleMat);
    nn::util::Vector3fType cubeMapScaleVector;
    nn::util::VectorSet(&cubeMapScaleVector, 0.75f, 0.75f, 0.75f);
    nn::util::MatrixSetScale(&cubeMapScaleMat, cubeMapScaleVector);

    nn::util::Matrix4x3fType cubeMapModelMat;
    nn::util::MatrixMultiply(&tempMat1, cubeMapScaleMat, cubeMapRotateYMat);
    nn::util::MatrixMultiply(&tempMat2, tempMat1, cubeMapRotateXMat);
    nn::util::MatrixMultiply(&cubeMapModelMat, tempMat2, cubeMapTranslateMat);

    std::vector<NVNTextureData*>& textureDatacubeMap = m_DataHolders[2]->GetTextureData();

    nn::util::Matrix4x4fType cubeMapModelMat44;
    nn::util::MatrixConvert(&cubeMapModelMat44, cubeMapModelMat);

    UpdateCubeMappedModelUniformBlock(&m_CubeMapUniformVertexBuffer, &m_CubeMapUniformFragmentBuffer, cubeMapModelMat44, projMat44, textureDatacubeMap[0]->m_TextureHandle);

    m_RotY += 0.15f;
    m_Oscillate += (g_Pi / 360.0f);

        /*!
         * Wait on sync that was received in UpdateRenderTargets now that we are
         * actually ready to use the render target
         */
    nvnSyncWait(&m_WindowSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Submit the command buffer to render the cube. */
    nvnQueueSubmitCommands(&m_Queue, 1, &m_CommandHandle);

        /* Present the target to the screen. */
    nvnQueuePresentTexture(&m_Queue, m_pWindow, index);
}

/*
 * MultiThreadedAssetFileLoading::UpdateRenderTargets
 * --------------------------------------------------
 * Gets the index of the current render target from the NVNwindow
 * and records a command buffer that sets it up to be rendered to.
 */
int MultiThreadedAssetFileLoading::UpdateRenderTargets()
{
        /* Get the next target index. */
    NVNwindowAcquireTextureResult result = nvnWindowAcquireTexture(m_pWindow, &m_WindowSync, &m_CurrentWindowIndex);

    NN_ASSERT(result == NVN_WINDOW_ACQUIRE_TEXTURE_RESULT_SUCCESS);

        /* Wait for the previous fram to finish before rewriting the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* Reset the command buffer memory. */
    nvnCommandBufferAddCommandMemory(&m_RenderTargetCommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_RenderTargetCommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_RenderTargetCommandBuffer, m_pRenderTargetControlPool, g_ControlMemorySize);

        /* Record the command buffer to set the target. */
    nvnCommandBufferBeginRecording(&m_RenderTargetCommandBuffer);
    nvnCommandBufferSetRenderTargets(&m_RenderTargetCommandBuffer, 1, &m_pRenderTargets[m_CurrentWindowIndex], NULL, m_pDepthBuffer, NULL);
    m_RenderTargetCommandHandle = nvnCommandBufferEndRecording(&m_RenderTargetCommandBuffer);

        /* Return the index. */
    return m_CurrentWindowIndex;
}

/*
 * MultiThreadedAssetFileLoading::Resize
 * -------------------------------------
 * This method is called everytime the window is resized and is passed
 * the new size of the window. It frees the old render target and creates a new
 * render target with the new screen size. A new set of commands for the command buffer
 * is recorded.
 */
void MultiThreadedAssetFileLoading::Resize(int width, int height)
{
        /* Check for the window being minimized or having no visible surface. */
    if(width == 0 || height == 0)
    {
        return;
    }

        /* Save the current height and width of the screen. */
    m_ScreenHeight = height;
    m_ScreenWidth = width;

        /* Sync to safely rewrite the command buffer. */
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, 0);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);

        /* If it's the first time Resize is called, allocate the NVNwindow. */
    if (m_pWindow == NULL)
    {
        m_pWindow = new NVNwindow;
    }
        /*
         * Otherwise finalize (free) the NVNwindow used for the previous window size.
         * The NVNWindow must be finalized before a render target it owns is finalized.
         */
    else
    {
        nvnWindowFinalize(m_pWindow);
    }

        /* Set up the builder for the render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT | NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);

    for(int i = 0; i < g_NumColorBuffers; ++i)
    {
            /* If it's the first time Resize is called, allocate the render target. */
        if (m_pRenderTargets[i] == NULL)
        {
            m_pRenderTargets[i] = new NVNtexture;
        }
            /* Otherwise finalize (free) the render target used for the previous window size. */
        else
        {
            nvnTextureFinalize(m_pRenderTargets[i]);
        }

        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * i);

            /* Create the texture using the current state of the texture builder. */
        if (!nvnTextureInitialize(m_pRenderTargets[i], &m_RenderTargetBuilder))
        {
            NN_ASSERT(0, "Failed to initialize render target");
        }
    }

        /* If it's the first time Resize is called, allocate the depth buffer. */
    if (m_pDepthBuffer == NULL)
    {
        m_pDepthBuffer = new NVNtexture;
    }
        /* Otherwise finalize (free) the depth buffer used for the previous window size. */
    else
    {
        nvnTextureFinalize(m_pDepthBuffer);
    }

        /* Initialize depth buffer for render target. */
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);
    nvnTextureBuilderSetFlags(&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_COMPRESSIBLE_BIT);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, m_ScreenWidth, m_ScreenHeight);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_DEPTH32F);
    nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, m_pRenderTargetMemoryPool->GetMemoryPool(), m_ColorTargetSize * g_NumColorBuffers);

    if (!nvnTextureInitialize(m_pDepthBuffer, &m_RenderTargetBuilder))
    {
        NN_ASSERT(0, "Failed to initialize depth target");
    }

        /* Pass off the render targets to the window. */
    nvnWindowBuilderSetTextures(&m_WindowBuilder, g_NumColorBuffers, m_pRenderTargets);
    nvnWindowInitialize(m_pWindow, &m_WindowBuilder);

        /* Reset the command buffer memory. */
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, m_pCommandMemoryPool->GetMemoryPool(), m_CommandPoolOffset, g_CommandMemorySize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, g_ControlMemorySize);


        /* Starts the recording of a new set of commands for the given command buffer. */
    nvnCommandBufferBeginRecording(&m_CommandBuffer);
    {
            /* Bind the texture and sampler descriptor pools. */
        m_pTextureIDManager->SetSamplerPool(&m_CommandBuffer);
        m_pTextureIDManager->SetTexturePool(&m_CommandBuffer);

            /* Sets the scissor rectangle and viewport to the full screen */
        nvnCommandBufferSetScissor(&m_CommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);
        nvnCommandBufferSetViewport(&m_CommandBuffer, 0, 0, m_ScreenWidth, m_ScreenHeight);

            /* Clears the currently set render target at a given index. */
        float clearColor[4] = { 0.4f, 0.55f, 0.6f, 1.0f };
        nvnCommandBufferClearColor(&m_CommandBuffer, 0, clearColor, NVN_CLEAR_COLOR_MASK_RGBA);
        nvnCommandBufferClearDepthStencil(&m_CommandBuffer, 1.0, NVN_TRUE, 0, 0);

            /* Bind the render state objects. */
        nvnCommandBufferBindBlendState(&m_CommandBuffer, &m_BlendState);
        nvnCommandBufferBindChannelMaskState(&m_CommandBuffer, &m_ChannelMaskState);
        nvnCommandBufferBindColorState(&m_CommandBuffer, &m_ColorState);
        nvnCommandBufferBindDepthStencilState(&m_CommandBuffer, &m_DepthStencilState);
        nvnCommandBufferBindMultisampleState(&m_CommandBuffer, &m_MultiSampleState);
        nvnCommandBufferBindPolygonState(&m_CommandBuffer, &m_PolygonState);
        nvnCommandBufferSetSampleMask(&m_CommandBuffer, static_cast<uint32_t>(~0));

        NVNbufferAddress vertBufferAddresses[3] = {m_BasicCubeUniformVertexBuffer.GetBufferAddress(),
                                                   m_MipmapCubeUniformVertexBuffer.GetBufferAddress(),
                                                   m_CubeMapUniformVertexBuffer.GetBufferAddress()};
        NVNbufferAddress fragBufferAddresses[3] = {m_BasicCubeUniformFragmentBuffer.GetBufferAddress(),
                                                   m_MipmapCubeUniformFragmentBuffer.GetBufferAddress(),
                                                   m_CubeMapUniformFragmentBuffer.GetBufferAddress()};

        if (m_pShaderScratchMemoryPool)
        {
            nvnCommandBufferSetShaderScratchMemory(&m_CommandBuffer, m_pShaderScratchMemoryPool->GetMemoryPool(), 0, m_ShaderScratchSize);
        }

            /* Render each object in a loop. */
        for(int i = 0; i < g_CubesToRender; ++i)
        {
            std::vector<NVNModelData*>& modelData = m_DataHolders[i]->GetModelData();
            Model* pModel = &modelData[0]->m_Model;

                /* Bind the vertex buffer(s). */
            NVNbufferAddress vboAddr = nvnBufferGetAddress(&modelData[0]->m_VertexBuffer);
            for (size_t j = 0; j < pModel->m_VertexAttributes.size(); ++j)
            {
                VertexAttribute& attr = pModel->m_VertexAttributes[j];
                nvnCommandBufferBindVertexBuffer(&m_CommandBuffer, attr.m_Location, vboAddr + modelData[0]->m_VertexAttributeBufferOffsets[j], attr.m_DataSize);
            }

                /* Bind the vertex states. */
            nvnCommandBufferBindVertexAttribState(&m_CommandBuffer, static_cast<int>(modelData[0]->m_VertexAttributeStates.size()), &modelData[0]->m_VertexAttributeStates[0]);
            nvnCommandBufferBindVertexStreamState(&m_CommandBuffer, static_cast<int>(modelData[0]->m_VertexStreamStates.size()), &modelData[0]->m_VertexStreamStates[0]);

                /* Bind the shader program and uniforms. */
            std::vector<NVNProgramData*>&   programData = m_DataHolders[i]->GetProgramData();
            ShaderTypes::ShaderType shaderType = programData[0]->m_ShaderType;
            uint32_t vertexUniformBlockBindingLocation = ShaderTypes::BlockVS_GetBinding(NVN_SHADER_STAGE_VERTEX, shaderType);
            uint32_t vertexUniformBlockSize = ShaderTypes::BlockVS_GetUniformBlockSize(shaderType);
            nvnCommandBufferBindUniformBuffer(&m_CommandBuffer,
                                              NVN_SHADER_STAGE_VERTEX,
                                              vertexUniformBlockBindingLocation,
                                              vertBufferAddresses[i],
                                              vertexUniformBlockSize);

            uint32_t fragmentUniformBlockBindingLocation = ShaderTypes::BlockFS_GetBinding(NVN_SHADER_STAGE_FRAGMENT, shaderType);
            uint32_t fragmentUniformBlockSize = ShaderTypes::BlockFS_GetUniformBlockSize(shaderType);
            nvnCommandBufferBindUniformBuffer(&m_CommandBuffer,
                                              NVN_SHADER_STAGE_FRAGMENT,
                                              fragmentUniformBlockBindingLocation,
                                              fragBufferAddresses[i],
                                              fragmentUniformBlockSize);

            programData[0]->BindShaderProgram(&m_CommandBuffer);

                /* Draw the primitives. */
            NVNindexType indexType = static_cast<NVNindexType>(pModel->m_IndexData.m_IndexType);
            uint32_t numIndices = pModel->m_IndexData.m_DataSize / pModel->m_IndexData.m_Stride;

            nvnCommandBufferDrawElements(&m_CommandBuffer,
                                         static_cast<NVNdrawPrimitive>(pModel->m_NvnDrawPrimitiveType),
                                         indexType,
                                         numIndices,
                                         nvnBufferGetAddress(&modelData[0]->m_IndexBuffer));
        }
    }

    m_CommandHandle = nvnCommandBufferEndRecording(&m_CommandBuffer);
}//NOLINT(impl/function_size)

// static
void MultiThreadedAssetFileLoading::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char* message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_LOG("NVN Debug Layer Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "Debug layer callback hit");
}

TutorialBaseClass* t()
{
    static MultiThreadedAssetFileLoading s_Tut;
    return (&s_Tut);
}

extern "C" void nnMain()
{
    TutorialRun();
}
