﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NvnTutorial02.cpp,PageSampleNvnTutorial02}
 *
 * @brief
 *  This tutorial demonstrates how to use the GLSLC offline
 *  shader compiler and the NVN texture packager to prepare
 *  assets to be consumed by an application using NVN.
 */

/**
 * @page PageSampleNvnTutorial02 NVN Tutorial 02: GPU Offline Build Tools
 * @tableofcontents
 *
 * @brief
 *  This tutorial demonstrates how to use the GLSLC offline
 *  shader compiler and the NVN texture packager to prepare
 *  assets to be consumed by an application using NVN.
 *
 * @section PageSampleNvnTutorial02_SectionBrief Overview
 *
 * @subsection PageSampleNvnTutorial02_SectionIntroduction Offline Build Tools
 *  This tutorial demonstrates how to use the GLSLC offline
 *  shader compiler and the NVN texture packager to prepare
 *  assets to be consumed by an application using NVN. This
 *  tool loads an text config file specified by the user that
 *  contains files paths to shader source, textures to be
 *  loaded, and model data. An example of this type of file
 *  can be found at Samples/Sources/Resources/NvnTutorial/assets/sampleConfig.txt.
 *  The tool runs the shader source through the GLSLC compiler,
 *  the texture data through the NVN texture packager, and
 *  outputs their results along with the model data in a
 *  binary asset file. All file I/O and work that is done
 *  on the raw asset data can be found in
 *  IntermediateFileManager.cpp.
 *
 * @subsection PageSampleNvnTutorial02_SectionGLSLC GLSLC
 *  GLSLC is used to compile shader source into binary gpu
 *  code to be used by the gpu directly while also outputting
 *  data reflection information for the shader program. The
 *  GLSLC library is loaded in GLSLC_Helper.cpp an is used
 *  in IntermediateFileManager.cpp.
 *
 * @subsection PageSampleNvnTutorial02_SectionTexturePackager NVN Texture Packager
 *  The NVN texture packager consist of three main parts. The
 *  image loader is used to load dds/tga texture files. The
 *  NVN image library performs operations like format conversion
 *  on generic pitch-linear images. The texture exporter converts
 *  the texture data to a format optimized for direct consumption
 *  by the gpu out of an NVNmemoryPool. The texture packager
 *  libraries are loaded in Texpkg_Helper.cpp and used in
 *  IntermediateFileManager.cpp.
 *
 * @section PageSampleNvnTutorial02_SectionFileStructure File Structure
 *  The main tutorial file and other files that make up the tutorial as well as the Visual Studio solutions can be found at
 *  @link ../../../Samples/Sources/Applications/NvnTutorial02OfflineBuildTools Samples/Sources/Applications/NvnTutorial02OfflineBuildTools @endlink
 *
 *  The tutorial library which contains common code shared by the tutorials can be found at
 *  Samples/Sources/Libraries/NvnTutorial
 *
 * @section PageSampleNvnTutorial02_SectionNecessaryEnvironment System Requirements
 *  Only runs on windows.
 *
 * @section PageSampleNvnTutorial02_SectionHowToOperate Operation Procedure
 *
 * @subsection PageSampleNvnTutorial02_SectionCommandLineArguments Command Line Arguments
 * @verbatim
 *  -help:          Brings up the help menu.
 *  -c  (filename): Specify config file with shader/texture inputs
 *                  See sampleConfig.txt in Samples/Sources/Resources/NvnTutorial/assets for an example
 *  -o  (filename): Desired name for output file;
 *  -ip (path):     Path to the folder containing the raw asset files
 *  -op (path):     Path to the folder where output files should be written
 *  -db (bool):     Whether to output debug GLSLC files for shader debugging @endverbatim
 *
 * @section PageSampleNvnTutorial02_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleNvnTutorial02_SectionHowToExecute Execution Procedure
 *  Build the Visual Solution, supply the proper command line arguments, and run it.
 *
 * @section PageSampleNvnTutorial02_SectionDetail Description
 *
 * @subsection PageSampleNvnTutorial02_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  NvnTutorial02.cpp
 *  @includelineno NvnTutorial02.cpp
 *
 * @subsection PageSampleNvnTutorial02_SectionSampleDetail Sample Program Description
 *  This tutorial loads the given text config file and creates an asset ".out"
 *  file from the shader source, textures, and/or model data specified by the
 *  config file.
 */

#include <iostream>
#include <fstream>
#include <string>
#include <unordered_set>
#include <unordered_map>
#include "IntermediateFileManager.h"
#include "Texpkg_Helper.h"

int main(int argc, char* argv[])
{
        /*
         * GLSLC function definitions are made available through linking NvnGlslc.lib.
         * Functions related to the Texture Packager however are retrieved at runtime
         * by loading NvnTexpkg32.dll and NVNImage32.dll at runtime and retrieving the
         * needed function pointers manually.
         */
    if(argc >= 2 && argv[1] == std::string("-help"))
    {
        std::cout << "Help Menu\n";
        std::cout << "----------------------------------------------------------------------------\n";
        std::cout << "-c  <filename>: Specify config file with shader/texture inputs\n";
        std::cout << "                See \"sampleConfig.txt\" for example\n";
        std::cout << "-o  <filename>: Desired name for output file\n";
        std::cout << "-ip <path>:     Path to the folder containing the raw asset files\n";
        std::cout << "-op <path>:     Path to the folder where output files should be written\n";
        std::cout << "-db <bool>:     Whether to output debug GLSLC files for shader debugging\n";
        system("pause");
        return 0;
    }
    else if(argc < 3)
    {
        std::cout << "No config file specified, type \"-help\" for command list.\n";
        system("pause");
        return 1;
    }

    std::string outputFileName = "default.out";

    std::string inPathName;
    std::string outPathName;

    std::string configFileName;
    bool outputDebugGlslcFiles = false;

    int argCounter = 1;
    bool output = false;
    bool config = false;
    bool outPath = false;
    bool inPath = false;
    bool outputDebug = false;

    IntermediateFileManager fileManager;

    while(argCounter < argc)
    {
        if(std::string(argv[argCounter]) == "-o")
        {
            output = true;
            config = false;
            outPath = false;
            inPath = false;
            outputDebug = false;
        }
        else if(std::string(argv[argCounter]) == "-c")
        {
            output = false;
            config = true;
            outPath = false;
            inPath = false;
            outputDebug = false;
        }
        else if (std::string(argv[argCounter]) == "-ip")
        {
            output = false;
            config = false;
            outPath = false;
            inPath = true;
            outputDebug = false;
        }
        else if (std::string(argv[argCounter]) == "-op")
        {
            output = false;
            config = false;
            outPath = true;
            inPath = false;
            outputDebug = false;
        }
        else if (std::string(argv[argCounter]) == "-db")
        {
            output = false;
            config = false;
            outPath = false;
            inPath = false;
            outputDebug = true;
        }
        else if(output)
        {
            outputFileName = argv[argCounter];
        }
        else if(config)
        {
            configFileName = argv[argCounter];

            std::cout << "Loading Config File: " << argv[argCounter] << "\n";

            if(!fileManager.LoadConfigFile(argv[argCounter]))
            {
                std::cout << "Failed to load config file: " << argv[argCounter] << "\n";
                system("pause");
                return 1;
            }
        }
        else if (inPath)
        {
            inPathName = argv[argCounter];
            char check = inPathName.back();
            if(check != '\\' && check != '/')
            {
                inPathName = inPathName + "\\";
            }
        }
        else if (outPath)
        {
            outPathName = argv[argCounter];
            char check = outPathName.back();
            if(check != '\\' && check != '/')
            {
                outPathName = outPathName + "\\";
            }
        }
        else if (outputDebug)
        {
            std::string debugCheck = argv[argCounter];
            if(debugCheck == "true")
            {
                outputDebugGlslcFiles = true;
            }
        }
        else
        {
            std::cout << "Invalid command line input: " << argv[argCounter] << ", type \"-help\" for command list.\n";
            system("pause");
            return 1;
        }

        ++argCounter;
    }

    if (!configFileName.size())
    {
        std::cout << "No config file specified, type \"-help\" for command list.\n";
        system("pause");
        return 1;
    }

    std::cout << "Loading Texpkg\n";
#ifdef _WIN64
    const wchar_t* nvnImageDllName = L"NVNImage.dll";
    const wchar_t* nvnTexPkgDllName = L"NvnTexpkg.dll";
#else
    const wchar_t* nvnImageDllName = L"NVNImage32.dll";
    const wchar_t* nvnTexPkgDllName = L"NvnTexpkg32.dll";
#endif
    if (!TexpkgHelperLoadLibraries(nvnImageDllName, nvnTexPkgDllName))
    {
        std::cout << "Texpkg failed to load\n";
        system("pause");
        return 1;
    }

    std::cout << "\nLoading shader source\n";
    if(!fileManager.LoadShaderPrograms(inPathName))
    {
        std::cout << "Failed to load shader file\n";
        system("pause");
        return 1;
    }

    std::cout << "Compiling shaders\n";
    if(!fileManager.CompileShaderPrograms())
    {
        std::cout << "Shader compilation failed\n";
        system("pause");
        return 1;
    }

    std::cout << "Writing shader reflection headers/source:\n";
    if(!fileManager.OutputUniformBlockHeaderFiles(outPathName))
    {
        std::cout << "Failed to write header file\n";
        system("pause");
        return 1;
    }

    std::cout << "\nLoading texture data\n";
    if(!fileManager.LoadTextureData(inPathName))
    {
        std::cout << "Failed to load texture file\n";
        system("pause");
        return 1;
    }

    std::cout << "Converting textures\n";
    if(!fileManager.ConvertTextures())
    {
        std::cout << "Failed to convert texture data\n";
        system("pause");
        return 1;
    }

    std::cout << "\nWriting output asset file\n";
    if(!fileManager.WriteOutputFile(outPathName, outputFileName, outputDebugGlslcFiles))
    {
        std::cout << "Failed to write output file\n";
        system("pause");
        return 1;
    }

    std::cout << "\nAsset file conversion successful\n\n";
    system("pause");
}//NOLINT(impl/function_size)
