﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{IntermediateFileManager.h,PageSampleNvnTutorial02}
 *
 * @brief
 *  This file defines a class that handles reading config
 *  xml files, loading raw asset data, and writing binary
 *  asset files for the tutorial. The config files are used
 *  to define what texture files need to loaded/converted,
 *  shader source to be compiled into programs, and model
 *  data to be output in the asset file.
 */

#pragma once

#include <string>
#include <unordered_map>
#include <vector>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nvnTool/texpkg/nvnTool_Image.h>
#include <nvnTool/texpkg/nvnTool_HwTexture.h>
#include <Model.h>
#include "ShaderData.h"
#include "TextureData.h"

class IntermediateFileManager
{
    public:
        IntermediateFileManager();
        ~IntermediateFileManager();

        bool LoadConfigFile(std::string fileName);
        bool LoadShaderPrograms(const std::string& path);
        bool CompileShaderPrograms();
        bool OutputUniformBlockHeaderFiles(const std::string& path);

        bool LoadTextureData(const std::string& path);
        bool ConvertTextures();

        bool WriteOutputFile(const std::string& outPath, const std::string& fileName, bool outputDebugGlslcFiles);

        size_t Align(size_t size, size_t alignment, size_t& bytesAlignedUp);
        size_t PadFileBuffer(FILE* file, size_t padSize);

    private:
        bool LoadShaderSource(const std::string& fileName, const std::string& path);
        std::string GetReferenceTypeName(GLSLCpiqTypeEnum type, const std::string& variableName);

        void LoadShaderConfig(std::ifstream& configFile);
        void LoadTextureConfig(std::ifstream& configFile);
        void LoadCubeMapConfig(std::ifstream& configFile);
        void LoadModelConfig(std::ifstream& configFile);
        void LoadVertexAttributeConfig(std::ifstream& configFile, Model& model);
        void LoadIndices(std::ifstream& configFile, Model& model);

        std::unordered_map<std::string, ShaderCompileData>  m_ShaderPrograms;
        std::unordered_map<std::string, std::string> m_RawShaderSource;

        std::vector<TextureData> m_RawTextureData;
        std::vector<CubeMapData> m_RawCubeMapData;
        std::unordered_map<std::string, nvnTool::texpkg::NVNHWTexture> m_ConvertedTextureData;

        std::vector<Model> m_RawModelData;
};
