﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NifmTemporaryNetworkSettings.cpp,PageSampleNifmTemporaryNetworkSettings}
 *
 * @brief
 *  テンポラリ接続設定を指定した利用要求の提出およびソケット通信処理のサンプルプログラムのメイン処理
 */

/**
 * @page PageSampleNifmTemporaryNetworkSettings テンポラリ接続設定を指定した利用要求の提出およびソケット通信
 * @tableofcontents
 *
 * @brief
 *  テンポラリ接続設定を指定した利用要求の提出およびソケット通信処理のサンプルプログラムの解説です。
 *
 * @section PageSampleNifmTemporaryNetworkSettings_SectionBrief 概要
 *  先ずテンポラリ接続設定を作成し、その ID を指定してネットワーク接続の利用要求をシステムに提出します。
 *  その結果、ネットワーク接続の利用が可能であればソケット通信処理を行います。
 *
 * @section PageSampleNifmTemporaryNetworkSettings_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NifmTemporaryNetworkSettings
 *  Samples/Sources/Applications/NifmTemporaryNetworkSettings @endlink 以下にあります。
 *
 * @section PageSampleNifmTemporaryNetworkSettings_SectionNecessaryEnvironment 必要な環境
 *  ネットワーク接続が可能な環境が必要です。
 *
 * @section PageSampleNifmTemporaryNetworkSettings_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNifmTemporaryNetworkSettings_SectionPrecaution 注意事項
 *  ネットワーク接続や通信処理に失敗した場合におこなうべき処理の詳細は未定です
 *
 * @section NifmTemporaryNetworkSettings_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNifmTemporaryNetworkSettings_SectionDetail 解説
 *
 * @subsection PageSampleNifmTemporaryNetworkSettings_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードの一部を引用します。
 *
 *  NifmTemporaryNetworkSettings.cpp
 *  @includelineno NifmTemporaryNetworkSettings.cpp
 *
 * @subsection PageSampleNifmTemporaryNetworkSettings_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの処理の流れは以下の通りです。
 *
 *  - NIFM の初期化を行います。
 *  - テンポラリ接続設定を作成します。
 *  - ネットワーク接続の利用要求をシステムに提出し、ネットワーク接続の利用可否を確認します。
 *  - example.com と HTTP 通信をおこない、取得した内容と結果をログに出力します。
 *
 * このサンプルプログラムの実行結果を以下に示します。ただし、Release 版のサンプルでは Curl の出力（受信内容）は表示されません。
 * また、実機の出力にも Curl の出力（受信内容）は含まれません。
 *
 * @verbinclude NifmTemporaryNetworkSettings_Output.txt
 */

#include "NifmTemporaryNetworkSettings.h"

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>

#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/os.h>
#include <nn/socket.h>
#include <curl/curl.h>

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

// ソケット通信処理のサンプル
// example.com と HTTP 通信をおこない、取得した内容と結果をログに出力します
// この通信に成功した場合は true を、失敗した場合は false を返します
// libcurl の API 詳細については libcurl のマニュアルおよびサンプルを参照してください
bool Program::ExecuteTcpIpCommunication() NN_NOEXCEPT
{
    nn::socket::Initialize(g_SocketConfigWithMemory);

    curl_global_init(CURL_GLOBAL_DEFAULT);

    CURL* curl = curl_easy_init();
    if (curl == nullptr)
    {
        nn::socket::Finalize();

        Printf("curl_easy_init failed.\n");
        return false;
    }

    curl_easy_setopt(curl, CURLOPT_URL, "http://example.com");

    // Debug/Develop 版では、サーバーからの受信内容が表示されます
    CURLcode curlCode = curl_easy_perform(curl);
    Printf("curl_easy_perform returned %d.\n\n", curlCode);

    char *pUrl;
    curl_easy_getinfo(curl, CURLINFO_EFFECTIVE_URL, &pUrl);
    Printf("url: %s\n", pUrl);

    long responseCode = -1;
    curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &responseCode);
    Printf("response code: %d\n", static_cast<int>(responseCode));

    curl_easy_cleanup(curl);

    curl_global_cleanup();

    nn::socket::Finalize();

    return curlCode == CURLE_OK;
}

// アプリケーションのメイン
void Program::Execute() NN_NOEXCEPT
{
    // nifm の API は、利用前にライブラリの初期化が必要です
    nn::nifm::Initialize();

    // 通信処理の前に、ネットワーク接続の利用要求をシステムに提出し、
    // ネットワーク接続の利用可否を確認します

    // 一時的な接続設定（テンポラリ接続設定）の構築

    const nn::nifm::WirelessSettingData wirelessSettingData = {
        {               // ssidConfig
            {           // ssid
                0,      // length
                {}      // hex
            },
            true        // nonBroadcast
        },
        {               // security
            {           // authEncryption
                nn::nifm::Authentication_Open,  // authentication
                nn::nifm::Encryption_None       // encryption
            },
            {           // sharedKey
                0,      // length
                {}      // keyMaterial
            }
        }
    };
    const nn::nifm::IpSettingData ipSettingData = {
        {               // ip
            true,       // isAuto
            {},         // ipAddress
            {},         // subnetMask
            {}          // defaultGateway
        },
        {               // dns
            true,       // isAuto
            {},         // preferredDns
            {}          // alternateDns
        },
        {               // proxy
            false,      // isEnabled
            0,          // port
            "",         // proxy
            {           // authentication
                false,  // isEnabled
                "",     // username
                ""      // password
            }
        },
        1400            // mtu
    };
    const nn::nifm::NetworkProfileData networkProfileData = {
        nn::util::InvalidUuid,  // id
        {},                     // profileName
        nn::nifm::NetworkProfileType_User,  // networkProfileType
        nn::nifm::NetworkInterfaceType::NetworkInterfaceType_Ethernet, // networkInterfaceType
        true,  // isAutoConnect
        true,  // isLargeCapacity
        {
            wirelessSettingData
        },
        ipSettingData
    };
    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(networkProfileData);

    // 利用する設定を利用要求に指定

    nn::nifm::NetworkConnection networkConnection;

    nn::nifm::RequestHandle requestHandle = networkConnection.GetRequestHandle();
    nn::util::Uuid profileId = temporaryNetworkProfile.GetId();
    nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId);

    while (NN_STATIC_CONDITION(1))
    {
#if 1
        // ブロッキング API を利用する場合
        networkConnection.SubmitRequestAndWait();
#else
        // ノンブロッキング API を利用し、ポーリングで完了を確認する場合
        networkConnection.SubmitRequest();
        while (networkConnection.IsRequestOnHold())
        {
            Printf("Network request is on hold...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
#endif

        // 接続要求提出の結果をハンドリングします
        nn::Result result = networkConnection.HandleNetworkRequestResult();

        if (result.IsSuccess())
        {
            Printf("Network is available.\n");
            break;
        }
        else if (result <= nn::nifm::ResultErrorHandlingCompleted())
        {
            // 利用要求を再提出することで受理される可能性があります
            Printf("Network is not available. SubmitNetworkRequest again.\n");
            continue;
        }
        else
        {
            Printf("Network is not available.\n");
            break;
        }
}

    // ネットワーク接続の利用が可能であれば、通信処理をおこないます
    if( networkConnection.IsAvailable() )
    {
        // 通信処理
        bool isSuccessful = ExecuteTcpIpCommunication();

        if ( !isSuccessful )
        {
            // ネットワーク接続が失われたことによる失敗であれば、そのように表示します
            if( networkConnection.IsAvailable() )
            {
                Printf("Communication failed.\n");
            }
            else
            {
                Printf("Lost network.\n");

                // 切断理由表示のために呼び出すこともできます（必須ではありません）
                networkConnection.HandleNetworkRequestResult();
            }
        }
    }
    else
    {
        // 将来的には、失敗した場合のエラーハンドリングを行う必要があります
        // エラーハンドリングの方法については、現在検討中です
        Printf("Network is not available.\n");
    }

    // ネットワーク接続が不要になったら要求を取り下げます
    // 予期しない理由でネットワーク接続が失われていた場合、
    // あらためて明示的に要求を取り下げる必要はありませんが、
    // 重ねて取り下げをおこなっても問題はありません
    networkConnection.CancelRequest();
    NN_ASSERT(!networkConnection.IsAvailable());
} //NOLINT(impl/function_size)
