<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Import network setting

    .DESCRIPTION
        Import network setting
#>

function Resolve-PathSafe
{
    param
    (
        [string] $Path
    )
     
    $ExecutionContext.SessionState.Path.GetUnresolvedProviderPathFromPSPath($Path)
}

function FindFullPath($path, $target)
{
    If(Test-Path $path)
    {
        $ret = (Get-ChildItem -path $path -r -include $target | Sort-Object LastWriteTime -Descending)
        If($null -eq $ret)
        {
            return $null
        }
        Elseif($ret -is [system.array])
        {
            $ret = $ret[0]
        }

        If(Test-Path $ret)
        {
            $ret = Resolve-Path $ret
            return $ret
        }
    }
    return $null
}

$SettingsManagerNsp = FindFullPath "..\..\..\..\Programs\Eris\Outputs\" "SettingsManager.nsp"
$SettingsManagerExe = FindFullPath "..\..\..\..\Programs\Eris\Outputs\" "SettingsManager.exe"

$RunOnTarget = "..\..\..\..\Tools\CommandLineTools\RunOnTarget.exe"
$RunOnTarget = If(Test-Path $RunOnTarget) { Resolve-Path $RunOnTarget } Else { $null }

$ControlTarget = "..\..\..\..\Tools\CommandLineTools\ControlTarget.exe"
$ControlTarget = If(Test-Path $ControlTarget) { Resolve-Path $ControlTarget } Else { $null }

$TempExportFile = Resolve-PathSafe tmp.txt

$SettingsList = Get-ChildItem -r -path .\ -include networkSettings*.txt

###############################

If(($SettingsManagerNsp -eq $null) -And $($SettingsManagerExe -eq $null))
{
    Write-Host "[ERROR] SettingsManager not found." -ForegroundColor Red
    return
}

$Options = @()
If($null -ne $SettingsManagerNsp)
{
    $Options += New-Object System.Management.Automation.Host.ChoiceDescription "&NX", "NX"
}
If($null -ne $SettingsManagerExe)
{
    $Options += New-Object System.Management.Automation.Host.ChoiceDescription "&Win", "Windows"
}

$Target = $host.ui.PromptForChoice("<Target>", "Select target", [System.Management.Automation.Host.ChoiceDescription[]]($Options), 0)

Switch($Options[$Target].label)
{
    "&NX"  { $SettingsManager = $SettingsManagerNsp }
    "&Win" { $SettingsManager = $SettingsManagerExe }
}

Write-Host $SettingsManager

$Options = @()
$Initials = @()
Foreach($i in $SettingsList)
{
    $Index = 0
    $OptionName = $i.Name.Replace("networkSettings_For","").Replace(".txt","")
    For($j = 0; $j -lt $OptionName.length; $j++)
    {
        If(-Not($Initials -contains $OptionName[$j]))
        {
            $Initials += $OptionName[$j]
            $Index = $j
            break
        }
    }
    $OptionName = $OptionName.Substring(0,$Index) + "&" + $OptionName.Substring($Index,$OptionName.length-$Index)
    $Options += new-object System.Management.Automation.Host.ChoiceDescription $OptionName, $i
}

$Setting = $host.ui.PromptForChoice("<Setting>", "Select setting", [System.Management.Automation.Host.ChoiceDescription[]]($Options), 0)

$PInfo = New-Object System.DIagnostics.ProcessStartInfo
$PInfo.RedirectStandardError = $true
$PInfo.RedirectStandardOutput = $true
$PInfo.UseShellExecute = $false

$P = New-Object System.Diagnostics.Process

# improt
Write-Host "Importing " $SettingsList[$Setting].Name
If($SettingsManager -eq $SettingsManagerNsp)
{
    $PInfo.FileName = $RunOnTarget
    $PInfo.Arguments = "$SettingsManager -- Import " + $SettingsList[$Setting]
}
Else
{
    $PInfo.FileName = $SettingsManager
    $PInfo.Arguments = "Import " + $SettingsList[$Setting]
}
$P.StartInfo = $PInfo
$P.Start() | Out-Null
$P.WaitForExit()

$Stdout = $P.StandardOutput.ReadToEnd()
$Stderr = $P.StandardError.ReadToEnd()

If($Stdout.Contains("ERROR") -or $Stderr.Contains("ERROR"))
{
    Write-Host $Stdout $Stderr -ForegroundColor Red
    return
}
Else
{
    Write-Host "[SUCCESS] Import network setting" -ForegroundColor Green
}

# reset
If($SettingsManager -eq $SettingsManagerNsp)
{
    Write-Host "Resetting"
    $PInfo.FileName = $ControlTarget
    $PInfo.Arguments = "reset"
    $P.StartInfo = $PInfo
    $P.Start() | Out-Null
    $P.WaitForExit()

    $Stdout = $P.StandardOutput.ReadToEnd()
    $Stderr = $P.StandardError.ReadToEnd()

    If($Stdout.Contains("ERROR") -or $Stderr.Contains("ERROR"))
    {
        Write-Host $Stdout $Stderr -ForegroundColor Red
        return
    }
    Else
    {
        Write-Host "[SUCCESS] Reset target" -ForegroundColor Green
    }
}

Start-Sleep -s 10

# export
Write-Host "Exporting to " $TempExportFile
If($SettingsManager -eq $SettingsManagerNsp)
{
    $PInfo.FileName = $RunOnTarget
    $PInfo.Arguments = "$SettingsManager -- Export " +  $TempExportFile
}
Else
{
    $PInfo.FileName = $SettingsManager
    $PInfo.Arguments = "Export " +  $TempExportFile
}

$P.StartInfo = $PInfo
$P.Start() | Out-Null
$P.WaitForExit()

$Stdout = $P.StandardOutput.ReadToEnd()
$Stderr = $P.StandardError.ReadToEnd()

If($Stdout.Contains("ERROR") -or $Stderr.Contains("ERROR"))
{
    Write-Host $Stdout $Stderr -ForegroundColor Red
    return
}
Else
{
    Write-Host "[SUCCESS] Export network setting" -ForegroundColor Green
}

