﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include <nn/os.h>

#include <nn/hid.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/util/util_BitFlagSet.h>

#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_ApiScan.h>

struct AccessPointList
{
    int count;
    nn::nifm::AccessPointData data[nn::nifm::AccessPointCountMax];

    void Update() NN_NOEXCEPT
    {
        nn::nifm::GetScanData(data, &count, NN_ARRAY_SIZE(data));

        if (count > NN_ARRAY_SIZE(data))
        {
            count = NN_ARRAY_SIZE(data);
        }
    }

    void Dump() NN_NOEXCEPT
    {
        for (int i = 0; i < count; ++i)
        {
            char ssid[nn::nifm::Ssid::HexSize + 1];
            std::memcpy(ssid, data[i].ssid.hex, nn::nifm::Ssid::HexSize);
            ssid[data[i].ssid.length] = '\0';
            NN_LOG("[%2d] %-32.32s Ch. %3.d RSSI: %3d\n", i, ssid, data[i].channel, data[i].rssi);
        }
    }
};

struct AccessPointStatistics
{
    AccessPointList addedList;
    AccessPointList stayingList;
    AccessPointList removedList;

    void Update(const AccessPointList& previousList, const AccessPointList& newList) NN_NOEXCEPT
    {
        addedList.count = 0;
        stayingList.count = 0;
        removedList.count = 0;

        for (int i = 0; i < newList.count; i++)
        {
            int j = 0;

            for (; j < previousList.count; ++j)
            {
                if (newList.data[i].bssid == previousList.data[j].bssid)
                {
                    break;
                }
            }

            if (j == previousList.count)
            {
                addedList.data[addedList.count++] = newList.data[i];
            }
            else
            {
                stayingList.data[stayingList.count++] = newList.data[i];
            }
        }

        for (int j = 0; j < previousList.count; ++j)
        {
            int i = 0;

            for (; i < newList.count; ++i)
            {
                if (newList.data[i].bssid == previousList.data[j].bssid)
                {
                    break;
                }
            }

            if (i == newList.count)
            {
                removedList.data[removedList.count++] = previousList.data[j];
            }
        }
    }

    void Dump() NN_NOEXCEPT
    {
        NN_LOG("Added (%d):\n", addedList.count);
        addedList.Dump();
        NN_LOG("Staying (%d):\n", stayingList.count);
        stayingList.Dump();
        NN_LOG("Removed (%d):\n", removedList.count);
        removedList.Dump();
    }
};

extern "C" void nnMain()
{
    nn::settings::DebugPadKeyboardMap map;                  // 割り当てを保持するための変数

    map.buttonA     = nn::hid::KeyboardKey::A::Index;
    map.buttonB     = nn::hid::KeyboardKey::B::Index;
    map.buttonStart = nn::hid::KeyboardKey::Return::Index;

    nn::settings::SetDebugPadKeyboardMap(map);              // 割り当てを反映

    NN_LOG("Key mapping ready.\n");

    nn::hid::InitializeDebugPad();

    NN_LOG("hid initialized.\n");

    nn::hid::DebugPadButtonSet buttonMaskConnect = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();
    buttonMaskConnect.Set<nn::hid::DebugPadButton::A>(true);

    nn::hid::DebugPadButtonSet buttonMaskDisconnect = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();
    buttonMaskDisconnect.Set<nn::hid::DebugPadButton::B>(true);

    nn::hid::DebugPadButtonSet buttonMaskScan = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();
    buttonMaskScan.Set<nn::hid::DebugPadButton::X>(true);

    nn::hid::DebugPadButtonSet buttonMaskExit = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();
    buttonMaskExit.Set<nn::hid::DebugPadButton::Start>(true);

    nn::hid::DebugPadButtonSet buttonsPrev = nn::util::MakeBitFlagSet<32, nn::hid::DebugPadButton>();

    nn::nifm::Initialize();

    nn::nifm::NetworkConnection networkConnection;
    bool isConnectedPrev = networkConnection.IsAvailable();

    AccessPointList accessPointList = {};

    NN_LOG("========================================\n");
    NN_LOG("  NIFM Testing Sample\n");
    NN_LOG("\n");
    NN_LOG("  Usage:\n");
    NN_LOG("    A: Connect\n");
    NN_LOG("    B: Disconnect\n");
    NN_LOG("    X: Scan\n");
    NN_LOG("    START: Exit\n");
    NN_LOG("========================================\n");
    NN_LOG(isConnectedPrev ? "Connected.\n" : "No connection.\n" );
    NN_LOG("Ready.\n");

    while( NN_STATIC_CONDITION(true) )
    {
        if (networkConnection.GetSystemEvent().TryWait())
        {
            bool isConnected = networkConnection.IsAvailable();

            NN_LOG(isConnected ? "Connected.\n" : "Disconnected.\n");
            if (!isConnected)
            {
                auto result = networkConnection.GetResult();
                NN_LOG("%03u-%04u\n", result.GetModule(), result.GetDescription());
            }
        }

        nn::hid::DebugPadState pad;
        nn::hid::GetDebugPadState(&pad);

        nn::hid::DebugPadButtonSet buttonsTrigger = pad.buttons & ~buttonsPrev;
        buttonsPrev = pad.buttons;

        if( (buttonsTrigger & buttonMaskConnect).IsAnyOn() )
        {
            NN_LOG("Connecting...\n");
            networkConnection.SubmitRequest();

            NN_LOG("Ready.\n");
        }
        else if( (buttonsTrigger & buttonMaskDisconnect).IsAnyOn() )
        {
            NN_LOG("Disconnecting...\n");
            networkConnection.CancelRequest();

            NN_LOG("Ready.\n");
        }
        else if ((buttonsTrigger & buttonMaskScan).IsAnyOn())
        {
            NN_LOG("Scanning...\n");
            AccessPointList previousList = accessPointList;
            nn::nifm::Scan();
            accessPointList.Update();

            AccessPointStatistics accessPointStatistics;
            accessPointStatistics.Update(previousList, accessPointList);
            accessPointStatistics.Dump();

            NN_LOG("Ready.\n");
        }
        else if( (buttonsTrigger & buttonMaskExit).IsAnyOn() )
        {
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    NN_LOG("End.\n");
}
