﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/TargetConfigs/build_Base.h>
#include <nn/nfp/nfp_Types.h>

namespace nns { namespace nfp { namespace detail {

    template <int Size>
    struct EndianConverter
    {
    };

    template <>
    struct EndianConverter<1>
    {
        template <typename T>
        static T Convert(T value) NN_NOEXCEPT
        {
            return value;
        }
    };

    template <>
    struct EndianConverter<2>
    {
        template <typename T>
        static T Convert(T value) NN_NOEXCEPT
        {
            return (value >> 8) | (value << 8);
        }
    };

    template <>
    struct EndianConverter<4>
    {
        template <typename T>
        static T Convert(T value) NN_NOEXCEPT
        {
            return (value << 24) | ((value << 8) & 0x00FF0000) |
                  ((value >> 8) & 0x0000FF00) | (value >> 24);
        }
    };

    template <>
    struct EndianConverter<8>
    {
        template <typename T>
        static T Convert(T value) NN_NOEXCEPT
        {
            return (value << 56) |
                  ((value << 40) & 0x00FF000000000000ULL) |
                  ((value << 24) & 0x0000FF0000000000ULL) |
                  ((value << 8)  & 0x000000FF00000000ULL) |
                  ((value >> 8)  & 0x00000000FF000000ULL) |
                  ((value >> 24) & 0x0000000000FF0000ULL) |
                  ((value >> 40) & 0x000000000000FF00ULL) |
                   (value >> 56);
        }
    };

    template <typename T>
    inline T ReverseEndian(T value) NN_NOEXCEPT
    {
        return detail::EndianConverter<sizeof(T)>::Convert(value);
    }

    template <typename T>
    inline T ConvertToLittleEndian(T value) NN_NOEXCEPT
    {
        #if defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_LITTLE )
            return value;
        #else
            return detail::EndianConverter<sizeof(T)>::Convert(value);
        #endif
    }

    template <typename T>
    inline T ConvertFromLittleEndian(T value) NN_NOEXCEPT
    {
        return ConvertToLittleEndian(value);
    }

    template <typename T>
    inline T ConvertToBigEndian(T value) NN_NOEXCEPT
    {
        #if defined( NN_BUILD_TARGET_PLATFORM_ENDIAN_BIG )
            return value;
        #else
            return detail::EndianConverter<sizeof(T)>::Convert(value);
        #endif
    }

    template <typename T>
    inline T ConvertFromBigEndian(T value) NN_NOEXCEPT
    {
        return ConvertToBigEndian(value);
    }

}}} // end of namespace nns::nfp::detail

namespace nns { namespace nfp {

    /*!
        @brief      Character ID を示す型です。
     */
    typedef char CharacterId[ nn::nfp::CharacterIdSize ];

    /*!
        @brief      ホストオーダからタグのバイトオーダに変換します。
        @param[in]      value  バイトオーダを変換したい値です。
        @return     value のバイトオーダを変換した値です。
     */
    template <typename T>
    inline T ConvertToTagByteOrder(T value) NN_NOEXCEPT
    {
        return detail::ConvertToBigEndian(value);
    }

    /*!
        @brief      タグのバイトオーダからホストオーダに変換します。
        @param[in]      value       バイトオーダを変換したい値です。
        @return     value のバイトオーダを変換した値です。
     */
    template <typename T>
    inline T ConvertFromTagByteOrder(T value) NN_NOEXCEPT
    {
        return detail::ConvertFromBigEndian(value);
    }

    /*!
        @brief      Character ID を nn::Bit32 型に変換します。
        @param[in]      characterId  変換対象の値です。
        @return     変換後の値です。
     */
    inline nn::Bit32 DecodeCharacterId(const CharacterId& characterId) NN_NOEXCEPT
    {
        return  (characterId[0] << 16) |
                (characterId[1] <<  8) |
                (characterId[2] <<  0);
    }

    /*!
        @brief      同じキャラクターを指す Character ID か確認します。
        @param[in]      id1         比較対象の ID です。
        @param[in]      id2         比較対象の ID です。
        @return     上位 2 バイトが一致していれば同じキャラクターとみなします。
     */
    inline bool IsSameCharacter(nn::Bit32 id1, nn::Bit32 id2) NN_NOEXCEPT
    {
        return (id1 & 0x00FFFF00) == (id2 & 0x00FFFF00);
    }

    /*!
        @brief      Character ID に対応するキャラクタ名を取得します。
        @param[in]      characterId 検索に使用する characterId です。
        @return     対応するキャラクタ名です。見つからない場合は Unknown です。
     */
    const char* GetCharacterName(const CharacterId& characterId) NN_NOEXCEPT;

    /*!
        @brief      指定のキャラクターにアプリケーションが対応しているかを判定します。
        @param[in]      characterId  判定対象の値です。
        @return     対応しているのならば true を返します。
     */
    bool IsKnownAmiibo(const CharacterId& characterId) NN_NOEXCEPT;

    /*!
        @brief      同じタグ ID (UID) かを判定します。
        @param[in]      id1         比較対象の ID です。
        @param[in]      id2         比較対象の ID です。
        @return     同じならば true を返します。
     */
    inline bool IsEqualId(const nn::nfp::TagId& id1, const nn::nfp::TagId& id2) NN_NOEXCEPT
    {
        return (id1.length == id2.length) && (std::memcmp(id1.uid, id2.uid, id2.length) == 0);
    }

}} // end of namespace nns::nfp
