﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{NfpSimple.cpp,PageSampleNfpSimple}

    @brief
    NFP ライブラリのサンプルプログラム
 */

/**
    @page PageSampleNfpSimple NFP ライブラリのサンプル
    @tableofcontents

    @brief
    NFP ライブラリのサンプルプログラムの解説です。

    @section PageSampleNfpSimple_SectionBrief 概要
    NFP ライブラリは、NFC（Near Field Communication）を介して、
    NFP タグ（NFP 用にフォーマットされた NFC タグ）の読み書きを行うライブラリです。
    NFP ライブラリを用いることで、アプリケーションと amiibo を連携させることが可能です。

    このサンプルプログラムは、以下の amiibo 操作に対応しています。
    @n
    @li amiibo のアプリケーション専用領域を読み書き
    @li データが破損した amiibo の復旧
    @li amiibo 設定から amiibo のアプリケーション専用領域を削除
    @li amiibo 設定からデータが破損した amiibo の復旧
    @li amiibo 設定から amiibo のオーナー Mii とニックネームを登録

    サンプルプログラムではアプリケーション専用領域に 4 バイトのカウンタを書き込みます。
    最初にアプリケーション専用領域を作成した時点ではカウンタを 0 に設定し、
    次回以降は書き込みを行うたびにカウンタを 1 つずつインクリメントしていきます。

    @section PageSampleNfpSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/NfpSimple
    Samples/Sources/Applications/NfpSimple @endlink 以下にあります。

    @section PageSampleNfpSimple_SectionNecessaryEnvironment 必要な環境
    事前に NoftWriter ツールを使用して NFP ライブラリで読み書き可能な NFP タグを用意してください。
    ただし、サンプルプログラムは amiibo データリストに登録されている一部のキャラクターにしか対応しておらず、
    非対応のキャラクターを書き込んだ amiibo に対してはアプリケーション専用領域の読み書きができません。
    現在、以下のキャラクタに対応しています。
    @snippet TagUtility.cpp SampleCharacterName

    @section PageSampleNfpSimple_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、NFP ライブラリの状態が表示されます。
    ゲームパッドの操作により、NFP ライブラリの状態を変化させることができます。

    @section PageSampleNfpSimple_SectionPrecaution 注意事項
    サンプルプログラムを実行する開発機に、一度もタッチされたことがない amiibo のデータが破損している場合、
    バックアップデータが保存されていないため、データを復旧することができません。
    事前に amiibo 設定から amiibo を初期化するか、バックアップデータが存在する開発機を使用して amiibo のデータを復旧してください。

    @section PageSampleNfpSimple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleNfpSimple_SectionDetail 解説
    NfpSimple では amiibo の状態によって異なる操作を実行します。
    特定の操作を狙って実行するためには、事前に以下の条件を満たす amiibo を用意しておく必要があります。

    @li アプリケーション専用領域の作成@n
    NoftWriter ツールで amiibo データリストを書き込んだばかりの amiibo を用意します。

    @li アプリケーション専用領域の削除@n
    NfpSimple のアクセス ID と異なるアプリケーションで書き込んだ amiibo を用意します。

    @li データが破損した amiibo の復旧@n
    データが破損した amiibo を用意します。

    @li オーナー Mii とニックネームの登録@n
    NoftWriter ツールで amiibo データリストを書き込んだばかりの amiibo を用意します。
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include "Graphics.h"
#include "HidController.h"

#include "Config.h"
#include "UpdateState.h"
#include "WriteState.h"

namespace {

    // Graphics.cpp で malloc しているヒープの内訳
    //  VisiblePoolMemory       16 MB
    //  InvisiblePoolMemory     20 MB
    //  for CommandBuffer       32 MB
    //  nv::InitializeGraphics   8 MB
    //  for DebugFont            2 MB
    const size_t MemoryHeapSize         = 128 * 1024 * 1024;
    const size_t MallocHeapSize         =  96 * 1024 * 1024;

    // アプリケーションで扱うデータと状態です。
    nns::nfp::ApplicationData g_Data;

} // namespace

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// アプリケーションのエントリポイント
extern "C" void nnMain()
{
    InitializeGraphicSystem();
    InitializeHidController();

    while( NN_STATIC_CONDITION(true) )
    {
        UpdateHidController();

        nns::nfp::UpdateState(g_Data);

        BeginText();
        nns::nfp::WriteState(g_Data);
        EndText();

        if( g_Data.state == nns::nfp::State_Exit )
        {
            break;
        }
    }

    FinalizeHidController();
    FinalizeGraphicSystem();
}
