﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif
#if defined( NN_BUILD_TARGET_PLATFORM_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#include "Graphics.h"

// 文字色の定義
const nn::util::Unorm8x4 Color::White   = { { 255, 255, 255, 255 } };
const nn::util::Unorm8x4 Color::Black   = { {  51,  51,  51, 255 } };
const nn::util::Unorm8x4 Color::Gray    = { { 153, 153, 153, 255 } };
const nn::util::Unorm8x4 Color::Green   = { { 153, 255,   0, 255 } };
const nn::util::Unorm8x4 Color::Orange  = { { 255, 153,   0, 255 } };
const nn::util::Unorm8x4 Color::Red     = { { 255,  25,   0, 255 } };

// デバッグフォント
// 参照：Samples/Sources/Applications/DebugFontSimple Samples/Sources/Applications/DebugFontSimple.cpp
namespace {

//------------------------------------------------------------------------------
//  レイヤを初期化
//------------------------------------------------------------------------------
nn::vi::Display* g_pDisplay;
nn::vi::Layer* g_pLayer;

const int RenderWidth = 1280;
const int RenderHeight = 720;

void InitializeLayer()
{
    nn::Result result = nn::vi::OpenDefaultDisplay( &g_pDisplay );
    NN_ASSERT( result.IsSuccess() );
    NN_UNUSED( result );

    result = nn::vi::CreateLayer( &g_pLayer, g_pDisplay );
    NN_ASSERT( result.IsSuccess() );

    result = nn::vi::SetLayerScalingMode( g_pLayer, nn::vi::ScalingMode_FitToLayer );
    NN_ASSERT( result.IsSuccess() );
}

//------------------------------------------------------------------------------
//  デバイスを初期化
//------------------------------------------------------------------------------
nn::gfx::Device g_Device;

void InitializeDevice()
{
    nn::gfx::Device::InfoType info;
    info.SetDefault();
    info.SetApiVersion(nn::gfx::ApiMajorVersion, nn::gfx::ApiMinorVersion);
    g_Device.Initialize(info);
}

//------------------------------------------------------------------------------
//  メモリプールを初期化
//------------------------------------------------------------------------------
static const size_t g_VisiblePoolMemorySize = 16 * 1024 * 1024;
static const size_t g_InvisiblePoolMemorySize = 20 * 1024 * 1024;
void* g_pVisiblePoolMemory = NULL;
void* g_pInvisiblePoolMemory = NULL;
void* g_pMemoryPoolStart = NULL;
ptrdiff_t g_MemoryPoolOffset = 0;
void* g_pInvisibleMemoryPoolStart = NULL;
ptrdiff_t g_InvisibleMemoryPoolOffset = 0;
nn::gfx::MemoryPool g_MemoryPool;
nn::gfx::MemoryPool g_InvisibleMemoryPool;

void InitializeMemoryPool()
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuUncached
        | nn::gfx::MemoryPoolProperty_GpuCached);

    // Determine alignment to allocate space on an alignment boundary
    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
    g_pVisiblePoolMemory = malloc(g_VisiblePoolMemorySize + alignment);

    g_pMemoryPoolStart = nn::util::BytePtr(g_pVisiblePoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(g_pMemoryPoolStart, nn::util::align_down(g_VisiblePoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
    g_MemoryPool.Initialize(&g_Device, info);

    g_MemoryPoolOffset = 0;
}

void InitializeInvisibleMemoryPool()
{
    nn::gfx::MemoryPool::InfoType info;
    info.SetDefault();
    info.SetMemoryPoolProperty(nn::gfx::MemoryPoolProperty_CpuInvisible
        | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);

    // Determine alignment to allocate space on an alignment boundary
    size_t alignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(&g_Device, info);
    g_pInvisiblePoolMemory = malloc(g_InvisiblePoolMemorySize + alignment);

    g_pInvisibleMemoryPoolStart = nn::util::BytePtr(g_pInvisiblePoolMemory).AlignUp(alignment).Get();
    info.SetPoolMemory(g_pInvisibleMemoryPoolStart, nn::util::align_down(g_InvisiblePoolMemorySize,
        nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(&g_Device, info)));
    g_InvisibleMemoryPool.Initialize(&g_Device, info);

    g_InvisibleMemoryPoolOffset = 0;
}

//------------------------------------------------------------------------------
//  スワップチェーンを初期化
//------------------------------------------------------------------------------
nn::gfx::SwapChain g_SwapChain;

void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;

    info.SetDefault();
    info.SetLayer(g_pLayer);
    info.SetWidth(RenderWidth);
    info.SetHeight(RenderHeight);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
    info.SetBufferCount(2);
    if (NN_STATIC_CONDITION(nn::gfx::SwapChain::IsMemoryPoolRequired))
    {
        size_t size = g_SwapChain.CalculateScanBufferSize(&g_Device, info);
        g_InvisibleMemoryPoolOffset = nn::util::align_up(g_InvisibleMemoryPoolOffset,
            nn::gfx::SwapChain::GetScanBufferAlignment(&g_Device, info));
        g_SwapChain.Initialize(&g_Device, info, &g_InvisibleMemoryPool, g_InvisibleMemoryPoolOffset, size);
        g_InvisibleMemoryPoolOffset += size;
    }
    else
    {
        g_SwapChain.Initialize(&g_Device, info, NULL, 0, 0);
    }
}

//------------------------------------------------------------------------------
//  キューを初期化
//------------------------------------------------------------------------------
nn::gfx::Queue g_Queue;

void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    g_Queue.Initialize(&g_Device, info);
}

//------------------------------------------------------------------------------
//  コマンドバッファを初期化
//------------------------------------------------------------------------------
nn::gfx::CommandBuffer g_CommandBuffer;

void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability(nn::gfx::QueueCapability_Graphics);
    info.SetCommandBufferType(nn::gfx::CommandBufferType_Direct);
    g_CommandBuffer.Initialize(&g_Device, info);
}

//------------------------------------------------------------------------------
//  ビューポートシザーを初期化
//------------------------------------------------------------------------------
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth(static_cast< float >(RenderWidth));
        viewportInfo.SetHeight(static_cast< float >(RenderHeight));
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth(RenderWidth);
        scissorInfo.SetHeight(RenderHeight);
    }
    info.SetViewportStateInfoArray(&viewportInfo, 1);
    info.SetScissorStateInfoArray(&scissorInfo, 1);
    g_ViewportScissor.Initialize(&g_Device, info);
}

//------------------------------------------------------------------------------
//  サンプラディスクリプタプールの初期化
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool g_SamplerDescriptorPool;
int g_SamplerDescriptorBaseIndex = 0;

void InitializeSamplerDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_Sampler);
    info.SetSlotCount(g_SamplerDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
    g_SamplerDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
    g_MemoryPoolOffset += size;
}

//------------------------------------------------------------------------------
//  テクスチャディスクリプタプールの初期化
//------------------------------------------------------------------------------
nn::gfx::DescriptorPool g_TextureDescriptorPool;
int g_TextureDescriptorBaseIndex = 0;

void InitializeTextureDescriptorPool()
{
    nn::gfx::DescriptorPool::InfoType info;
    info.SetDefault();
    info.SetDescriptorPoolType(nn::gfx::DescriptorPoolType_TextureView);
    info.SetSlotCount(g_TextureDescriptorBaseIndex + 1);
    size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize(&g_Device, info);
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
        nn::gfx::DescriptorPool::GetDescriptorPoolAlignment(&g_Device, info));
    g_TextureDescriptorPool.Initialize(&g_Device, info, &g_MemoryPool, g_MemoryPoolOffset, size);
    g_MemoryPoolOffset += size;
}

//------------------------------------------------------------------------------
//  gfx オブジェクトの初期化
//------------------------------------------------------------------------------
nn::util::BytePtr g_pMemoryHeap(NULL);
nn::util::BytePtr g_pMemory(NULL);

void InitializeGfxObjects()
{
    g_pMemoryHeap.Reset(malloc(1024 * 1024 * 32));
    g_pMemory = g_pMemoryHeap;

    InitializeDevice();

#if NN_GFX_IS_TARGET_NVN
    nn::gfx::Device::DataType& deviceData = nn::gfx::AccessorToData(g_Device);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_TEXTURE_DESCRIPTORS, &g_TextureDescriptorBaseIndex);
    nvnDeviceGetInteger(deviceData.pNvnDevice,
        NVN_DEVICE_INFO_RESERVED_SAMPLER_DESCRIPTORS, &g_SamplerDescriptorBaseIndex);
#endif

    InitializeMemoryPool();
    InitializeInvisibleMemoryPool();

    InitializeSwapChain();
    InitializeQueue();

    InitializeCommandBuffer();
    InitializeViewport();

    InitializeSamplerDescriptorPool();
    InitializeTextureDescriptorPool();

    NN_ASSERT(g_pMemoryHeap.Distance(g_pMemory.Get()) < 1024 * 1024 * 32);
    NN_ASSERT(g_MemoryPoolOffset < g_VisiblePoolMemorySize);
    NN_ASSERT(g_InvisibleMemoryPoolOffset < g_InvisiblePoolMemorySize);
}

//------------------------------------------------------------------------------
//  gfx オブジェクトの終了処理
//------------------------------------------------------------------------------
void FinalizeGfxObjects()
{
    // 各オブジェクトを破棄
    g_TextureDescriptorPool.Finalize(&g_Device);
    g_SamplerDescriptorPool.Finalize(&g_Device);

    g_ViewportScissor.Finalize(&g_Device);
    g_CommandBuffer.Finalize(&g_Device);
    g_SwapChain.Finalize(&g_Device);
    g_Queue.Finalize(&g_Device);
    g_InvisibleMemoryPool.Finalize(&g_Device);
    g_MemoryPool.Finalize(&g_Device);
    g_Device.Finalize();

    free(g_pMemoryHeap.Get());
    free(g_pVisiblePoolMemory);
    free(g_pInvisiblePoolMemory);
}

} // namespace

//------------------------------------------------------------------------------
//  グラフィックス機能の初期化・終了、描画処理
//------------------------------------------------------------------------------
namespace {

    bool g_Initialized;

    nn::gfx::util::DebugFontTextWriter g_Writer;
    nn::util::BytePtr g_DebugFontHeap(nullptr);

    const float WriteCharWidth  = 10;
    const float WriteCharHeight = 25;

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

bool IsGraphicSystemInitialized() NN_NOEXCEPT
{
    return g_Initialized;
}

void InitializeGraphicSystem() NN_NOEXCEPT
{
    NN_ASSERT( !g_Initialized );

#if defined( NN_BUILD_TARGET_PLATFORM_NX )
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif

    nn::vi::Initialize();
    InitializeLayer();

    nn::gfx::Initialize();
    InitializeGfxObjects();

    // デバッグフォント初期化
    const bool userMemoryPoolEnable = false;    // true にすると、ユーザーのメモリプールを使用します
    const int charCountMax = 1024;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetUserMemoryPoolEnabled(userMemoryPoolEnable);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(&g_Device, info);
    g_DebugFontHeap.Reset(new uint8_t[debugFontHeapSize]);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(&g_Device, info);

    g_Writer.Initialize(
        &g_Device,
        info,
        g_DebugFontHeap.Get(),
        debugFontHeapSize,
        userMemoryPoolEnable ? &g_MemoryPool : nullptr,
        userMemoryPoolEnable ? g_MemoryPoolOffset : 0,
        userMemoryPoolEnable ? debugFontMemoryPoolSize : 0
        );

    g_MemoryPoolOffset += userMemoryPoolEnable ? debugFontMemoryPoolSize : 0;

    g_Writer.SetDisplayWidth(RenderWidth);
    g_Writer.SetDisplayHeight(RenderHeight);
    g_Writer.SetTextureDescriptor(&g_TextureDescriptorPool, g_TextureDescriptorBaseIndex);
    g_Writer.SetSamplerDescriptor(&g_SamplerDescriptorPool, g_SamplerDescriptorBaseIndex);

    g_Initialized = true;
}

void BeginText() NN_NOEXCEPT
{
    NN_ASSERT( g_Initialized );
}

void EndText() NN_NOEXCEPT
{
    NN_ASSERT( g_Initialized );

    // コマンド生成
    g_CommandBuffer.Reset();
    g_MemoryPoolOffset = nn::util::align_up(g_MemoryPoolOffset,
        nn::gfx::CommandBuffer::GetCommandMemoryAlignment(&g_Device));
    g_CommandBuffer.AddCommandMemory(&g_MemoryPool, g_MemoryPoolOffset, 1024 * 1024);
    g_pMemory.AlignUp(256);
    g_CommandBuffer.AddControlMemory(g_pMemory.Get(), 256);
    g_CommandBuffer.Begin();
    {
        nn::gfx::ColorTargetView* pTarget = g_SwapChain.AcquireNextScanBufferView();
        g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
        g_CommandBuffer.SetDescriptorPool(&g_TextureDescriptorPool);
        g_CommandBuffer.SetDescriptorPool(&g_SamplerDescriptorPool);
        g_CommandBuffer.ClearColor(pTarget, 0.1f, 0.1f, 0.1f, 1.0f, NULL);
        g_CommandBuffer.SetRenderTargets(1, &pTarget, NULL);
        g_CommandBuffer.SetViewportScissorState(&g_ViewportScissor);

        // デバッグフォント用のコマンド生成
        g_Writer.Draw(&g_CommandBuffer);
    }
    g_CommandBuffer.End();

    // コマンドの実行
    g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);

    // 結果の表示
    g_Queue.Present(&g_SwapChain, 1);
    g_Queue.Sync();
}

void FinalizeGraphicSystem() NN_NOEXCEPT
{
    NN_ASSERT( g_Initialized );

    // デバッグフォント終了
    g_Writer.Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_DebugFontHeap.Get());
    g_DebugFontHeap.Reset(nullptr);

    FinalizeGfxObjects();
    nn::gfx::Finalize();

    nn::vi::DestroyLayer( g_pLayer );
    nn::vi::CloseDisplay( g_pDisplay );
    nn::vi::Finalize();
}

//------------------------------------------------------------------------------
//  文字列の描画
//------------------------------------------------------------------------------
void WriteText(
        int x, int y,
        const char* text,
        const nn::util::Unorm8x4& color
    ) NN_NOEXCEPT
{
    g_Writer.SetTextColor(color);
    g_Writer.SetScale(1.0f, 1.0f);
    g_Writer.SetCursor(WriteCharWidth * x, WriteCharHeight * y);
    g_Writer.Print(text);
}

void WriteText(
        int x, int y,
        const char* text
    ) NN_NOEXCEPT
{
    WriteText(x, y, text, Color::White);
}

void WriteErrorText(
        int x, int y,
        const char* text
    ) NN_NOEXCEPT
{
    WriteText(x, y, text, Color::Red);
}

void WriteLoadingText(
        int x, int y,
        const char* text,
        int elapsedFrame
    ) NN_NOEXCEPT
{
    int len = static_cast<int>(std::strlen(text));
    g_Writer.SetTextColor(Color::Green);
    g_Writer.SetScale(1.0f, 1.0f);
    g_Writer.SetCursor(WriteCharWidth * x, WriteCharHeight * y);
    g_Writer.Print(text);

    for( int i = 0; i < (elapsedFrame / 10) % 6; ++i )
    {
        g_Writer.SetCursor(WriteCharWidth * (x + i + len), WriteCharHeight * y);
        g_Writer.Print(".");
    }
}

nn::vi::Layer* GetViLayer() NN_NOEXCEPT
{
    return g_pLayer;
}
