﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nfp.h>
#include "TagUtility.h"

#define NFPDEMO_SEARCH_METHOD_SEQUENTIAL 's'
#define NFPDEMO_SEARCH_METHOD_BINARY     'b'
#define NFPDEMO_SEARCH_METHOD            NFPDEMO_SEARCH_METHOD_SEQUENTIAL

namespace nns { namespace nfp { namespace {

    //! サンプルデモでアクセス可能な CharacterID のリストです。
    const nn::Bit32 CharacterIdList[] =
    {
        0x000000,
        0x000100,
        0x000200,
        0x000300,
        0x000400,
        0x000500,
        0x000600,
        0x000700,
        0x000800,
        0x000900,
        0x000A00,
        0x001300,
        0x001400,
        0x001500,
        0x001700,
        0x002300,
        0x008000,
        0x010000,
        0x010100,
        0x010200,
        0x010300,
        0x010500,
        0x010600,
        0x010700,
        0x010800,
        0x014000,
        0x014100,
        0x018000,
        0x018100,
        0x018200,
        0x018300,
        0x018400,
        0x018800,
        0x018A00,
        0x018B00,
        0x018C00,
        0x018D00,
        0x018E00,
        0x019200,
        0x019300,
        0x019400,
        0x019600,
        0x01C100,
        0x058000,
        0x058100,
        0x05C000,
        0x05C100,
        0x060000,
        0x064000,
        0x064200,
        0x06C000,
        0x070000,
        0x074000,
        0x074100,
        0x074200,
        0x078000,
        0x078100,
        0x078200,
        0x07C000,
        0x080000,
        0x080100,
        0x080200,
        0x190600,
        0x191900,
        0x192700,
        0x199600,
        0x1AC000,
        0x1B9200,
        0x1F0000,
        0x1F0100,
        0x1F0200,
        0x1F0300,
        0x1F4000,
        0x210000,
        0x210100,
        0x210200,
        0x210300,
        0x210400,
        0x210500,
        0x210600,
        0x210700,
        0x210800,
        0x210900,
        0x224000,
        0x228000,
        0x228100,
        0x22C000,
        0x320000,
        0x324000,
        0x334000,
        0x348000,
        0x34C000,
        0x360000
    };

    //! サンプルデモで利用できるキャラクタの名前です。
    const char* CharacterNameList[] =
    {
        //! [SampleCharacterName]
        "Mario",
        "Luigi",
        "Peach",
        "Yoshi",
        "Rosalina",
        "Bowser",
        "Bowser Jr.",
        "Wario",
        "Donkey Kong",
        "Diddy Kong",
        "Toad",
        "Daisy",
        "Waluigi",
        "Goomba",
        "Boo",
        "Koopa Troopa",
        "Poochy",
        "Link",
        "Zelda",
        "Ganon",
        "Wolf Link",
        "Daruk",
        "Urbosa",
        "Mipha",
        "Revali",
        "Guardian",
        "Bokoblin",
        "Villager",
        "Isabelle",
        "K.K.",
        "Tom Nook",
        "Timmy & Tommy",
        "Mabel",
        "Reese",
        "Cyrus",
        "Digby",
        "Rover",
        "Resetti",
        "Blathers",
        "Celeste",
        "Kicks",
        "Kapp'n",
        "Lottie",
        "Fox",
        "Falco",
        "Samus",
        "Metroid",
        "Captain Falcon",
        "Olimar",
        "Pikmin",
        "Little Mac",
        "Wii Fit Trainer",
        "Pit",
        "Dark Pit",
        "Palutena",
        "Mr. Game & Watch",
        "R.O.B.",
        "Duck Hunt",
        "Mii Brawler",
        "Inkling",
        "Callie",
        "Marie",
        "Charizard",
        "Pikachu",
        "Jigglypuff",
        "Mewtwo",
        "Lucario",
        "Greninja",
        "Kirby",
        "Meta Knight",
        "King Dedede",
        "Waddle Dee",
        "Qbby",
        "Marth",
        "Ike",
        "Lucina",
        "Robin",
        "Roy",
        "Corrin",
        "Alm",
        "Celica",
        "Chrom",
        "Tiki",
        "Shulk",
        "Ness",
        "Lucas",
        "Chibi-Robo",
        "Sonic the Hedgehog",
        "Bayonetta",
        "PAC-MAN",
        "Mega Man",
        "Ryu",
        "Cloud"
        //! [SampleCharacterName]
    };

    const int CharacterIdCount = sizeof(CharacterNameList) / sizeof(CharacterNameList[0]);
    NN_STATIC_ASSERT( CharacterIdCount == (sizeof(CharacterIdList) / sizeof(CharacterIdList[0])) );

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    const char* GetCharacterName(const CharacterId& characterId) NN_NOEXCEPT
    {
        nn::Bit32 id = DecodeCharacterId(characterId);

        #if NFPDEMO_SEARCH_METHOD == NFPDEMO_SEARCH_METHOD_BINARY
        int min = 0;
        int max = CharacterIdCount - 1;
        while( min <= max )
        {
            int mid = (min + max) / 2;
            if( IsSameCharacter(CharacterIdList[mid], id) )
            {
                return CharacterNameList[mid];
            }
            else if( CharacterIdList[mid] < id )
            {
                min = mid + 1;
            }
            else
            {
                max = mid - 1;
            }
        }
        #elif NFPDEMO_SEARCH_METHOD == NFPDEMO_SEARCH_METHOD_SEQUENTIAL
        for( int i = 0; i < CharacterIdCount; ++i )
        {
            if( IsSameCharacter(id, CharacterIdList[i]) )
            {
                return CharacterNameList[i];
            }
        }
        #endif

        return "Unknown";
    }

    bool IsKnownAmiibo(const CharacterId& characterId) NN_NOEXCEPT
    {
        // アプリケーションが対応する Character ID であることを確認します。
        nn::Bit32 id = DecodeCharacterId(characterId);
        int i = 0;
        for(i = 0; i < CharacterIdCount; ++i )
        {
            if( IsSameCharacter(id, CharacterIdList[i]) )
            {
                break;
            }
        }
        return i < CharacterIdCount;
    }

}} // end of namespace nns::nfp
