﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nfp.h>
#include "Config.h"
#include "TagAccessorBase.h"
#include "PlayerTagAccessor.h"

namespace nns { namespace nfp {

    /*!
        @brief      デバイスの情報です。
     */
    struct DeviceInfo
    {
        nn::nfp::DeviceHandle               deviceHandle;
        nn::hid::NpadIdType                 npadId;
    };

    /*!
        @brief      スレッドに依頼する処理のコマンドです。
     */
    enum Command
    {
        Command_None = TagAccessorBase::Command_None,
        Command_Initialize,
        Command_Finalize,
        Command_ListDevices,
        Command_AttachAvailabilityChangeEvent,

        Command_CountMax
    };

    /*!
        @brief      NFP ライブラリの処理に非同期に行うクラスです。

        @details
                    NFP ライブラリで提供される API のほとんどは長時間処理をブロックしてしまいます。
                    このクラスは NFP ライブラリの API を別スレッドで実行する実装のサンプルです。

                    実行する処理に対応するコマンド識別子を @ref Enqueue() で登録し、
                    @ref Run() を呼びだすことで登録した処理をバックグラウンドで実行します。
                    登録できるコマンド識別子の最大数は @ref CommandCountMax です。

                    登録した処理が全て完了した場合、あるいは何らかのエラーが発生して中断された場合には
                    @ref Run() の引数に指定したイベントにシグナルが発行されます。
                    いくつの処理が実行されたかと最後の処理結果は、
                    @ref GetExecutedCount() と @ref GetLastResult() で取得することができます。

                    本クラスのメソッドはスレッドセーフではありませんので、一つのスレッドから呼び出すようにしてください。
     */
    class TagAccessor : public TagAccessorBase
    {
        NN_DISALLOW_COPY( TagAccessor );
        NN_DISALLOW_MOVE( TagAccessor );

    public:

        /*!
            @brief      TagAccessor のインスタンスを生成します。
            @return     TagAccessor のインスタンスです。
        */
        static TagAccessor& CreateInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスを取得します。
            @return     TagAccessor のインスタンスです。
        */
        static TagAccessor& GetInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスを破棄します。
        */
        static void DestroyInstance() NN_NOEXCEPT;

        /*!
            @brief      TagAccessor のインスタンスが存在するか判定します。
            @return     存在する場合は true を返します。
        */
        static bool HasInstance() NN_NOEXCEPT;

        /*!
            @brief      NFP ライブラリを初期化します。
            @return     登録された処理の位置です。
        */
        int Initialize() NN_NOEXCEPT;

        /*!
            @brief      NFP ライブラリを終了します。
            @return     登録された処理の位置です。

            @details    タグがマウントされている場合には自動的にアンマウントされ、検知も停止します。
                        NFC デバイスの状態が DeviceState_Active 状態または DeviceState_Mount 状態であれば、
                        @ref AttachEvents() で設定した Deactivate イベントが発生します。
        */
        int Finalize() NN_NOEXCEPT;

        /*!
            @brief      NFC デバイス情報のリストを取得します。
            @param[out]     pOutBuffer      取得した情報のリストを格納するバッファへのポインタを指定します。
            @param[out]     pOutCount       実際に取得した情報の数を格納するバッファへのポインタを指定します。
            @param[in]      bufferCount     取得する情報の最大数（バッファの要素数）を指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。
        */
        int ListDevices(DeviceInfo* pOutBuffer, int* pOutCount, int bufferCount) NN_NOEXCEPT;

        /*!
            @brief      デバイスの利用可/不可の変化を通知するイベントを設定します。
            @param[out]     pAvailabilityChangeEvent      通知に使用するイベントを指定します。
            @return     登録された処理の位置です。

            @details    引数に指定する値は @ref Run() で開始されるバックグラウンド処理が完全に終了し、
                        @ref Run() の引数で指定したイベントがシグナルされるまで有効でなければいけません。

                        引数として渡すイベントの初期化と解放は NFP ライブラリ側が担当します。
        */
        int AttachAvailabilityChangeEvent(nn::os::SystemEventType* pAvailabilityChangeEvent) NN_NOEXCEPT;

        /*!
            @brief      指定番目の登録された処理の結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      index       登録した処理の位置です。
            @return     指定番目の登録された処理の有無です。
                        指定番目の処理が実行されなかった場合には false を返します。
         */
        bool GetResult(nn::Result* pOutResult, int index) const NN_NOEXCEPT
        {
            return TagAccessorBase::GetNthResult(pOutResult, index);
        }

        /*!
            @brief      指定されたコマンド識別子の実行結果を取得します。
            @param[out]     pOutResult  結果を格納するバッファへのポインタを指定します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @return     指定されたコマンド識別子の実行結果の有無です。
                        指定されたコマンド識別子が実行されなかった場合には false を返します。

            @details    複数回実行されたコマンド識別子を指定した場合には、先の実行結果を取得します。
         */
        bool GetResult(nn::Result* pOutResult, Command command) const NN_NOEXCEPT
        {
            return TagAccessorBase::GetCommandResult(pOutResult, static_cast<uint32_t>(command));
        }

        /*!
            @brief      タグに処理を行っているかを判定します。
            @return     タグへの処理中なら真を返します。
         */
        virtual bool IsBusy() NN_NOEXCEPT NN_OVERRIDE;

        /*!
            @brief      PlayerTagAccessor を取得します。
            @param[in]      playerIndex     対象プレイヤーの Index です。
            @return     PlayerTagAccessor です。
        */
        PlayerTagAccessor& GetPlayerTagAccessor(int playerIndex) NN_NOEXCEPT;

    protected:

        /*!
            @brief      コマンド識別子で表される処理を実行します。
            @param[in]      command     実行する処理を表すコマンド識別子です。
            @param[in]      param       実行する処理に渡すパラメータです。
            @return     実行された処理の結果です。
         */
        virtual nn::Result Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT NN_OVERRIDE;

    private:

        /*!
            @brief      NFP ライブラリの処理をバックグラウンドで実行するスレッドを開始します。
         */
        TagAccessor() NN_NOEXCEPT {}

        /*!
            @brief      スレッドが稼働中であれば終了します。
         */
        virtual ~TagAccessor() NN_NOEXCEPT NN_OVERRIDE {}

        //! 唯一存在する TagAccessor のインスタンスのポインタです。
        static TagAccessor* g_pInstance;

        PlayerTagAccessor m_PlayerTagAccessor[ PlayerCountMax ];
    };

}} // end of namespace nns::nfp
