﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include "Config.h"
#include "PlayerTagAccessor.h"

namespace nns { namespace nfp { namespace {

    typedef nn::Result (*ProcessFunc)( const void* pParam );

    struct ArgDeviceHandle
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    // NFP ライブラリ API が ResultNeedRetry() を返す場合、成功するまで RetryCountMax 回再試行します。
    const int RetryCountMax = 5;
    const int RetryInterval = 200;

    nn::Result ProcessRetryFunction(const void* pParam, ProcessFunc pFunc) NN_NOEXCEPT
    {
        nn::Result result;
        for( int i = 0; i < RetryCountMax; ++i )
        {
            result = (*pFunc)(pParam);
            if( !nn::nfp::ResultNeedRetry().Includes(result) )
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryInterval));
        }
        return result;
    }

    // タグの発見・喪失を通知するイベントを設定します。
    struct ArgAttachEvents
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        nn::os::SystemEventType*        pActivateEvent;
        nn::os::SystemEventType*        pDeactivateEvent;
    };

    nn::Result ProcessAttachEvents(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgAttachEvents*>(pParam);

        auto result = nn::nfp::AttachActivateEvent(pArg->pActivateEvent, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        result = nn::nfp::AttachDeactivateEvent(pArg->pDeactivateEvent, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        return nn::ResultSuccess();
    }

    // タグの検知を開始します。
    nn::Result ProcessStartDetection(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfp::StartDetection(*pArg->pDeviceHandle);
    }

    // タグの検知を終了します。
    nn::Result ProcessStopDetection(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfp::StopDetection(*pArg->pDeviceHandle);
    }

    // タグをマウントします。
    nn::Result ProcessMount(const void* pParam) NN_NOEXCEPT
    {
        // タグをマウントには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Mount(*pArg->pDeviceHandle, nn::nfp::ModelType_Amiibo);
            }
        );
    }

    // タグをアンマウントします。
    nn::Result ProcessUnmount(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfp::Unmount(*pArg->pDeviceHandle);
    }

    // タグのアプリケーション専用領域を作成します。
    struct ArgCreateApplicationArea
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        nn::Bit32                       accessId;
        void*                           pInitialData;
        uint32_t                        initialDataSize;
    };

    nn::Result ProcessCreateApplicationArea(const void* pParam) NN_NOEXCEPT
    {
        // タグの書き込みには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgCreateApplicationArea*>(pParam);
                nn::nfp::ApplicationAreaCreateInfo info;
                info.accessId = pArg->accessId;
                info.pInitialData = pArg->pInitialData;
                info.initialDataSize = pArg->initialDataSize;

                return nn::nfp::CreateApplicationArea(*pArg->pDeviceHandle, info);
            }
        );
    }

    // マウント時に用意した内部バッファのアプリケーション専用領域へのアクセスを有効化します。
    struct ArgOpenApplicationArea
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        nn::Bit32                       accessId;
    };

    nn::Result ProcessOpenApplicationArea(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgOpenApplicationArea*>(pParam);
        return nn::nfp::OpenApplicationArea(*pArg->pDeviceHandle, pArg->accessId);
    }

    // アプリケーション専用領域のデータを、マウント時に用意した内部バッファから取得します。
    struct ArgGetApplicationArea
    {
        void*                           pOutBuffer;
        size_t*                         pOutSize;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        uint32_t                        bufferSize;
    };

    nn::Result ProcessGetApplicationArea(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetApplicationArea*>(pParam);
        return nn::nfp::GetApplicationArea(pArg->pOutBuffer, pArg->pOutSize, *pArg->pDeviceHandle, pArg->bufferSize);
    }

    // アプリケーション専用領域のデータを、マウント時に用意した内部バッファに設定します。
    struct ArgSetApplicationArea
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        const void*                     pData;
        uint32_t                        dataSize;
    };

    nn::Result ProcessSetApplicationArea(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgSetApplicationArea*>(pParam);
        return nn::nfp::SetApplicationArea(*pArg->pDeviceHandle, pArg->pData, pArg->dataSize);
    }

    // マウント時に用意した内部バッファの内容をタグに書き込みます。
    nn::Result ProcessFlush(const void* pParam) NN_NOEXCEPT
    {
        // タグの書き込みには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Flush(*pArg->pDeviceHandle);
            }
        );
    }

    // データが破損したタグをフォーマットし、バックアップデータから復旧します。
    nn::Result ProcessRestore(const void* pParam) NN_NOEXCEPT
    {
        // タグを復旧には数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Restore(*pArg->pDeviceHandle);
            }
        );
    }

    // タグの情報を取得します。
    struct ArgGetTagInfo
    {
        nn::nfp::TagInfo*               pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetTagInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetTagInfo*>(pParam);
        return nn::nfp::GetTagInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグのオーナー登録情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetRegisterInfo
    {
        nn::nfp::RegisterInfo*          pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetRegisterInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetRegisterInfo*>(pParam);
        return nn::nfp::GetRegisterInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグの共用領域の情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetCommonInfo
    {
        nn::nfp::CommonInfo*            pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetCommonInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetCommonInfo*>(pParam);
        return nn::nfp::GetCommonInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグの ROM 領域の情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetModelInfo
    {
        nn::nfp::ModelInfo*             pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetModelInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetModelInfo*>(pParam);
        return nn::nfp::GetModelInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // ニックネームとオーナーの設定を行う amiibo 設定を開始します。
    struct ArgStartNicknameAndOwnerSettings
    {
        nn::nfp::DeviceHandle*          pOutDeviceHandle;
        bool*                           pOutIsRegistered;
        nn::nfp::RegisterInfo*          pOutRegisterInfo;
        const nn::nfp::AmiiboSettingsStartParam* pStartParam;
        const nn::nfp::TagInfo*         pTagInfo;
    };

    nn::Result ProcessStartNicknameAndOwnerSettings(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgStartNicknameAndOwnerSettings*>(pParam);

        NFPDEMO_LOG("Invoke StartNicknameAndOwnerSettings()\n");
        auto result = nn::nfp::StartNicknameAndOwnerSettings(pArg->pOutDeviceHandle, pArg->pOutIsRegistered, pArg->pOutRegisterInfo, *pArg->pStartParam, *pArg->pTagInfo);
        NFPDEMO_LOG("Returned from StartNicknameAndOwnerSettings()\n");

        return result;
    }

    // ゲームデータの消去を行う amiibo 設定を開始します。
    struct ArgStartGameDataEraser
    {
        nn::nfp::DeviceHandle*          pOutDeviceHandle;
        const nn::nfp::AmiiboSettingsStartParam* pStartParam;
        const nn::nfp::TagInfo*         pTagInfo;
    };

    nn::Result ProcessStartGameDataEraser(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgStartGameDataEraser*>(pParam);

        NFPDEMO_LOG("Invoke ProcessStartGameDataEraser()\n");
        auto result = nn::nfp::StartGameDataEraser(pArg->pOutDeviceHandle, *pArg->pStartParam, *pArg->pTagInfo);
        NFPDEMO_LOG("Returned from ProcessStartGameDataEraser()\n");

        return result;
    }

    // データの復旧を行う amiibo 設定を開始します。
    struct ArgStartRestorer
    {
        nn::nfp::DeviceHandle*          pOutDeviceHandle;
        const nn::nfp::AmiiboSettingsStartParam* pStartParam;
        const nn::nfp::TagInfo*         pTagInfo;
    };

    nn::Result ProcessStartRestorer(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgStartRestorer*>(pParam);

        NFPDEMO_LOG("Invoke StartRestorer()\n");
        auto result = nn::nfp::StartRestorer(pArg->pOutDeviceHandle, *pArg->pStartParam, *pArg->pTagInfo);
        NFPDEMO_LOG("Returned from StartRestorer()\n");

        return result;
    }

    // デバイスの状態を取得します。
    struct ArgGetDeviceState
    {
        nn::nfp::DeviceState*           pOutDeviceState;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetDeviceState(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetDeviceState*>(pParam);
        *pArg->pOutDeviceState = nn::nfp::GetDeviceState(*pArg->pDeviceHandle);
        return nn::ResultSuccess();
    }

    // コマンド識別子に対応した処理関数の定義
    struct ProcessEntry
    {
        PlayerCommand      command;
        ProcessFunc     func;
    };

    const ProcessEntry ProcessTable[] = {
        {   PlayerCommand_None,                           nullptr                         },
        {   PlayerCommand_AttachEvents,                   ProcessAttachEvents             },
        {   PlayerCommand_StartDetection,                 ProcessStartDetection           },
        {   PlayerCommand_StopDetection,                  ProcessStopDetection            },
        {   PlayerCommand_Mount,                          ProcessMount                    },
        {   PlayerCommand_Unmount,                        ProcessUnmount                  },
        {   PlayerCommand_CreateApplicationArea,          ProcessCreateApplicationArea    },
        {   PlayerCommand_OpenApplicationArea,            ProcessOpenApplicationArea      },
        {   PlayerCommand_GetApplicationArea,             ProcessGetApplicationArea       },
        {   PlayerCommand_SetApplicationArea,             ProcessSetApplicationArea       },
        {   PlayerCommand_Flush,                          ProcessFlush                    },
        {   PlayerCommand_Restore,                        ProcessRestore                  },
        {   PlayerCommand_GetTagInfo,                     ProcessGetTagInfo               },
        {   PlayerCommand_GetRegisterInfo,                ProcessGetRegisterInfo          },
        {   PlayerCommand_GetCommonInfo,                  ProcessGetCommonInfo            },
        {   PlayerCommand_GetModelInfo,                   ProcessGetModelInfo             },
        {   PlayerCommand_StartNicknameAndOwnerSettings,  ProcessStartNicknameAndOwnerSettings },
        {   PlayerCommand_StartGameDataEraser,            ProcessStartGameDataEraser      },
        {   PlayerCommand_StartRestorer,                  ProcessStartRestorer            },
        {   PlayerCommand_GetDeviceState,                 ProcessGetDeviceState           },
    };
    const int ProcessCountMax = sizeof(ProcessTable) / sizeof(ProcessTable[0]);
    NN_STATIC_ASSERT( ProcessCountMax == static_cast<int>(PlayerCommand_CountMax) );

    ProcessFunc GetProcessFunc(PlayerCommand command)
    {
        const int index = static_cast<int>(command);
        NN_ASSERT( index >= 1 && index < ProcessCountMax );
        NN_ASSERT( ProcessTable[index].command == command );
        return ProcessTable[index].func;
    }

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    int PlayerTagAccessor::AttachEvents(
            nn::os::SystemEventType* pActivateEvent,
            nn::os::SystemEventType* pDeactivateEvent,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgAttachEvents arg;
        arg.pActivateEvent = pActivateEvent;
        arg.pDeactivateEvent = pDeactivateEvent;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(PlayerCommand_AttachEvents, arg);
    }

    int PlayerTagAccessor::StartDetection(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(PlayerCommand_StartDetection, pDeviceHandle);
    }

    int PlayerTagAccessor::StopDetection(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(PlayerCommand_StopDetection, pDeviceHandle);
    }

    int PlayerTagAccessor::Mount(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(PlayerCommand_Mount, pDeviceHandle);
    }

    int PlayerTagAccessor::Unmount(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(PlayerCommand_Unmount, pDeviceHandle);
    }

    int PlayerTagAccessor::CreateApplicationArea(
            const nn::nfp::DeviceHandle* pDeviceHandle,
            nn::Bit32 accessId,
            void* pInitialData,
            uint32_t initialDataSize
        ) NN_NOEXCEPT
    {
        ArgCreateApplicationArea arg;
        arg.pDeviceHandle = pDeviceHandle;
        arg.accessId = accessId;
        arg.pInitialData = pInitialData;
        arg.initialDataSize = initialDataSize;

        return Enqueue(PlayerCommand_CreateApplicationArea, arg);
    }

    int PlayerTagAccessor::OpenApplicationArea(
            const nn::nfp::DeviceHandle* pDeviceHandle,
            nn::Bit32 accessId
        ) NN_NOEXCEPT
    {
        ArgOpenApplicationArea arg;
        arg.pDeviceHandle = pDeviceHandle;
        arg.accessId = accessId;

        return Enqueue(PlayerCommand_OpenApplicationArea, arg);
    }

    int PlayerTagAccessor::GetApplicationArea(
            void* pOutBuffer,
            size_t* pOutSize,
            const nn::nfp::DeviceHandle* pDeviceHandle,
            uint32_t bufferSize
        ) NN_NOEXCEPT
    {
        ArgGetApplicationArea arg;
        arg.pOutBuffer = pOutBuffer;
        arg.pOutSize = pOutSize;
        arg.pDeviceHandle = pDeviceHandle;
        arg.bufferSize = bufferSize;

        return Enqueue(PlayerCommand_GetApplicationArea, arg);
    }

    int PlayerTagAccessor::SetApplicationArea(
            const nn::nfp::DeviceHandle* pDeviceHandle,
            const void* pData,
            uint32_t dataSize
        ) NN_NOEXCEPT
    {
        ArgSetApplicationArea arg;
        arg.pDeviceHandle = pDeviceHandle;
        arg.pData = pData;
        arg.dataSize = dataSize;

        return Enqueue(PlayerCommand_SetApplicationArea, arg);
    }

    int PlayerTagAccessor::Flush(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(PlayerCommand_Flush, pDeviceHandle);
    }

    int PlayerTagAccessor::Restore(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(PlayerCommand_Restore, pDeviceHandle);
    }

    int PlayerTagAccessor::GetTagInfo(
            nn::nfp::TagInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetTagInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(PlayerCommand_GetTagInfo, arg);
    }

    int PlayerTagAccessor::GetRegisterInfo(
            nn::nfp::RegisterInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetRegisterInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(PlayerCommand_GetRegisterInfo, arg);
    }

    int PlayerTagAccessor::GetCommonInfo(
            nn::nfp::CommonInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetCommonInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(PlayerCommand_GetCommonInfo, arg);
    }

    int PlayerTagAccessor::GetModelInfo(
            nn::nfp::ModelInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetModelInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(PlayerCommand_GetModelInfo, arg);
    }

    int PlayerTagAccessor::StartNicknameAndOwnerSettings(nn::nfp::DeviceHandle* pOutDeviceHandle, bool* pOutIsRegistered, nn::nfp::RegisterInfo* pOutRegisterInfo, const nn::nfp::AmiiboSettingsStartParam* pStartParam, const nn::nfp::TagInfo* pTagInfo) NN_NOEXCEPT
    {
        ArgStartNicknameAndOwnerSettings arg;
        arg.pOutDeviceHandle = pOutDeviceHandle;
        arg.pOutIsRegistered = pOutIsRegistered;
        arg.pOutRegisterInfo = pOutRegisterInfo;
        arg.pStartParam = pStartParam;
        arg.pTagInfo = pTagInfo;

        return Enqueue(PlayerCommand_StartNicknameAndOwnerSettings, arg);
    }

    int PlayerTagAccessor::StartGameDataEraser(nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam* pStartParam, const nn::nfp::TagInfo* pTagInfo) NN_NOEXCEPT
    {
        ArgStartGameDataEraser arg;
        arg.pOutDeviceHandle = pOutDeviceHandle;
        arg.pStartParam = pStartParam;
        arg.pTagInfo = pTagInfo;

        return Enqueue(PlayerCommand_StartGameDataEraser, arg);
    }

    int PlayerTagAccessor::StartRestorer(nn::nfp::DeviceHandle* pOutDeviceHandle, const nn::nfp::AmiiboSettingsStartParam* pStartParam, const nn::nfp::TagInfo* pTagInfo) NN_NOEXCEPT
    {
        ArgStartRestorer arg;
        arg.pOutDeviceHandle = pOutDeviceHandle;
        arg.pStartParam = pStartParam;
        arg.pTagInfo = pTagInfo;

        return Enqueue(PlayerCommand_StartRestorer, arg);
    }

    int PlayerTagAccessor::GetDeviceState(
            nn::nfp::DeviceState* pOutDeviceState,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetDeviceState arg;
        arg.pOutDeviceState = pOutDeviceState;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(PlayerCommand_GetDeviceState, arg);
    }

    bool PlayerTagAccessor::IsBusy() NN_NOEXCEPT
    {
        return TagAccessorBase::IsBusy() && m_IsWaitForAmiiboSetting == false;
    }

    nn::Result PlayerTagAccessor::Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT
    {
        if(command == PlayerCommand_StartNicknameAndOwnerSettings
           || command == PlayerCommand_StartGameDataEraser
           || command == PlayerCommand_StartRestorer)
        {
            m_IsWaitForAmiiboSetting = true;
        }
        auto result = (*GetProcessFunc(static_cast<PlayerCommand>(command)))(&param);
        m_IsWaitForAmiiboSetting = false;
        return result;
    }

}} // end of namespace nns::nfp
