﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <map>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include "HidController.h"

//#define NFPDEMO_ENABLE_HIDLOG
#if defined( NFPDEMO_ENABLE_HIDLOG )
#define NFPDEMO_HIDLOG(...)     NN_LOG(__VA_ARGS__)
#else
#define NFPDEMO_HIDLOG(...)
#endif // defined( NFPDEMO_ENABLE_HIDLOG )

//------------------------------------------------------------------------------
//  ヒューマンインターフェイスデバイス機能の初期化・終了
//------------------------------------------------------------------------------
namespace {

    //--------------------------------------------------------------------------
    //  定数、型定義
    //--------------------------------------------------------------------------
    const int SupportedNpadStyleIndexes[] =
    {
        nn::hid::NpadStyleHandheld::Flag::Index,
        nn::hid::NpadStyleJoyDual::Flag::Index,
        nn::hid::NpadStyleFullKey::Flag::Index,
    };
    const int SupportedNpadStyleCountMax = sizeof(SupportedNpadStyleIndexes) / sizeof(int);

    struct NpadState
    {
        nn::hid::NpadStyleSet style;
        union
        {
            nn::hid::NpadHandheldState  handheld;
            nn::hid::NpadJoyDualState   dual;
            nn::hid::NpadFullKeyState   full;
        };
    };

    //--------------------------------------------------------------------------
    //  共通関数
    //--------------------------------------------------------------------------
    nn::hid::NpadStyleSet GetSupportedNpadStyleSet() NN_NOEXCEPT
    {
        nn::hid::NpadStyleSet style;
        style.Reset();
        for( int i = 0; i < SupportedNpadStyleCountMax; i++ )
        {
            style.Set(SupportedNpadStyleIndexes[i]);
        }
        return style;
    }

    //
    int ToNpadStyleIndex(nn::hid::NpadStyleSet style) NN_NOEXCEPT
    {
        for( int i = 0; i < SupportedNpadStyleCountMax; i++ )
        {
            const int index = SupportedNpadStyleIndexes[i];
            if( style.Test(index) == true )
            {
                return index;
            }
        }
        return -1;
    }

    //--------------------------------------------------------------------------
    //  ログ出力関数
    //--------------------------------------------------------------------------
    void PrintNpadButtonState(const nn::hid::NpadButtonSet& state)
    {
        char buttons[37];
        buttons[0]   = (state.Test<nn::hid::NpadJoyButton::A>())         ? 'A' : '_';
        buttons[1]   = (state.Test<nn::hid::NpadJoyButton::B>())         ? 'B' : '_';
        buttons[2]   = (state.Test<nn::hid::NpadJoyButton::X>())         ? 'X' : '_';
        buttons[3]   = (state.Test<nn::hid::NpadJoyButton::Y>())         ? 'Y' : '_';
        buttons[4]   = (state.Test<nn::hid::NpadJoyButton::StickL>())    ? 'L' : '_';
        buttons[5]   = (state.Test<nn::hid::NpadJoyButton::StickL>())    ? 'S' : '_';
        buttons[6]   = (state.Test<nn::hid::NpadJoyButton::StickR>())    ? 'R' : '_';
        buttons[7]   = (state.Test<nn::hid::NpadJoyButton::StickR>())    ? 'S' : '_';
        buttons[8]   = (state.Test<nn::hid::NpadJoyButton::L>())         ? 'L' : '_';
        buttons[9]   = (state.Test<nn::hid::NpadJoyButton::R>())         ? 'R' : '_';
        buttons[10]  = (state.Test<nn::hid::NpadJoyButton::ZL>())        ? 'Z' : '_';
        buttons[11]  = (state.Test<nn::hid::NpadJoyButton::ZL>())        ? 'L' : '_';
        buttons[12]  = (state.Test<nn::hid::NpadJoyButton::ZR>())        ? 'Z' : '_';
        buttons[13]  = (state.Test<nn::hid::NpadJoyButton::ZR>())        ? 'R' : '_';
        buttons[14]  = (state.Test<nn::hid::NpadJoyButton::Plus>())      ? '+' : '_';
        buttons[15]  = (state.Test<nn::hid::NpadJoyButton::Minus>())     ? '-' : '_';
        buttons[16]  = (state.Test<nn::hid::NpadJoyButton::Left>())      ? '<' : '_';
        buttons[17]  = (state.Test<nn::hid::NpadJoyButton::Up>())        ? '^' : '_';
        buttons[18]  = (state.Test<nn::hid::NpadJoyButton::Right>())     ? '>' : '_';
        buttons[19]  = (state.Test<nn::hid::NpadJoyButton::Down>())      ? 'v' : '_';
        buttons[20]  = (state.Test<nn::hid::NpadJoyButton::LeftSL>())    ? 'S' : '_';
        buttons[21]  = (state.Test<nn::hid::NpadJoyButton::LeftSL>())    ? 'L' : '_';
        buttons[22]  = (state.Test<nn::hid::NpadJoyButton::LeftSR>())    ? 'S' : '_';
        buttons[23]  = (state.Test<nn::hid::NpadJoyButton::LeftSR>())    ? 'R' : '_';
        buttons[24]  = (state.Test<nn::hid::NpadJoyButton::RightSL>())   ? 'S' : '_';
        buttons[25]  = (state.Test<nn::hid::NpadJoyButton::RightSL>())   ? 'L' : '_';
        buttons[26]  = (state.Test<nn::hid::NpadJoyButton::RightSR>())   ? 'S' : '_';
        buttons[27]  = (state.Test<nn::hid::NpadJoyButton::RightSR>())   ? 'R' : '_';
        buttons[28]  = (state.Test<nn::hid::NpadJoyButton::StickLRight>()) ? '>' : '_';
        buttons[29]  = (state.Test<nn::hid::NpadJoyButton::StickLUp>())    ? '^' : '_';
        buttons[30]  = (state.Test<nn::hid::NpadJoyButton::StickLLeft>())  ? '<' : '_';
        buttons[31]  = (state.Test<nn::hid::NpadJoyButton::StickLDown>())  ? 'v' : '_';
        buttons[32]  = (state.Test<nn::hid::NpadJoyButton::StickRRight>()) ? '>' : '_';
        buttons[33]  = (state.Test<nn::hid::NpadJoyButton::StickRUp>())    ? '^' : '_';
        buttons[34]  = (state.Test<nn::hid::NpadJoyButton::StickRLeft>())  ? '<' : '_';
        buttons[35]  = (state.Test<nn::hid::NpadJoyButton::StickRDown>())  ? 'v' : '_';
        buttons[36]  = '\0';

        NFPDEMO_HIDLOG("%s", buttons);
    }

    void PrintNpadStickState(const nn::hid::AnalogStickState& state)
    {
        NN_UNUSED(state);
        NFPDEMO_HIDLOG("(%6d, %6d)", state.x,
                             state.y);
    }

    template <typename TState>
    void PrintNpadState(const TState& state)
    {
        NFPDEMO_HIDLOG("[%6lld] ", state.samplingNumber);
        PrintNpadButtonState(state.buttons);
        NFPDEMO_HIDLOG(" L");
        PrintNpadStickState(state.analogStickL);
        NFPDEMO_HIDLOG(" R");
        PrintNpadStickState(state.analogStickR);
        NFPDEMO_HIDLOG("\n");
    }

    //--------------------------------------------------------------------------
    //  コントローラーの抽象クラス
    //--------------------------------------------------------------------------
    class Controller
    {
        NN_DISALLOW_COPY( Controller );
        NN_DISALLOW_MOVE( Controller );

    public:
        Controller() NN_NOEXCEPT
        {
            m_ButtonsDown.Reset();
            m_ButtonsUp.Reset();
            m_ButtonsOn.Reset();
        }

        virtual ~Controller() NN_NOEXCEPT {}

        // コントローラーを初期化します。
        virtual void Initialize() NN_NOEXCEPT = 0;

        // コントローラーの入力状態を更新します。
        virtual void UpdateState() NN_NOEXCEPT = 0;

        // コントローラーの入力状態をログ出力します。
        virtual void PrintState() NN_NOEXCEPT = 0;

        // コントローラのプレイヤーランプの点灯パターンを返します。
        virtual nn::Bit8 GetPlayerLedPattern() NN_NOEXCEPT = 0;

        // デジタルボタンの何れかが最後の更新で新たに押下・開放されたか否かを返します。
        bool IsAnyButtonsUpdated() const NN_NOEXCEPT
        {
            return m_ButtonsDown.IsAnyOn() || m_ButtonsUp.IsAnyOn();
        }

        // 指定されたデジタルボタンの何れかが最後の更新で新たに押下されたか否かを返します。
        bool HasAnyButtonsDown(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT
        {
            return (m_ButtonsDown & buttons).IsAnyOn();
        }

        // 指定されたデジタルボタンの何れかが最後の更新で新たに開放されたか否かを返します。
        bool HasAnyButtonsUp(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT
        {
            return (m_ButtonsUp & buttons).IsAnyOn();
        }

        // 指定されたデジタルボタンの何れかが押下中か否かを返します。
        bool HasAnyButtonsOn(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT
        {
            return (m_ButtonsOn & buttons).IsAnyOn();
        }

        // 指定されたデジタルボタンの全てが押下中か否かを返します。
        bool HasAllButtonsOn(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT
        {
            return ((m_ButtonsOn & buttons) == buttons);
        }

    protected:
        // デジタルボタンを最新の値に更新します。
        void UpdateButtons(nn::hid::NpadButtonSet buttons)
        {
            // 新たに押下されたデジタルボタンを最新の値に更新します。
            m_ButtonsDown = ~m_ButtonsOn & buttons;

            // 新たに開放されたデジタルボタンを最新の値に更新します。
            m_ButtonsUp = m_ButtonsOn & ~buttons;

            // 押下中のデジタルボタンを最新の値に更新します。
            m_ButtonsOn = buttons;
        }

        nn::hid::NpadButtonSet m_ButtonsDown;
        nn::hid::NpadButtonSet m_ButtonsUp;
        nn::hid::NpadButtonSet m_ButtonsOn;
    };

    //--------------------------------------------------------------------------
    //  Npad コントローラークラス
    //--------------------------------------------------------------------------
    class NpadController : public Controller
    {
        NN_DISALLOW_COPY( NpadController );
        NN_DISALLOW_MOVE( NpadController );

    public:
        explicit NpadController(nn::hid::NpadIdType id) NN_NOEXCEPT
        :   m_Id(id), m_PlayerLedPattern(nn::hid::GetPlayerLedPattern(id))
        {
            std::memset(&m_State, 0, sizeof(m_State));
        }

        virtual ~NpadController() NN_NOEXCEPT NN_OVERRIDE {}

        // コントローラーを初期化します。
        void Initialize() NN_NOEXCEPT NN_OVERRIDE
        {
            // Npad の初期化は何もしない
        }

        // コントローラーの入力状態を更新します。
        void UpdateState() NN_NOEXCEPT NN_OVERRIDE
        {
            m_State.style = nn::hid::GetNpadStyleSet(m_Id);

            nn::hid::NpadButtonSet buttons;
            switch( ToNpadStyleIndex(m_State.style) )
            {
            case nn::hid::NpadStyleHandheld::Flag::Index:
                {
                    nn::hid::GetNpadState(&m_State.handheld, m_Id);
                    buttons = m_State.handheld.buttons;
                }
                break;
            case nn::hid::NpadStyleJoyDual::Flag::Index:
                {
                    nn::hid::GetNpadState(&m_State.dual, m_Id);
                    buttons = m_State.dual.buttons;
                }
                break;
            case nn::hid::NpadStyleFullKey::Flag::Index:
                {
                    nn::hid::GetNpadState(&m_State.full, m_Id);
                    buttons = m_State.full.buttons;
                }
                break;
            default:
                {
                    buttons.Reset();
                }
                break;
            }
            UpdateButtons(buttons);
        }

        // コントローラーの入力状態をログ出力します。
        void PrintState() NN_NOEXCEPT NN_OVERRIDE
        {
            switch( ToNpadStyleIndex(m_State.style) )
            {
            case nn::hid::NpadStyleHandheld::Flag::Index:
                {
                    NFPDEMO_HIDLOG("NpadHandheld        ");
                    PrintNpadState(m_State.handheld);
                }
                break;
            case nn::hid::NpadStyleJoyDual::Flag::Index:
                {
                    NFPDEMO_HIDLOG("NpadJoyDual (%d)     ", m_Id);
                    PrintNpadState(m_State.dual);
                }
                break;
            case nn::hid::NpadStyleFullKey::Flag::Index:
                {
                    NFPDEMO_HIDLOG("NpadFullKey (%d)     ", m_Id);
                    PrintNpadState(m_State.full);
                }
                break;
            default:
                break;
            }
        }

        // コントローラのプレイヤーランプの点灯パターンを返します。
        nn::Bit8 GetPlayerLedPattern() NN_NOEXCEPT NN_OVERRIDE
        {
            return m_PlayerLedPattern;
        }

    private:
        const nn::hid::NpadIdType m_Id;
        NpadState m_State;
        nn::Bit8 m_PlayerLedPattern;
    };

    //--------------------------------------------------------------------------
    //  コントローラ管理クラス
    //--------------------------------------------------------------------------
    class ControllerManager
    {
        NN_DISALLOW_COPY( ControllerManager );
        NN_DISALLOW_MOVE( ControllerManager );

    public:
        ControllerManager() NN_NOEXCEPT : m_IsInitialized(false) {}
        ~ControllerManager() NN_NOEXCEPT { Finalize(); }

        // 初期化されたか否かを返します。
        bool IsInitialized() const NN_NOEXCEPT { return m_IsInitialized; }

        // コントローラーを初期化します。
        void Initialize() NN_NOEXCEPT
        {
            NN_ASSERT( m_IsInitialized == false );

            nn::hid::InitializeNpad();

            auto style = GetSupportedNpadStyleSet();
            nn::hid::SetSupportedNpadStyleSet(style);
            nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

            for( int i = 0; i < NpadIdCountMax; i++ )
            {
                const auto id = NpadIds[i];
                NN_ASSERT( m_Controllers.find(id) == m_Controllers.end() );
                (m_Controllers[id] = new NpadController(id))->Initialize();
            }
            m_IsInitialized = true;
        }

        // コントローラーを終了します。
        void Finalize() NN_NOEXCEPT
        {
            if( m_IsInitialized )
            {
                m_IsInitialized = false;
                for( auto pair : m_Controllers )
                {
                    delete (pair.second);
                }
                m_Controllers.clear();
            }
        }

        // コントローラーの入力状態を更新します。
        void UpdateState() NN_NOEXCEPT
        {
            for( auto pair : m_Controllers )
            {
                (pair.second)->UpdateState();
            }
        }

        // コントローラーの入力状態をログ出力します。
        void PrintState() NN_NOEXCEPT
        {
            for( auto pair : m_Controllers )
            {
                if( (pair.second)->IsAnyButtonsUpdated() )
                {
                    (pair.second)->PrintState();
                }
            }
        }

        // 指定されたコントローラーを返します。
        Controller* GetController(nn::hid::NpadIdType id) const NN_NOEXCEPT
        {
            auto it = m_Controllers.find(id);
            return it != m_Controllers.end() ? it->second : nullptr;
        }

        // 指定されたデジタルボタンの何れかが最後の更新で新たに押下されたか否かを返します。
        bool HasAnyButtonsDown(nn::hid::NpadIdType id, nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT
        {
            auto controller = this->GetController(id);
            if(controller)
            {
                return controller->HasAnyButtonsDown(buttons);
            }
            return false;
        }

        // 接続されている Npad の ID リストを返します。
        int ListConnectedNpadIds(nn::hid::NpadIdType* pOutBuffer, int bufferCount) NN_NOEXCEPT
        {
            int count = 0;
            for( auto pair : m_Controllers )
            {
                if(bufferCount <= count)
                {
                    return count;
                }

                nn::hid::NpadIdType id = pair.first;
                if(!nn::hid::GetNpadStyleSet(id).IsAllOff())
                {
                    pOutBuffer[count] = id;
                    count++;
                }
            }

            return count;
        }

        // 指定されたコントローラのプレイヤーランプの点灯パターンを返します。
        nn::Bit8 GetPlayerLedPattern(nn::hid::NpadIdType id) NN_NOEXCEPT
        {
            auto controller = this->GetController(id);
            if(controller)
            {
                return controller->GetPlayerLedPattern();
            }

            return 0;
        }

    private:
        bool m_IsInitialized;
        std::map<nn::hid::NpadIdType, Controller*> m_Controllers;
    };

    //--------------------------------------------------------------------------
    //  グローバル変数
    //--------------------------------------------------------------------------
    ControllerManager* g_pControllerManager;

} // namespace

bool IsHidControllerInitialized() NN_NOEXCEPT
{
    return (g_pControllerManager != nullptr) ? g_pControllerManager->IsInitialized() : false;
}

void InitializeHidController() NN_NOEXCEPT
{
    NN_ASSERT( IsHidControllerInitialized() == false );

    g_pControllerManager = new ControllerManager();
    g_pControllerManager->Initialize();
}

void UpdateHidController() NN_NOEXCEPT
{
    NN_ASSERT( IsHidControllerInitialized() != false );

    g_pControllerManager->UpdateState();
    #if defined( NFPDEMO_ENABLE_HIDLOG )
    g_pControllerManager->PrintState();
    #endif // defined( NFPDEMO_ENABLE_HIDLOG )
}

void FinalizeHidController() NN_NOEXCEPT
{
    NN_ASSERT( IsHidControllerInitialized() != false );

    delete g_pControllerManager;
    g_pControllerManager = nullptr;
}

//------------------------------------------------------------------------------
//  ヒューマンインターフェイスデバイスのデジタルボタンの状態確認
//------------------------------------------------------------------------------

// 指定されたデジタルボタンが最後の更新で新たに押下されたか否かを返します。
bool HasHidControllerAnyButtonsDown(nn::hid::NpadIdType id, nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
{
    NN_ASSERT( IsHidControllerInitialized() != false );

    return g_pControllerManager->HasAnyButtonsDown(id, buttons);
}

// 接続されている Npad の ID リストを返します。
int ListHidControllerConnectedNpadIds(nn::hid::NpadIdType* pOutBuffer, int bufferCount) NN_NOEXCEPT
{
    NN_ASSERT( IsHidControllerInitialized() != false );

    return g_pControllerManager->ListConnectedNpadIds(pOutBuffer, bufferCount);
}

// 指定されたコントローラのプレイヤーランプの点灯パターンを返します。
nn::Bit8 GetHidControllerPlayerLedPattern(nn::hid::NpadIdType id) NN_NOEXCEPT
{
    NN_ASSERT( IsHidControllerInitialized() != false );

    return g_pControllerManager->GetPlayerLedPattern(id);
}
