﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include <nn/nfp/nfp_PrivateApi.h>
#include "TagAccessorSystem.h"

// #define NFPDEMO_PERMIT_HIGH_SPEED_TRANSITION 1

namespace nns { namespace nfp { namespace {

    NN_ALIGNAS(16) char g_WorkBuffer[ sizeof(TagAccessor) ];

    typedef nn::Result (*ProcessFunc)( const void* pParam );

    struct ArgDeviceHandle
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    // NFP ライブラリ API が ResultNeedRetry() を返す場合、成功するまで RetryCountMax 回再試行します。
    const int RetryCountMax = 5;
    const int RetryInterval = 200;

    nn::Result ProcessRetryFunction(const void* pParam, ProcessFunc pFunc) NN_NOEXCEPT
    {
        nn::Result result;
        for( int i = 0; i < RetryCountMax; ++i )
        {
            result = (*pFunc)(pParam);
            if( !nn::nfp::ResultNeedRetry().Includes(result) )
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryInterval));
        }
        return result;
    }

    // NFP ライブラリをシステムモードで初期化します。
    nn::Result ProcessInitializeSystem(const void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);
        nn::nfp::InitializeSystem();
        return nn::ResultSuccess();
    }

    // NFP ライブラリのシステムモードの使用を終了します。
    nn::Result ProcessFinalizeSystem(const void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);
        nn::nfp::FinalizeSystem();
        return nn::ResultSuccess();
    }

    // NFC デバイスのハンドルのリストを取得します。
    struct ArgListDevices
    {
        nn::nfp::DeviceHandle*          pOutBuffer;
        int*                            pOutCount;
        int                             bufferCount;
    };

    nn::Result ProcessListDevices(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgListDevices*>(pParam);
        return nn::nfp::ListDevices(pArg->pOutBuffer, pArg->pOutCount, pArg->bufferCount);
    }

    // タグの発見・喪失を通知するイベントを設定し、NFC デバイスの Npad ID を取得します。
    struct ArgAttachEvents
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        nn::os::SystemEventType*        pActivateEvent;
        nn::os::SystemEventType*        pDeactivateEvent;
        nn::hid::NpadIdType*            pOutNpadId;
    };

    nn::Result ProcessAttachEvents(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgAttachEvents*>(pParam);

        nn::Result result = nn::nfp::GetNpadId(pArg->pOutNpadId, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        result = nn::nfp::AttachActivateEvent(pArg->pActivateEvent, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        result = nn::nfp::AttachDeactivateEvent(pArg->pDeactivateEvent, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        return nn::ResultSuccess();
    }

    // タグの検知を開始します。
    nn::Result ProcessStartDetection(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfp::StartDetection(*pArg->pDeviceHandle);
    }

    // タグの検知を終了します。
    nn::Result ProcessStopDetection(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfp::StopDetection(*pArg->pDeviceHandle);
    }

    // タグをマウントします。
    nn::Result ProcessMount(const void* pParam) NN_NOEXCEPT
    {
        // タグをマウントには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Mount(*pArg->pDeviceHandle, nn::nfp::ModelType_Amiibo);
            }
        );
    }

    // タグをアンマウントします。
    nn::Result ProcessUnmount(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfp::Unmount(*pArg->pDeviceHandle);
    }

    // タグをフォーマットします。
    nn::Result ProcessFormat(const void* pParam) NN_NOEXCEPT
    {
        // タグをフォーマットには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Format(*pArg->pDeviceHandle);
                }
            );
    }

    // マウント時に用意した内部バッファの内容をタグに書き込みます。
    nn::Result ProcessFlush(const void* pParam) NN_NOEXCEPT
    {
        // タグの書き込みには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Flush(*pArg->pDeviceHandle);
            }
        );
    }

    // データが破損したタグをフォーマットし、バックアップデータから復旧します。
    nn::Result ProcessRestore(const void* pParam) NN_NOEXCEPT
    {
        // タグを復旧には数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::Restore(*pArg->pDeviceHandle);
            }
        );
    }

    // タグの情報を取得します。
    struct ArgGetTagInfo
    {
        nn::nfp::TagInfo*               pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetTagInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetTagInfo*>(pParam);
        return nn::nfp::GetTagInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグの共用領域の情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetCommonInfo
    {
        nn::nfp::CommonInfo*            pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetCommonInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetCommonInfo*>(pParam);
        return nn::nfp::GetCommonInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグの ROM 領域の情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetModelInfo
    {
        nn::nfp::ModelInfo*             pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetModelInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetModelInfo*>(pParam);
        return nn::nfp::GetModelInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグの管理情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetAdminInfo
    {
        nn::nfp::AdminInfo*             pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetAdminInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetAdminInfo*>(pParam);
        return nn::nfp::GetAdminInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグのオーナー登録情報を、マウント時に用意した内部バッファから取得します。
    struct ArgGetRegisterInfo
    {
        nn::nfp::RegisterInfo*          pOutInfo;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetRegisterInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetRegisterInfo*>(pParam);
        return nn::nfp::GetRegisterInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // タグのシステム領域の登録情報を、マウント時に用意した内部バッファに設定します。
    struct ArgSetRegisterInfo
    {
        const nn::nfp::DeviceHandle*    pDeviceHandle;
        const nn::nfp::RegisterInfoPrivate* pInfoPrivate;
    };

    nn::Result ProcessSetRegisterInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgSetRegisterInfo*>(pParam);
        return nn::nfp::SetRegisterInfo(*pArg->pDeviceHandle, *pArg->pInfoPrivate);
    }

    // タグの登録情報を削除します。
    nn::Result ProcessDeleteRegisterInfo(const void* pParam) NN_NOEXCEPT
    {
        // タグの書き込みには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::DeleteRegisterInfo(*pArg->pDeviceHandle);
            }
        );
    }

    // タグのアプリケーション専用領域の情報を削除します。
    nn::Result ProcessDeleteApplicationArea(const void* pParam) NN_NOEXCEPT
    {
        // タグの書き込みには数秒程度かかります。
        // ResultNeedRetry() のエラーが発生した場合には再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
                return nn::nfp::DeleteApplicationArea(*pArg->pDeviceHandle);
            }
        );
    }

    // アプリケーション専用領域が存在するか判定します。
    struct ArgExistsApplicationArea
    {
        bool*                           pOutValue;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessExistsApplicationArea(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgExistsApplicationArea*>(pParam);
        return nn::nfp::ExistsApplicationArea(pArg->pOutValue, *pArg->pDeviceHandle);
    }

    // デバイスの状態を取得します。
    struct ArgGetDeviceState
    {
        nn::nfp::DeviceState*           pOutDeviceState;
        const nn::nfp::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetDeviceState(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetDeviceState*>(pParam);
        *pArg->pOutDeviceState = nn::nfp::GetDeviceState(*pArg->pDeviceHandle);
        return nn::ResultSuccess();
    }

    // デバイスの利用可/不可の変化を通知するイベントを設定します。
    struct ArgAttachAvailabilityChangeEvent
    {
        nn::os::SystemEventType*        pAvailabilityChangeEvent;
    };

    nn::Result ProcessAttachAvailabilityChangeEvent(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgAttachAvailabilityChangeEvent*>(pParam);

        nn::nfp::AttachAvailabilityChangeEvent(pArg->pAvailabilityChangeEvent);
        return nn::ResultSuccess();
    }

    // コマンド識別子に対応した処理関数の定義
    struct ProcessEntry
    {
        Command         command;
        ProcessFunc     func;
    };

    const ProcessEntry ProcessTable[] = {
        {   Command_None,                       nullptr                         },
        {   Command_InitializeSystem,           ProcessInitializeSystem         },
        {   Command_FinalizeSystem,             ProcessFinalizeSystem           },
        {   Command_ListDevices,                ProcessListDevices              },
        {   Command_AttachEvents,               ProcessAttachEvents             },
        {   Command_StartDetection,             ProcessStartDetection           },
        {   Command_StopDetection,              ProcessStopDetection            },
        {   Command_Mount,                      ProcessMount                    },
        {   Command_Unmount,                    ProcessUnmount                  },
        {   Command_Format,                     ProcessFormat                   },
        {   Command_Flush,                      ProcessFlush                    },
        {   Command_Restore,                    ProcessRestore                  },
        {   Command_GetTagInfo,                 ProcessGetTagInfo               },
        {   Command_GetCommonInfo,              ProcessGetCommonInfo            },
        {   Command_GetModelInfo,               ProcessGetModelInfo             },
        {   Command_GetAdminInfo,               ProcessGetAdminInfo             },
        {   Command_GetRegisterInfo,            ProcessGetRegisterInfo          },
        {   Command_SetRegisterInfo,            ProcessSetRegisterInfo          },
        {   Command_DeleteRegisterInfo,         ProcessDeleteRegisterInfo       },
        {   Command_DeleteApplicationArea,      ProcessDeleteApplicationArea    },
        {   Command_ExistsApplicationArea,      ProcessExistsApplicationArea    },
        {   Command_GetDeviceState,             ProcessGetDeviceState           },
        {   Command_AttachAvailabilityChangeEvent, ProcessAttachAvailabilityChangeEvent },
    };
    const int ProcessCountMax = sizeof(ProcessTable) / sizeof(ProcessTable[0]);
    NN_STATIC_ASSERT( ProcessCountMax == static_cast<int>(Command_CountMax) );

    ProcessFunc GetProcessFunc(Command command)
    {
        const int index = static_cast<int>(command);
        NN_ASSERT( index >= 1 && index < ProcessCountMax );
        NN_ASSERT( ProcessTable[index].command == command );
        return ProcessTable[index].func;
    }

}}} // end of namespace nns::nfp::unnamed

namespace nns { namespace nfp {

    TagAccessor* TagAccessor::g_pInstance;

    TagAccessor& TagAccessor::CreateInstance() NN_NOEXCEPT
    {
        NN_ASSERT( g_pInstance == nullptr );

        g_pInstance = new(g_WorkBuffer) TagAccessor();
        return *g_pInstance;
    }

    TagAccessor& TagAccessor::GetInstance() NN_NOEXCEPT
    {
        NN_ASSERT( g_pInstance != nullptr );

        return *g_pInstance;
    }

    void TagAccessor::DestroyInstance() NN_NOEXCEPT
    {
        if( g_pInstance != nullptr )
        {
            g_pInstance->~TagAccessor();
            g_pInstance = nullptr;
        }
    }

    bool TagAccessor::HasInstance() NN_NOEXCEPT
    {
        return g_pInstance != nullptr;
    }

    int TagAccessor::InitializeSystem() NN_NOEXCEPT
    {
        return Enqueue(Command_InitializeSystem, nullptr, 0);
    }

    int TagAccessor::FinalizeSystem() NN_NOEXCEPT
    {
        return Enqueue(Command_FinalizeSystem, nullptr, 0);
    }

    int TagAccessor::ListDevices(
            nn::nfp::DeviceHandle* pOutBuffer,
            int* pOutCount,
            int bufferCount
        ) NN_NOEXCEPT
    {
        ArgListDevices arg;
        arg.pOutBuffer = pOutBuffer;
        arg.pOutCount = pOutCount;
        arg.bufferCount = bufferCount;

        return Enqueue(Command_ListDevices, arg);
    }

    int TagAccessor::AttachEvents(
            nn::hid::NpadIdType* pOutNpadId,
            nn::os::SystemEventType* pActivateEvent,
            nn::os::SystemEventType* pDeactivateEvent,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgAttachEvents arg;
        arg.pOutNpadId = pOutNpadId;
        arg.pActivateEvent = pActivateEvent;
        arg.pDeactivateEvent = pDeactivateEvent;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_AttachEvents, arg);
    }

    int TagAccessor::StartDetection(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_StartDetection, pDeviceHandle);
    }

    int TagAccessor::StopDetection(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_StopDetection, pDeviceHandle);
    }

    int TagAccessor::Mount(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_Mount, pDeviceHandle);
    }

    int TagAccessor::Unmount(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_Unmount, pDeviceHandle);
    }

    int TagAccessor::Format(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_Format, pDeviceHandle);
    }

    int TagAccessor::Flush(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_Flush, pDeviceHandle);
    }

    int TagAccessor::Restore(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_Restore, pDeviceHandle);
    }

    int TagAccessor::GetTagInfo(
            nn::nfp::TagInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetTagInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetTagInfo, arg);
    }

    int TagAccessor::GetCommonInfo(
            nn::nfp::CommonInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetCommonInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetCommonInfo, arg);
    }

    int TagAccessor::GetModelInfo(
            nn::nfp::ModelInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetModelInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetModelInfo, arg);
    }

    int TagAccessor::GetAdminInfo(
            nn::nfp::AdminInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetAdminInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetAdminInfo, arg);
    }

    int TagAccessor::GetRegisterInfo(
            nn::nfp::RegisterInfo* pOutInfo,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetRegisterInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetRegisterInfo, arg);
    }

    int TagAccessor::SetRegisterInfo(
            const nn::nfp::DeviceHandle* pDeviceHandle,
            const nn::nfp::RegisterInfoPrivate* pInfoPrivate
        ) NN_NOEXCEPT
    {
        ArgSetRegisterInfo arg;
        arg.pDeviceHandle = pDeviceHandle;
        arg.pInfoPrivate = pInfoPrivate;

        return Enqueue(Command_SetRegisterInfo, arg);
    }

    int TagAccessor::DeleteRegisterInfo(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_DeleteRegisterInfo, pDeviceHandle);
    }

    int TagAccessor::DeleteApplicationArea(
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_DeleteApplicationArea, pDeviceHandle);
    }

    int TagAccessor::ExistsApplicationArea(
            bool* pOutValue,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgExistsApplicationArea arg;
        arg.pOutValue = pOutValue;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_ExistsApplicationArea, arg);
    }

    int TagAccessor::GetDeviceState(
            nn::nfp::DeviceState* pOutDeviceState,
            const nn::nfp::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetDeviceState arg;
        arg.pOutDeviceState = pOutDeviceState;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetDeviceState, arg);
    }

    int TagAccessor::AttachAvailabilityChangeEvent(
            nn::os::SystemEventType* pAvailabilityChangeEvent
        ) NN_NOEXCEPT
    {
        ArgAttachAvailabilityChangeEvent arg;
        arg.pAvailabilityChangeEvent = pAvailabilityChangeEvent;

        return Enqueue(Command_AttachAvailabilityChangeEvent, arg);
    }

    nn::Result TagAccessor::Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT
    {
        return (*GetProcessFunc(static_cast<Command>(command)))(&param);
    }

}} // end of namespace nns::nfp
