﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{NfpCabinet.cpp,PageSampleNfpCabinet}

    @brief
    NFP ライブラリ amiibo 設定機能のサンプルプログラム
 */

/**
    @page PageSampleNfpCabinet NFP ライブラリ amiibo 設定機能のサンプル
    @tableofcontents

    @brief
    NFP ライブラリ amiibo 設定機能のサンプルプログラムの解説です。

    @section PageSampleNfpCabinet_SectionBrief 概要
    NFP ライブラリは、NFC（Near Field Communication）を介して、
    NFP タグ（NFP 用にフォーマットされた NFC タグ）の読み書きを行うライブラリです。
    NFP ライブラリを用いることで、アプリケーションと amiibo を連携させることが可能です。

    このサンプルプログラムは amiibo 設定機能を用いて、以下の amiibo 操作に対応しています。
    @n
    @li データが破損した amiibo の復旧
    @li amiibo のアプリケーション専用領域を削除
    @li amiibo のオーナー Mii とニックネームを登録
    @li amiibo の初期化

    サンプルプログラムでは固定のオーナー Mii とニックネームを登録します。

    @section PageSampleNfpCabinet_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/NfpCabinet
    Samples/Sources/Applications/NfpCabinet @endlink 以下にあります。

    @section PageSampleNfpCabinet_SectionNecessaryEnvironment 必要な環境
    事前に NoftWriter ツールを使用して NFP ライブラリで読み書き可能な NFP タグを用意してください。

    @section PageSampleNfpCabinet_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、NFP ライブラリの状態が表示されます。
    ゲームパッドの操作により、NFP ライブラリの状態を変化させることができます。

    @section PageSampleNfpCabinet_SectionPrecaution 注意事項
    サンプルプログラムを実行する開発機に、一度もタッチされたことがない amiibo のデータが破損している場合、
    バックアップデータが保存されていないため、データを復旧することができません。
    amiibo を初期化するか、バックアップデータが存在する開発機を使用して amiibo のデータを復旧してください。

    @section PageSampleNfpCabinet_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleNfpCabinet_SectionDetail 解説
    NfpCabinet では amiibo の状態によって異なる操作を実行します。
    特定の操作を狙って実行するためには、事前に以下の条件を満たす amiibo を用意しておく必要があります。

    @li アプリケーション専用領域の削除@n
    NfpSimple でアプリケーション専用領域を作成した amiibo を用意します。

    @li データが破損した amiibo の復旧@n
    データが破損した amiibo を用意します。

    @li オーナー Mii とニックネームの登録@n
    NoftWriter ツールで amiibo データリストを書き込んだばかりの amiibo を用意します。
    または、NfpCabinet から amiibo を初期化します。
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/init.h>
#include "Graphics.h"
#include "HidController.h"

#include "ConfigSystem.h"
#include "UpdateStateSystem.h"
#include "WriteStateSystem.h"

namespace {

    // Graphics.cpp で malloc しているヒープの内訳
    //  VisiblePoolMemory       16 MB
    //  InvisiblePoolMemory     20 MB
    //  for CommandBuffer       32 MB
    //  nv::InitializeGraphics   8 MB
    //  for DebugFont            2 MB
    const size_t MemoryHeapSize         = 128 * 1024 * 1024;
    const size_t MallocHeapSize         =  96 * 1024 * 1024;

    // アプリケーションで扱うデータと状態です。
    nns::nfp::ApplicationData g_Data;

} // namespace

void InitializeParameters()
{
    g_Data.startParam.mode = nn::nfp::AmiiboSettingsMode_NicknameAndOwnerSettings;
    g_Data.startParam.infoFlags = nn::nfp::AmiiboSettingsInfoFlags_None;

    NFPDEMO_LOG("StartParam.mode %x\n", g_Data.startParam.mode);
    NFPDEMO_LOG("StartParam.infoFlags %x\n", g_Data.startParam.infoFlags);

    const auto flag = g_Data.startParam.infoFlags;
    const auto mask = nn::nfp::AmiiboSettingsInfoFlags_TagInfo;
    if( (flag & mask) == mask )
    {
        switch( g_Data.startParam.mode )
        {
        case nn::nfp::AmiiboSettingsMode_NicknameAndOwnerSettings:
            {
                g_Data.action = nns::nfp::Action_Register;
            }
            break;
        case nn::nfp::AmiiboSettingsMode_GameDataEraser:
            {
                g_Data.action = nns::nfp::Action_Delete;
            }
            break;
        case nn::nfp::AmiiboSettingsMode_Restorer:
            {
                g_Data.action = nns::nfp::Action_Restore;
            }
            break;
        case nn::nfp::AmiiboSettingsMode_Formatter:
            {
                g_Data.action = nns::nfp::Action_Format;
            }
            break;
        default:
            {
                g_Data.action = nns::nfp::Action_None;
            }
            break;
        }
    }
    else
    {
        g_Data.action = nns::nfp::Action_None;
    }
}

void FinalizeParameters()
{
}

// ライブラリアプレットのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

// ライブラリアプレットのエントリポイント
extern "C" void nnMain()
{
    NFPDEMO_LOG("Launched\n");

    InitializeParameters();
    InitializeGraphicSystem();
    InitializeHidController();

    while( NN_STATIC_CONDITION(true) )
    {
        UpdateHidController();

        nns::nfp::UpdateState(g_Data);

        BeginText();
        nns::nfp::WriteState(g_Data);
        EndText();

        if( g_Data.state == nns::nfp::State_Exit )
        {
            break;
        }
    }

    FinalizeParameters();
    FinalizeHidController();
    FinalizeGraphicSystem();
}
