﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Assert.h>
#include "TagAccessorBase.h"

//#define NFC_PASSTHROUGHDEMO_PERMIT_HIGH_SPEED_TRANSITION

namespace nns { namespace nfc { namespace pt { namespace {

    const int ThreadStackSize = 16 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ ThreadStackSize ];

    int g_InstanceCount;
    volatile bool g_IsCanceled;

}}}} // end of namespace nns::nfc::pt::unnamed

namespace nns { namespace nfc { namespace pt {

    TagAccessorBase::TagAccessorBase() NN_NOEXCEPT
    :   m_IsThreadRunnable(true),
        m_IsRunning(false),
        m_CommandCount(0),
        m_ExecutedCommandCount(0),
        m_LastCommand(Command_None),
        m_pNotifyEvent(0)
    {
        NN_ABORT_UNLESS( g_InstanceCount++ == 0, "tag accessor is singleton pattern" );

        nn::os::InitializeEvent(&m_RunEvent, false, nn::os::EventClearMode_AutoClear);

        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::os::CreateThread(
                &m_Thread, NfcPassThroughThreadFunc, this,
                g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority) );
        nn::os::StartThread(&m_Thread);
    }

    TagAccessorBase::~TagAccessorBase() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS( --g_InstanceCount == 0, "tag accessor is singleton pattern" );

        m_IsThreadRunnable = false;
        nn::os::SignalEvent(&m_RunEvent);
        nn::os::WaitThread(&m_Thread);
        nn::os::DestroyThread(&m_Thread);
        nn::os::FinalizeEvent(&m_RunEvent);
    }

    int TagAccessorBase::GetEnqueuedCount() const NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );

        return m_CommandCount;
    }

    int TagAccessorBase::GetExecutedCount() const NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );

        return m_ExecutedCommandCount;
    }

    nn::Result TagAccessorBase::GetLastResult() const NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );

        return m_LastResult;
    }

    bool TagAccessorBase::GetNthResult(nn::Result* pOutResult, int index) const NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );
        NN_ASSERT( m_CommandCount == 0 );
        NN_ASSERT( index >= 0 );

        if( index >= 0 && index < (m_ExecutedCommandCount - 1) )
        {
            *pOutResult = nn::ResultSuccess();
            return true;
        }

        if(index == (m_ExecutedCommandCount - 1))
        {
            *pOutResult = m_LastResult;
            return true;
        }
        else
        {
            return false;
        }
    }

    bool TagAccessorBase::GetCommandResult(nn::Result* pOutResult, uint32_t command) const NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );
        NN_ASSERT( m_CommandCount == 0 );

        int i;
        for( i = 0; m_CommandList[i] != m_LastCommand && i < m_ExecutedCommandCount; ++i )
        {
            if( m_CommandList[i] == command )
            {
                *pOutResult = nn::ResultSuccess();
                return true;
            }
        }

        if(m_CommandList[i] == m_LastCommand)
        {
            *pOutResult = m_LastResult;
            return true;
        }
        else
        {
            return false;
        }
    }

    int TagAccessorBase::Enqueue(uint32_t command, const void* pParam, size_t paramSize) NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );
        NN_ASSERT( m_CommandCount < CommandCountMax );

        const int index = m_CommandCount++;
        m_CommandList[index] = command;
        if( pParam != nullptr && paramSize > 0 )
        {
            const size_t bufferSize = sizeof(ParameterBuffer);
            NN_ASSERT( paramSize < bufferSize );
            std::memcpy(&m_ParameterList[index], pParam, std::min(paramSize, bufferSize));
        }
        return index;
    }

    void TagAccessorBase::Run(nn::os::EventType* pEvent) NN_NOEXCEPT
    {
        NN_ASSERT( !m_IsRunning );

        m_IsRunning = true;
        m_ExecutedCommandCount = 0;

        m_pNotifyEvent = pEvent;
        nn::os::ClearEvent(m_pNotifyEvent);
        nn::os::SignalEvent(&m_RunEvent);
    }

    void TagAccessorBase::Cancel() NN_NOEXCEPT
    {
        // キューを全てクリアしてキャンセルするようにします。
        g_IsCanceled = true;

        // NfcPassThroughThread で実行中の処理がキャンセルされるまで待機します。
        while( IsBusy() )
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        }
        g_IsCanceled = false;
    }

    void TagAccessorBase::NfcPassThroughThreadFunc(void* arg) NN_NOEXCEPT
    {
        TagAccessorBase* pAccessor = reinterpret_cast<TagAccessorBase*>(arg);
        while( NN_STATIC_CONDITION(true) )
        {
            // TagAccessorBase::Run() が実行されるかデストラクタが呼ばれるまで待機します。
            nn::os::WaitEvent(&(pAccessor->m_RunEvent));
            if( !pAccessor->m_IsThreadRunnable )
            {
                break;
            }

            // 現在時刻を記憶します。
            #ifndef NFC_PASSTHROUGHDEMO_PERMIT_HIGH_SPEED_TRANSITION
            nn::os::Tick startTime = nn::os::GetSystemTick();
            #endif

            // 事前に登録された全てのコマンドを処理します。
            nn::Result result = nn::ResultSuccess();
            uint32_t command = Command_None;
            int i;
            for( i = 0; i < pAccessor->m_CommandCount && !g_IsCanceled && result.IsSuccess(); ++i )
            {
                // コマンドの種類に応じた処理を実行します。
                command = pAccessor->m_CommandList[i];
                const ParameterBuffer& param = pAccessor->m_ParameterList[i];
                result = pAccessor->Execute(command, param);
            }
            pAccessor->m_LastResult = result;
            pAccessor->m_LastCommand = command;
            pAccessor->m_ExecutedCommandCount = i;

            pAccessor->m_CommandCount = 0;
            pAccessor->m_IsRunning = false;

            #ifndef NFC_PASSTHROUGHDEMO_PERMIT_HIGH_SPEED_TRANSITION
            const int64_t minTransitionTime = 600;
            nn::os::Tick endTime = nn::os::GetSystemTick();
            int64_t ms = nn::os::ConvertToTimeSpan(endTime - startTime).GetMilliSeconds();
            if( ms < minTransitionTime )
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(minTransitionTime - ms));
            }
            #endif

            // 処理の完了を通知します。
            nn::os::SignalEvent(pAccessor->m_pNotifyEvent);
        }
    }

}}} // end of namespace nns::nfc::pt
