﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h> //hoge
#include "Config.h"
#include "TagAccessor.h"

namespace nns { namespace nfc { namespace pt { namespace {

    NN_ALIGNAS(16) char g_WorkBuffer[ sizeof(TagAccessor) ];
    bool g_IsWaitForAmiiboSetting = false;

    typedef nn::Result (*ProcessFunc)( const void* pParam );

    struct ArgDeviceHandle
    {
        const nn::nfc::DeviceHandle*    pDeviceHandle;
    };

    // NFC ライブラリ API が ResultAccessError() を返す場合、成功するまで RetryCountMax 回再試行します。
    const int RetryCountMax = 5;
    const int RetryInterval = 200;

    nn::Result ProcessRetryFunction(const void* pParam, ProcessFunc pFunc) NN_NOEXCEPT
    {
        nn::Result result;
        for( int i = 0; i < RetryCountMax; ++i )
        {
            result = (*pFunc)(pParam);
            if( !nn::nfc::ResultAccessError().Includes(result) )
            {
                break;
            }
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(RetryInterval));
        }
        return result;
    }

    // NFC ライブラリを初期化します。
    nn::Result ProcessInitialize(const void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);
        nn::nfc::Initialize();
        return nn::ResultSuccess();
    }

    // NFC ライブラリを終了します。
    nn::Result ProcessFinalize(const void* pParam) NN_NOEXCEPT
    {
        NN_UNUSED(pParam);
        nn::nfc::Finalize();
        return nn::ResultSuccess();
    }

    // NFC デバイスのハンドルのリストを取得します。
    struct ArgListDevices
    {
        nn::nfc::DeviceHandle*  pOutBuffer;
        int*                            pOutCount;
        int                             bufferCount;
    };

    nn::Result ProcessListDevices(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgListDevices*>(pParam);
        return nn::nfc::ListDevices(pArg->pOutBuffer, pArg->pOutCount, pArg->bufferCount);
    }

    // タグの発見・喪失を通知するイベントを設定し、NFC デバイスの Npad ID を取得します。
    struct ArgAttachEvents
    {
        const nn::nfc::DeviceHandle* pDeviceHandle;
        nn::os::SystemEventType*             pActivateEvent;
        nn::os::SystemEventType*             pDeactivateEvent;
        nn::hid::NpadIdType*                 pOutNpadId;
    };

    nn::Result ProcessAttachEvents(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgAttachEvents*>(pParam);

        nn::Result result = nn::nfc::GetNpadId(pArg->pOutNpadId, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        result = nn::nfc::AttachActivateEvent(pArg->pActivateEvent, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        result = nn::nfc::AttachDeactivateEvent(pArg->pDeactivateEvent, *pArg->pDeviceHandle);
        if( result.IsFailure() ) { return result; }

        return nn::ResultSuccess();
    }

    struct ArgStartDetection
    {
        const nn::nfc::DeviceHandle*       pDeviceHandle;
        nn::nfc::NfcProtocol               protocolFilter;
    };

    // タグの検知を開始します。
    nn::Result ProcessStartDetection(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgStartDetection*>(pParam);
        return nn::nfc::StartDetection(*pArg->pDeviceHandle, pArg->protocolFilter);
    }

    // タグの検知を終了します。
    nn::Result ProcessStopDetection(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfc::StopDetection(*pArg->pDeviceHandle);
    }

    // セッションの維持を行います。
    nn::Result ProcessKeepSession(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfc::KeepPassThroughSession(*pArg->pDeviceHandle);
    }

    // セッションを解放します。
    nn::Result ProcessReleaseSession(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgDeviceHandle*>(pParam);
        return nn::nfc::ReleasePassThroughSession(*pArg->pDeviceHandle);
    }

    struct ArgSendCommand
    {
        void*                                  pOutBuffer;
        size_t*                                pOutSize;
        const nn::nfc::DeviceHandle*       pDeviceHandle;
        const void*                            pData;
        size_t                                 dataSize;
        size_t                                 bufferSize;
        nn::TimeSpan                           timeout;
    };

    // タグにコマンドを送信します。
    nn::Result ProcessSendCommand(const void* pParam) NN_NOEXCEPT
    {
        // タグへのコマンド送信には数秒程度かかります。
        // ResultAccessError() のエラーが発生した場合には正しいアクセス方法を用いた再試行で成功する可能性があります。
        return ProcessRetryFunction(
            pParam,
            [](const void* pParam) -> nn::Result
            {
                auto pArg = reinterpret_cast<const ArgSendCommand*>(pParam);
                return nn::nfc::SendCommandByPassThrough(pArg->pOutBuffer, pArg->pOutSize, *pArg->pDeviceHandle, pArg->pData, pArg->dataSize, pArg->bufferSize, pArg->timeout);
            }
        );
    }

    // タグの情報を取得します。
    struct ArgGetTagInfo
    {
        nn::nfc::TagInfo*               pOutInfo;
        const nn::nfc::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetTagInfo(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetTagInfo*>(pParam);
        return nn::nfc::GetTagInfo(pArg->pOutInfo, *pArg->pDeviceHandle);
    }

    // デバイスの状態を取得します。
    struct ArgGetDeviceState
    {
        nn::nfc::DeviceState*           pOutDeviceState;
        const nn::nfc::DeviceHandle*    pDeviceHandle;
    };

    nn::Result ProcessGetDeviceState(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgGetDeviceState*>(pParam);
        *pArg->pOutDeviceState = nn::nfc::GetDeviceState(*pArg->pDeviceHandle);
        return nn::ResultSuccess();
    }

    // デバイスの利用可/不可の変化を通知するイベントを設定します。
    struct ArgAttachAvailabilityChangeEvent
    {
        nn::os::SystemEventType*        pAvailabilityChangeEvent;
    };

    nn::Result ProcessAttachAvailabilityChangeEvent(const void* pParam) NN_NOEXCEPT
    {
        auto pArg = reinterpret_cast<const ArgAttachAvailabilityChangeEvent*>(pParam);

        nn::nfc::AttachAvailabilityChangeEvent(pArg->pAvailabilityChangeEvent);
        return nn::ResultSuccess();
    }

    // コマンド識別子に対応した処理関数の定義
    struct ProcessEntry
    {
        Command         command;
        ProcessFunc     func;
    };

    const ProcessEntry ProcessTable[] = {
        {   Command_None,                           nullptr                         },
        {   Command_Initialize,                     ProcessInitialize               },
        {   Command_Finalize,                       ProcessFinalize                 },
        {   Command_ListDevices,                    ProcessListDevices              },
        {   Command_AttachEvents,                   ProcessAttachEvents             },
        {   Command_StartDetection,                 ProcessStartDetection           },
        {   Command_StopDetection,                  ProcessStopDetection            },
        {   Command_KeepSession,                    ProcessKeepSession              },
        {   Command_ReleaseSession,                 ProcessReleaseSession           },
        {   Command_SendReadCommand,                ProcessSendCommand              },
        {   Command_SendPwdAuthCommand,             ProcessSendCommand              },
        {   Command_SendWriteCommand,               ProcessSendCommand              },
        {   Command_GetTagInfo,                     ProcessGetTagInfo               },
        {   Command_GetDeviceState,                 ProcessGetDeviceState           },
        {   Command_AttachAvailabilityChangeEvent,  ProcessAttachAvailabilityChangeEvent },
    };
    const int ProcessCountMax = sizeof(ProcessTable) / sizeof(ProcessTable[0]);
    NN_STATIC_ASSERT( ProcessCountMax == static_cast<int>(Command_CountMax) );

    ProcessFunc GetProcessFunc(Command command)
    {
        const int index = static_cast<int>(command);
        NN_ASSERT( index >= 1 && index < ProcessCountMax );
        NN_ASSERT( ProcessTable[index].command == command );
        return ProcessTable[index].func;
    }

}}}} // end of namespace nns::nfc::pt::unnamed

namespace nns { namespace nfc { namespace pt {

    TagAccessor* TagAccessor::g_pInstance;

    TagAccessor& TagAccessor::CreateInstance() NN_NOEXCEPT
    {
        NN_ASSERT( g_pInstance == nullptr );

        g_pInstance = new(g_WorkBuffer) TagAccessor();
        return *g_pInstance;
    }

    TagAccessor& TagAccessor::GetInstance() NN_NOEXCEPT
    {
        NN_ASSERT( g_pInstance != nullptr );

        return *g_pInstance;
    }

    void TagAccessor::DestroyInstance() NN_NOEXCEPT
    {
        if( g_pInstance != nullptr )
        {
            g_pInstance->~TagAccessor();
            g_pInstance = nullptr;
        }
    }

    bool TagAccessor::HasInstance() NN_NOEXCEPT
    {
        return g_pInstance != nullptr;
    }

    int TagAccessor::Initialize() NN_NOEXCEPT
    {
        return Enqueue(Command_Initialize, nullptr, 0);
    }

    int TagAccessor::Finalize() NN_NOEXCEPT
    {
        return Enqueue(Command_Finalize, nullptr, 0);
    }

    int TagAccessor::ListDevices(
            nn::nfc::DeviceHandle* pOutBuffer,
            int* pOutCount,
            int bufferCount
        ) NN_NOEXCEPT
    {
        ArgListDevices arg;
        arg.pOutBuffer = pOutBuffer;
        arg.pOutCount = pOutCount;
        arg.bufferCount = bufferCount;

        return Enqueue(Command_ListDevices, arg);
    }

    int TagAccessor::AttachEvents(
            nn::hid::NpadIdType* pOutNpadId,
            nn::os::SystemEventType* pActivateEvent,
            nn::os::SystemEventType* pDeactivateEvent,
            const nn::nfc::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgAttachEvents arg;
        arg.pOutNpadId = pOutNpadId;
        arg.pActivateEvent = pActivateEvent;
        arg.pDeactivateEvent = pDeactivateEvent;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_AttachEvents, arg);
    }

    int TagAccessor::StartDetection(
            const nn::nfc::DeviceHandle* pDeviceHandle,
            nn::nfc::NfcProtocol protocolFilter
        ) NN_NOEXCEPT
    {
        ArgStartDetection arg;
        arg.pDeviceHandle = pDeviceHandle;
        arg.protocolFilter = protocolFilter;
        return Enqueue(Command_StartDetection, arg);
    }

    int TagAccessor::StopDetection(
            const nn::nfc::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_StopDetection, pDeviceHandle);
    }

    int TagAccessor::KeepSession(
            const nn::nfc::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_KeepSession, pDeviceHandle);
    }

    int TagAccessor::ReleaseSession(
            const nn::nfc::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        return Enqueue(Command_ReleaseSession, pDeviceHandle);
    }

    int TagAccessor::SendCommand(
            void* pOutBuffer,
            size_t* pOutSize,
            const nn::nfc::DeviceHandle* pDeviceHandle,
            const void* pData,
            size_t dataSize,
            size_t bufferSize,
            nn::TimeSpan timeout,
            Command command
        ) NN_NOEXCEPT
    {
        ArgSendCommand arg;
        arg.pOutBuffer = pOutBuffer;
        arg.pOutSize = pOutSize;
        arg.pDeviceHandle = pDeviceHandle;
        arg.pData = pData;
        arg.dataSize = dataSize;
        arg.bufferSize = bufferSize;
        arg.timeout = timeout;

        return Enqueue(command, arg);
    }

    int TagAccessor::GetTagInfo(
            nn::nfc::TagInfo* pOutInfo,
            const nn::nfc::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetTagInfo arg;
        arg.pOutInfo = pOutInfo;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetTagInfo, arg);
    }

    int TagAccessor::GetDeviceState(
            nn::nfc::DeviceState* pOutDeviceState,
            const nn::nfc::DeviceHandle* pDeviceHandle
        ) NN_NOEXCEPT
    {
        ArgGetDeviceState arg;
        arg.pOutDeviceState = pOutDeviceState;
        arg.pDeviceHandle = pDeviceHandle;

        return Enqueue(Command_GetDeviceState, arg);
    }

    int TagAccessor::AttachAvailabilityChangeEvent(
            nn::os::SystemEventType* pAvailabilityChangeEvent
        ) NN_NOEXCEPT
    {
        ArgAttachAvailabilityChangeEvent arg;
        arg.pAvailabilityChangeEvent = pAvailabilityChangeEvent;

        return Enqueue(Command_AttachAvailabilityChangeEvent, arg);
    }

    bool TagAccessor::IsBusy() NN_NOEXCEPT
    {
        return TagAccessorBase::IsBusy() && g_IsWaitForAmiiboSetting == false;
    }

    nn::Result TagAccessor::Execute(uint32_t command, const ParameterBuffer& param) NN_NOEXCEPT
    {
        return (*GetProcessFunc(static_cast<Command>(command)))(&param);
    }

}}} // end of namespace nns::nfc::pt
